/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_TIMER_HPP
#define SL_INET_TIMER_HPP

#include <iostream>
#include <string>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/thread.hpp>
#include <boost/thread/condition.hpp>

namespace sl {


class timer {
public :
    timer() : _watcher(0), _is_continue(false)
    { }

    ~timer()
    {
        destroy();
    }

    void timeout(const boost::function<void (void)>& timeout_handler)
    {
        _timeout_h = timeout_handler;
    }

    void init()
    {
        boost::mutex::scoped_lock lock(_mutex);

        _is_continue = true;
        _is_started = false;
        if (!_watcher)
            _watcher = new boost::thread(boost::bind(&timer::do_watch, this));
    }

    void destroy()
    {
        boost::mutex::scoped_lock lock(_mutex);

        /*
         * _is_started ե饰ѹԤ碌֤¸ߤΤ
         * _is_continue ե饰ѹ _is_started ѹ notify ¹Ԥơ
         * åɤ򴺤ư _is_continue ե饰 false 
         * ʤäƤ뤳Ȥդ.
         */
        _is_continue = false;
        _is_started = true;
        _cond.notify_all();
        lock.unlock();

        if (_watcher) {
            _watcher->join();
            delete _watcher;
            _watcher = 0;
        }
    }

    void start(time_t t)
    {
        if (!_watcher)
            init();

        boost::mutex::scoped_lock lock(_mutex);

        _timeout = t;
        _is_started = true;
        _cond.notify_all();
    }

    void restart(time_t t)
    {
        stop();
        start(t);
    }

    void stop()
    {
        boost::mutex::scoped_lock lock(_mutex);
        _is_started = false;
    }

private :

    void do_watch()
    {
        try {
            for (;;) {
                boost::mutex::scoped_lock lock(_mutex);
                while (!_is_started)
                    _cond.wait(lock);
                lock.unlock();

                lock.lock();
                if (!_is_continue)
                    break;
                lock.unlock();

                if (_timeout == 0) {
                    stop();
                    _timeout_h();
                }
                sleep(1);
                --_timeout;
            }
        } catch (std::exception& e) {
            std::cerr << "timer::do_watch() exception:"
                      << e.what() << std::endl;
        }
    }

    boost::function<void (void)> _timeout_h;

    boost::thread* _watcher;
    boost::mutex _mutex;
    boost::condition _cond;

    bool _is_started;
    bool _is_continue;
    time_t _timeout;
};

} // namespace sl


#endif // SL_INET_TIMER_HPP
