/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_SSL_ACCEPTOR_HPP
#define SL_INET_SSL_ACCEPTOR_HPP

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <unistd.h>

#include <cctype>
#include <cerrno>
#include <csignal>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <string>

#include <openssl/err.h>
#include <openssl/ssl.h>

#include <sl/inet/ssl_socket.hpp>

namespace sl {


class ssl_acceptor {
public :
    struct init_args_type {
        init_args_type(const std::string& port,
                       const std::string& key1,
                       const std::string& key2)
            : _port(port), _key1(key1), _key2(key2)
        { }

        std::string        _port;
        std::string        _key1;
        std::string        _key2;
    };

    ssl_acceptor() { }
    virtual ~ssl_acceptor() { destroy(); }

    void init(const init_args_type& args)
    {
        std::string port = args._port;

        if ((_socket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
            throw std::runtime_error("sl::acceptr::init :init failed.");

        int ret;
        int opt = 1;
        ret = setsockopt(_socket, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
        if (ret != 0)
            throw std::runtime_error("sl::acceptor::init :setsockopt failed.");

        setsockopt(_socket, SOL_SOCKET, SO_KEEPALIVE, &opt, sizeof(opt));
        if (ret != 0)
            throw std::runtime_error("sl::acceptor::init :setsockopt failed.");

        opt = 15;
        setsockopt(_socket, SOL_SOCKET, SO_LINGER, &opt, sizeof(opt));
        if (ret != 0)
            throw std::runtime_error("sl::acceptor::init :setsockopt failed.");

        opt = 8192;
        setsockopt(_socket, SOL_SOCKET, SO_RCVBUF, &opt, sizeof(opt));
        if (ret != 0)
            throw std::runtime_error("sl::acceptor::init :setsockopt failed.");

        opt = 8192;
        setsockopt(_socket, SOL_SOCKET, SO_SNDBUF, &opt, sizeof(opt));
        if (ret != 0)
            throw std::runtime_error("sl::acceptor::init :setsockopt failed.");

        setsockopt(_socket, IPPROTO_TCP, TCP_NODELAY, &opt, sizeof(opt));
        if (ret != 0)
            throw std::runtime_error("sl::acceptor::init :setsockopt failed.");

        struct sockaddr_in  saddr;
        memset((char *)&saddr, 0x00, sizeof(saddr));
        saddr.sin_family = AF_INET;
        saddr.sin_addr.s_addr = INADDR_ANY;
        saddr.sin_port = htons(atoi(port.c_str()));

        if (::bind(_socket, (struct sockaddr *)&saddr, sizeof(saddr)) < 0)
            throw std::runtime_error(strerror(errno));

#if defined(__linux__) || defined(__linux) || defined(linux)
        if(listen(_socket, SOMAXCONN) < 0)
#elif defined(__FreeBSD__)
        if(listen(_socket, -1) < 0)
#else
        if(listen(_socket, 128) < 0)
#endif
            throw std::runtime_error("sl::acceptor::init :listen failed.");

        // initialization SSL libraries.
        SSL_library_init();
        SSL_load_error_strings();

        _use_certificate_file = args._key1;
        _use_privateKey_file  = args._key2;
    }

    void destroy() throw()
    {
        ::close(_socket);
    }

    void accept(ssl_socket& s) // throw()
    {    
        struct sockaddr_in  addr;
        memset((char *)&addr, 0x00, sizeof(addr));
        socklen_t addr_len = sizeof(addr);

        for (;;) {
            fd_set  fds;
            FD_ZERO(&fds);
            FD_SET(_socket, &fds);

            struct timeval t = { 5, 0 };

            int ret = ::select(FD_SETSIZE, &fds, NULL, NULL, &t);
            if (ret == 0)
                continue;
            else if (ret < 0)
                throw std::runtime_error("sl::acceptor::accept :select failed.");

            if (!FD_ISSET(_socket, &fds)) {
                continue;
            }

            s._socket = ::accept(_socket, (struct sockaddr *)&addr, &addr_len);
            if (s._socket != -1) {
                ssl_setup(s);
                return;
            }

            switch (errno) {
            case EINTR :
            case EMFILE :
            case ENFILE :
            case ECONNABORTED :
                break;
            default :
                throw std::runtime_error("sl::ssl_acceptr::accept :accept failed.");
            }
        }
    }

private :

    void ssl_setup(ssl_socket& s)
    {
        int ret;

        if (SSL_set_fd(s.socket(), s.descriptor()) != 1) {
            std::cerr << "Error :SSL_set_fd" << std::endl;
            throw std::runtime_error("ssl_acceptor::ssl_setup");
               return;
        }

        ret = SSL_use_certificate_file(s.socket(),
                                       _use_certificate_file.c_str(),
                                       SSL_FILETYPE_PEM);
        if (ret != 1) {
            std::cerr << "Error :SSL_use_certificate_file" << std::endl;
            throw std::runtime_error("ssl_acceptor::ssl_setup");
        }

        ret = SSL_use_PrivateKey_file(s.socket(),
                                      _use_privateKey_file.c_str(),
                                      SSL_FILETYPE_PEM);
        if (ret != 1) {
            std::cerr << "Error :SSL_use_PrivateKey_file" << std::endl;
            throw std::runtime_error("ssl_acceptor::ssl_setup");
        }

        if (SSL_accept(s.socket()) != 1) {
            std::cerr << "Error :SSL_accept" << std::endl;
            throw std::runtime_error("ssl_acceptor::ssl_setup");
        }
        return;
    }

    int _socket;

    std::string _use_certificate_file;
    std::string _use_privateKey_file;
};


} // namespace sl

#endif // SL_INET_SSL_ACCEPTOR_HPP
