/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_SOCKSTREAM_HPP
#define SL_INET_SOCKSTREAM_HPP

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <unistd.h>

#include <algorithm>
#include <iomanip>
#include <ios>
#include <iostream>
#include <stdexcept>

#include <sl/inet/basic_socket.hpp>
#include <sl/inet/basic_socket_reader.hpp>
#include <sl/inet/basic_socket_writer.hpp>

#ifndef SL_INET_SOCKSTREAM_BUFLEN
#define SL_INET_SOCKSTREAM_BUFLEN 4092
#endif

namespace sl {

template <
    class SocketType=basic_socket,
    class ReaderType=basic_socket_reader,
    class WriterType=basic_socket_writer
    >
class sockstreambuf : public std::streambuf {

    enum { _default_buffer = SL_INET_SOCKSTREAM_BUFLEN };

public :
    typedef char char_type;
    typedef std::char_traits<char> traits_type;
    typedef traits_type::int_type int_type;
    typedef traits_type::pos_type pos_type;
    typedef traits_type::off_type off_type;

    typedef SocketType socket_type;
    typedef ReaderType reader_type;
    typedef WriterType writer_type;

    sockstreambuf(const socket_type& d)
        : _socket(d),
          _timeout(0),
          _curtime(0),
          _length(_default_buffer),
          _rlength(-1),
          _currlen(0)
    {
        _gbuffer = new char[_length];
        _pbuffer = new char[_length];
        setg(_gbuffer, _gbuffer + _length, _gbuffer + _length);
        setp(_pbuffer, _pbuffer + _length);
    }

    ~sockstreambuf()
    {
        sync();
        delete _gbuffer;
        delete _pbuffer;
    }

    socket_type& socket()
    {
        return _socket;
    }

    /**
     * եǥץꤷޤ.
     * ޤߤΥХåե򥯥ꥢޤ.
     *
     * @param    ꤹե/åȥǥץ.
     */
    void reset(const socket_type& d)
    {
        _socket = d;
        rlength(-1);

        setg(_gbuffer, _gbuffer + _length, _gbuffer + _length);
        setp(_pbuffer, _pbuffer + _length);
    }

    /**
     * ɤ߹߻ΥǡԤ֤ꤷޤ.
     *
     * @param    Ԥ().
     */
    void timeout(time_t t)
    {
        _curtime = t;
        _timeout = t;
    }

    /**
     * ɤ߹ߥǡκꤷޤ.
     *
     * ºݤˤŬʥΥǡ٤ɤ߹िᡢ
     * δؿǻꤹͤɤ߹ߤξ¤ǤϤʤ
     * [ɬפʺ]ȤƻѤ졢ǡμФԤʤäˡ
     * ɤ߹ǡĤǤäƤ n ХȤͤФ
     * EOF ֤ưȤʤޤ.
     *
     * HTTP  Content-Length  ܥǥδطͤʡ
     * ɤ߽ФǡΥХȿϤ狼äƤꡢͤ
     * ɤߤǡʤԤ碌Ƥ
     * EOF֤ưȤ뤿˺ޤ.
     *
     * Υ饹󥹥󥹤ڤӡåȤˤ
     * ǡĤΤǡδؿ˰ -1 ꤷ
     * ĤΥǡɤ߹ɬפޤ.
     *
     * ɤ߹Ǥδؿ¹Ԥưʤ
     * ǽޤ.
     *
     * @paran    int  ɬפʺ. -1 ̵. -1ʳͭ.
     *
     * @note    ServletAPI  getContentLength ̤ -1 ֤
     *            Ǥ⥤󥿡ե碌int ˤƤ.
     */
    void rlength(int n)
    {
        _rlength = n;
        _currlen = egptr() - gptr();
    }

    std::streambuf* pubsetbuf(char_type* s, std::streamsize n)
    {
        if (n <= 0)
            throw std::runtime_error("std::streambuf::pubsetbuf");

        delete _gbuffer;
        delete _pbuffer;

        _length = n;
        _gbuffer = new char[_length];
        _pbuffer = new char[_length];
        setg(_gbuffer, _gbuffer + _length, _gbuffer + _length);
        setp(_pbuffer, _pbuffer + _length);
    }

protected :

    std::streamsize xsputn(const char_type* s, std::streamsize n)
    {
        int wval = epptr() - pptr();
        if (n <= wval) {
            memcpy(pptr(), s, n * sizeof(char_type));
            pbump(n);
            return n;
        }
        memcpy(pptr(), s, wval * sizeof(char_type));
        pbump(wval);

        if (overflow() != traits_type::eof())
            return wval + xsputn(s + wval, n - wval);

        return wval;
    }

    int sync()
    {
        if (pptr() && pbase() < pptr() && pptr() <= epptr()) {
            int size = _writer(_socket.socket(), pbase(), pptr() - pbase());
            if (size <= 0)
                return -1;

            std::copy(pbase() + size, pptr(), pbase());
            pbump(pptr() - pbase() - size);
            setp(pbase(), epptr());
            return traits_type::not_eof(size);
        }
        return traits_type::eof();
    }

    int_type overflow(int_type c = traits_type::eof())
    {
        if (!pbase())
            return traits_type::eof();

        if (c == traits_type::eof())
            return sync();

        if (pptr() == epptr())
            sync();

        *pptr() = traits_type::to_char_type(c);
        pbump(1);

        return c;
    }

    int_type underflow()
    {
        // ɤ߹߿λ꤬
        // ɤ߹߿ȸߤɤ߹߿Ʊξ
        // ʾɤ߹ޤʤ.
        if (_rlength != -1 && _rlength <= _currlen)
            return traits_type::eof();

        if (!gptr())
            return traits_type::eof();

        if (gptr() < egptr())
            return traits_type::to_int_type(*gptr());

        if (!select(_timeout))
            return traits_type::eof();

        int read_size = _length - static_cast<size_t>(gptr() - egptr());
        if (_rlength != -1) {
            read_size = _rlength - _currlen < read_size ?
                                    _rlength - _currlen : read_size;
        }
        int size = _reader(_socket.socket(), eback(), read_size);

        // ɤ߹߿λ꤬
        if (_rlength != -1)
            _currlen += size;

        // XXXXX
        //if (size <= 0)
        //    throw std::runtime_error("read failed in `streambuf::underflow'");
        if (size < 0)
            throw std::runtime_error("read failed in `streambuf::underflow'");

        if (size == 0) {
            setg(_gbuffer, _gbuffer + _length, _gbuffer + _length);
            return traits_type::eof();
        }
        setg(eback(), eback(), eback() + size);
        return traits_type::to_int_type(*gptr());
    }

    int_type uflow()
    {
        return std::streambuf::uflow();
    }

private :

    bool select(time_t timeout)
    {
        _curtime = timeout;
        do {
            fd_set  fds;
            FD_ZERO(&fds);
            FD_SET(_socket.descriptor(), &fds);

            struct timeval t = { timeout, 0 };

            int ret = ::select(_socket.descriptor()+1, &fds, 0, 0, &t);
            if (ret == -1)
                throw std::runtime_error("select failed in 'sockstreambuf'");
               return true;
        } while (--_curtime > 0);
        return false;
    }

    socket_type _socket;
    reader_type _reader;
    writer_type _writer;

    char* _gbuffer;
    char* _pbuffer;
    time_t _timeout;
    time_t _curtime;
    size_t _length;

    int _rlength;
    int _currlen;
};


template <class StreambufType>
class basic_isockstream : public std::istream {
public:

    typedef StreambufType streambuf_type;
    typedef typename StreambufType::socket_type socket_type;

    basic_isockstream(const socket_type& s=socket_type())
        : std::istream(0), _sockbuf(s)
    {
        this->init(&_sockbuf);
    }

    ~basic_isockstream() { }

    void reset(const socket_type& s)
    {
        _sockbuf.reset(s);
    }

    void timeout(time_t t)
    {
        _sockbuf.timeout(t);
    }

    void rlength(int n)
    {
        _sockbuf.rlength(n);
    }

private:
    streambuf_type _sockbuf;
};


template <class StreambufType>
class basic_osockstream : public std::ostream {
public:

    typedef StreambufType streambuf_type;
    typedef typename StreambufType::socket_type socket_type;

    basic_osockstream(const socket_type& s=socket_type())
        : std::ostream(0), _sockbuf(s)
    {
        this->init(&_sockbuf);
        unsetf(std::ios::skipws);
    }

    ~basic_osockstream() { }

    void reset(const socket_type& s)
    {
        _sockbuf.reset(s);
    }

    void timeout(time_t t)
    {
        _sockbuf.timeout(t);
    }

private:
    streambuf_type _sockbuf;
};


template <class StreambufType>
class basic_iosockstream : public std::iostream {
public:

    typedef StreambufType streambuf_type;
    typedef typename StreambufType::socket_type socket_type;

    basic_iosockstream(const socket_type& s=socket_type())
        : std::iostream(0), _sockbuf(s)
    {
        this->init(&_sockbuf);
        unsetf(std::ios::skipws);
    }

    ~basic_iosockstream() { }

    void reset(const socket_type& s)
    {
        _sockbuf.reset(s);
    }

    void timeout(time_t t)
    {
        _sockbuf.timeout(t);
    }

    void rlength(int n)
    {
        _sockbuf.rlength(n);
    }

private:
    streambuf_type    _sockbuf;
};

typedef basic_isockstream<sockstreambuf<> >  isstream;
typedef basic_osockstream<sockstreambuf<> >  osstream;
typedef basic_iosockstream<sockstreambuf<> > iosstream;

} // namespace sl

#endif // SL_INET_SOCKSTREAM_HPP
