/*-
 * Copyright (c) 2007 Masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_REQUEST_LINE_PARSAR_HPP
#define SL_INET_REQUEST_LINE_PARSAR_HPP

/*
       Method SP Request-URI SP HTTP-Version CRLF

       Method         = "OPTIONS"                ; Section 9.2
                      | "GET"                    ; Section 9.3
                      | "HEAD"                   ; Section 9.4
                      | "POST"                   ; Section 9.5
                      | "PUT"                    ; Section 9.6
                      | "DELETE"                 ; Section 9.7
                      | "TRACE"                  ; Section 9.8
                      | "CONNECT"                ; Section 9.9
                      | extension-method
       extension-method = token
       Request-URI    = "*" | absoluteURI | abs_path | authority
       HTTP-Version   = "HTTP" "/" 1*DIGIT "." 1*DIGIT
       CHAR           = <US-ASCII ʸ (0 - 127 ƥå)>
       DIGIT          = <US-ASCII  "0".."9">
       CTL            = <US-ASCII ʸ
                        (0 - 31 ƥå)  DEL (127)>
       CR             = <US-ASCII CR, å꥿ (13)>
       LF             = <US-ASCII LF, 饤ե (10)>
       SP             = <US-ASCII SP, ڡ (32)>
       HT             = <US-ASCII HT, ʿ (9)>
       CRLF           = CR LF

       token          = 1*<CTL  separators  CHAR>
       separators     = "(" | ")" | "<" | ">" | "@"
                      | "," | ";" | ":" | "\" | <">
                      | "/" | "[" | "]" | "?" | "="
                      | "{" | "}" | SP  | HT

    [414(Request-URI Too Long)]
*/

#define BOOST_SPIRIT_THREADSAFE
#include <boost/spirit.hpp>
#include <boost/spirit/phoenix.hpp>
using namespace boost::spirit;


struct request_line_parser : public grammar<request_line_parser> {

    std::string& _method;
    std::string& _uri;
    std::string& _version;

    request_line_parser(std::string& method,
                        std::string& uri,
                        std::string& version)
        : _method(method), _uri(uri), _version(version),
          _assign(assign()), _append(append())
    { }

    ~request_line_parser()
    { }

    struct assign
    {
        typedef void result_type;

        void operator()(std::string& value,
                        const char* first, const char* last) const
        {
            value.assign(first, last);
        }
    };

    struct append
    {
        typedef void result_type;

        void operator()(std::string& value,
                        const char* first, const char* last) const
        {
            value.append(first, last);
        }
    };

    phoenix::functor<assign>    _assign;
    phoenix::functor<append>    _append;

    struct print
    {
        template <typename T>
        void operator()(T first, T last) const
        {
            std::cerr << std::string(first, last) << std::endl;
        }

        template <typename T>
        void operator()(T l) const
        {
            std::cerr << "parse:[" << std::string(l) << "]" << std::endl;
        }
    };


    struct uri_closure : closure<uri_closure, std::string>
    {
        member1 val;
    };

    template<typename ScannerT>
    struct definition {
        rule<ScannerT>
            top;
        rule<ScannerT, uri_closure::context_t>
            Method, extension_method, GET,
            Request_URI, absoluteURI, abs_path, authority,
            HTTP_Version,
            CHAR, DIGIT, CTL, CR, LF, SP, HT, CRLF, NON_SP,
            token,
            separators;

        definition(const request_line_parser& self)
        {
#define ASSIGN(X)    self._assign(X.val, arg1, arg2)
#define APPEND(X)    self._append(X.val, arg1, arg2)

            using phoenix::arg1;
            using phoenix::arg2;

    top            = Method             [assign_a(self._method)] >>
                     SP >>
                     Request_URI        [assign_a(self._uri)] >>
                     SP >>
                     HTTP_Version       [assign_a(self._version)] >>
                     CRLF;

    Method         = str_p("OPTIONS")   [ASSIGN(Method)]
                   | str_p("GET")       [ASSIGN(Method)]
                   | str_p("HEAD")      [ASSIGN(Method)]
                   | str_p("POST")      [ASSIGN(Method)]
                   | str_p("PUT")       [ASSIGN(Method)]
                   | str_p("DELETE")    [ASSIGN(Method)]
                   | str_p("TRACE")     [ASSIGN(Method)]
                   | str_p("CONNECT")   [ASSIGN(Method)]
                   | extension_method   [Method.val = arg1];
    extension_method = token            [extension_method.val = arg1];

    Request_URI    = str_p("*")         [ASSIGN(Request_URI)]
                   | absoluteURI        [Request_URI.val = arg1]
                   | abs_path           [Request_URI.val = arg1]
                   | authority          [Request_URI.val = arg1];

    absoluteURI    = +(CHAR - SP)       [APPEND(absoluteURI)];
    abs_path       = +(CHAR - SP)       [abs_path.val  += arg1];
    authority      = +(CHAR - SP)       [authority.val += arg1];

    HTTP_Version   = str_p("HTTP")      [HTTP_Version.val  = arg1] >>
                     str_p("/")         [HTTP_Version.val += arg1] >>
                     +DIGIT             [HTTP_Version.val += arg1] >>
                     str_p(".")         [HTTP_Version.val += arg1] >>
                     +DIGIT             [HTTP_Version.val += arg1];

    CHAR           = anychar_p          [CHAR.val = arg1];
    DIGIT          = digit_p            [DIGIT.val = arg1];
    CTL            = range_p(0, 31)     [CTL.val = arg1];
    CR             = str_p("\r")        [CR.val = arg1];
    LF             = str_p("\n")        [LF.val = arg1];
    SP             = str_p(" ")         [SP.val = arg1];
    HT             = str_p("\t")        [HT.val = arg1];
    CRLF           = !(CR) >> !(LF);

    NON_SP         = CHAR - SP          [NON_SP.val = arg1];

    token          = +(CHAR - (CTL | separators))  [token.val += arg1];
    separators     = chset<>("()<>@,;:\\\"/[]?={}") | SP  | HT;

    BOOST_SPIRIT_DEBUG_NODE(top);
    BOOST_SPIRIT_DEBUG_NODE(Method);
    BOOST_SPIRIT_DEBUG_NODE(extension_method);
    BOOST_SPIRIT_DEBUG_NODE(Request_URI);
    BOOST_SPIRIT_DEBUG_NODE(absoluteURI);
    BOOST_SPIRIT_DEBUG_NODE(abs_path);
    BOOST_SPIRIT_DEBUG_NODE(authority);
    BOOST_SPIRIT_DEBUG_NODE(HTTP_Version);
    BOOST_SPIRIT_DEBUG_NODE(CHAR);
    BOOST_SPIRIT_DEBUG_NODE(DIGIT);
    BOOST_SPIRIT_DEBUG_NODE(CTL);
    BOOST_SPIRIT_DEBUG_NODE(CR);
    BOOST_SPIRIT_DEBUG_NODE(LF);
    BOOST_SPIRIT_DEBUG_NODE(SP);
    BOOST_SPIRIT_DEBUG_NODE(HT);
    BOOST_SPIRIT_DEBUG_NODE(token);
    BOOST_SPIRIT_DEBUG_NODE(separators);

#undef ASSING
        }

        const rule<ScannerT>& start() const { return top; }
    };
};


#endif // SL_INET_REQUEST_LINE_PARSAR_HPP
