/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_MESSAGE_HPP
#define SL_INET_MESSAGE_HPP

#include <iostream>
#include <map>
#include <string>
#include <vector>
#include <stdexcept>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/lexical_cast.hpp>


namespace sl { namespace http {

class message {
public :
    message() { }
    virtual ~message() { }

    /**
     * HTTP-Message Υפ֤ν貾۴ؿǤ.
     * HTTP-Request ޤϡHTTP-Response Ѥ֤ͤޤ.
     *
     * @return    int  Request/Response μѤ.
     */
    virtual int type() const=0;

    /**
     * HTTP-Message ΰܤ֤ޤ.
     * HTTP-Message  Start-Line ϡHTTP-Request  Request-Line
     * HTTP-Response  Status-Line ƱǤ.
     *
     * @return    HTTP-Message ΰܤ string.
     */
    virtual std::string start_line() const=0;

    /**
     * ꥯȤꤵƤƤξޤ.
     * ʹߡ٤ parse()  ¾ΥǡѴؿˤ
     * ʥǡ꤬ʤޤǡ text() ǡѤδؿˤä
     * ֤ͤݾڤޤ.
     */
    virtual void reset()
    {
        _headers_map.clear();
        _body.clear();
    }

    /**
     * ˥ǡɤ߹िδؿǤ.
     * ɤ߹ߤνȤơ
     * 1. start_line() Ǥ start_line() ¹Ԥޤ.
     * 2. start_line() Ǥʤ parse_line() ¹Ԥ
     * ƹԤإåեɤȤ¸Ƥޤ.
     *
     * @param    line    HTTP-Message ΰԤ string.
     */
    virtual void header_line(const std::string& line)
    {
        std::string::size_type pos = line.find(':');
        if (pos != std::string::npos) {
            std::string name  = line.substr(0, pos);
            std::string value = line.substr(pos + 1);
            msg_header(boost::trim_copy(name), boost::trim_copy(value));

        } else if (line[0] == ' ' || line[0] == '\t')
            append_last_header_value(line);
        else {
            std::cerr << __FUNCTION__ << " invalid header." << std::endl;
            throw std::runtime_error("invalid header.");
        }
    }

    /**
     *  name إå̾Ȥ"name: value" η¸ޤ.
     *  name Ʊ̾ĥإåեɤ¸ߤƤƤ
     * "name: value" ϹԤʤ졢value ͤޤƱǤ
     * value ͤǾ񤭤ޤ.
     *
     * @param    name    general_code 򻲾.
     * @param    value     code Фإåեɤ.
     */
    virtual
    void msg_header(const std::string& name, const std::string& value)
    {
        _headers_map.insert(std::make_pair(name, value));
    }

    virtual
    void msg_header(const std::string& name, int value)
    {
        _headers_map.insert(std::make_pair(name,
                            boost::lexical_cast<std::string>(value)));
    }

    /**
     *  name إåե̾ĥإåեɤ֤ͤޤ.
     * name ͤ̾ȤƤĥإåեɤʣ¸ߤƤ
     * Ƭ(ɤ߹ߤԤʤä)Ԥ֤ޤ.
     *
     * @return    إåեɤ.
     */
    virtual
    std::string msg_header(const std::string& name) const
    {
        std::multimap<std::string, std::string>::const_iterator i =
                                                    _headers_map.find(name);
        if (i != _headers_map.end())
            return i->second;

        for ( i = _headers_map.begin(); i != _headers_map.end(); i++)
            if (boost::equals(i->first, name, boost::is_iequal()))
                return i->second;

        return std::string();
    }

    /**
     *  name إåե̾Ȥƻĥإåեɤͤ
     * ꥹȤˤ֤ޤ.
     * name ͤ̾ȤƤĥإåեɤʣ¸ߤƤ
     * Ƭ(ɤ߹ߤԤʤä)إå龺Ԥ֤ޤ.
     *
     * @param    name    إå̾.
     * @return    إåեɤ.
     */
    virtual
    std::vector<std::string> msg_headers(const std::string& name) const
    {
        std::vector<std::string> result;
        std::multimap<std::string, std::string>::const_iterator i =
                                                    _headers_map.begin();
        for (; i != _headers_map.end(); i++) {
            if (boost::equals(i->first, name, boost::is_iequal()))
                result.push_back(i->second);
        }
        return result;
    }

    /**
     * ݻƤإåեɤ̾ͤΥڥޥåפȤ֤ޤ.
     *
     * @return    إåեɤ̾ͤΥޥå.
     */
    virtual
    std::multimap<std::string, std::string> msg_headers() const
    {
        return _headers_map;
    }

    /**
     *  name ǻꤵ줿إåե̾
     * ƤΥإåեɤޤ.
     *
     * @param    name    إå̾.
     */
    virtual
    void erase_header(const std::string& name)
    {
        std::multimap<std::string, std::string>::iterator i =
                                                    _headers_map.begin();
        for (; i != _headers_map.end(); i++) {
            if (boost::equals(i->first, name, boost::is_iequal()))
                _headers_map.erase(i);
        }
    }

    /**
     * ֺǸɲä줿إå˰ value Ϣ뤷ޤ.
     */
    void append_last_header_value(const std::string& value)
    {
        std::multimap<std::string, std::string>::iterator i = 
                                                    _headers_map.begin();
        std::advance(i, _headers_map.size() -1);
        i->second += value;
    }

    /**
     *  body 򥳥ƥĥܥǥȤ¸ޤ.
     * ˥ܥǥȤꤵƤƤ˴ޤ.
     *
     * @param    body    ƥĤΥܥǥȤꤹ.
     */
    virtual
    void msg_body(const std::string& body) { _body = body; }

    /**
     * HTTP-Message λĥƥĤΥܥǥ֤ޤ.
     *
     * @return    ƥĥܥǥ.
     */
    virtual
    std::string msg_body() const { return _body; }

    /**
     *  body 򥳥ƥĥܥǥȤɲäޤ.
     * ˥ܥǥȤꤵƤƤ¸ߤƤϡ
     * ͤ˰ body Ƥɲäޤ.
     *
     * @param    partial    ƥĤΥܥǥȤꤹ.
     */
    void add_body(const std::string& partial) { _body.append(partial); }

    /**
     * ꤵƤƤ HTTP-Message η֤ޤ.
     *
     * @return    HTTP-Message  string.
     */
    std::string text() const { return text_header() + msg_body(); }

    /**
     * HTTP-Message λĤ٤ƤΥإåƥȤȤ֤ޤ.
     *
     * @return    ƥȤѴإå.
     */
    virtual
    std::string text_header() const
    {
        std::string text;
        text.append(start_line() + "\r\n");

        std::multimap<std::string, std::string>::const_iterator i =
                                                _headers_map.begin();
        for (; i != _headers_map.end(); i++)
            text.append(i->first + ": " + i->second + "\r\n");
        text.append("\r\n");

        return text;
    }

    /**
     * HTTP-Message λĥܥǥ iostream Ȥ¸ޤ.
     *
     * @param    is        std::istream ֥.
     */
    void stream(std::iostream& is) { _stream = &is; }

    /**
     * HTTP-Message λĥܥǥ istream Ȥ֤ޤ.
     *
     * @return        std::iostream ֥.
     */
    std::iostream& stream() { return *_stream; }

protected :
    std::multimap<std::string, std::string> _headers_map;
    std::string _body;
    std::iostream* _stream;
};

} } // namespace sl::http


#endif // SL_INET_MESSAGE_HPP
