/*-
 * Copyright (c) 2009 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_HTTP_UTIL_REQUEST_LINE_P_HPP
#define SL_INET_HTTP_UTIL_REQUEST_LINE_P_HPP

#include <string>

#include <sl/inet/http_util/basic_parse_rules_in_BNF.hpp>


namespace sl { namespace http {

/**
 * HTTP Request-URI Υѡ饹.
 *
 * <pre>
 *
 *  3.2.2 http URL
 *  http_URL = "http:" "//" host [ ":" port ] [ abs_path [ "?" query ]]
 *
 *  5.1 ꥯȥ饤
 *  Method SP Request-URI SP HTTP-Version CRLF
 *
 *  5.1.1 ᥽å
 *  Method = "OPTIONS" | "GET" | "POST" | ... | ... | extension-method
 *  extention-method = token
 *
 *  5.1.2  Request-URI
 *  Request-URI    = "*" | absoluteURI | abs_path | authority
 *  Request-URI = "*" | absoluteURI | [ abs_path[ "?" query ]] | authority
 *
 *
 *  RFC2396 -
 *  URI-reference = [ absoluteURI | relativeURI ] [ "#" fragment ]
 *  absoluteURI   = scheme ":" ( hier_part | opaque_part )
 *  relativeURI   = ( net_path | abs_path | rel_path ) [ "?" query ]
 *
 *  hier_part     = ( net_path | abs_path ) [ "?" query ]
 *  opaque_part   = uric_no_slash *uric
 *
 *  uric_no_slash = unreserved | escaped | ";" | "?" | ":" | "@" |
 *                  "&" | "=" | "+" | "$" | ","
 *
 *  net_path      = "//" authority [ abs_path ]
 *  abs_path      = "/"  path_segments
 *  rel_path      = rel_segment [ abs_path ]
 *
 *  rel_segment   = 1*( unreserved | escaped |
 *                      ";" | "@" | "&" | "=" | "+" | "$" | "," )
 *
 *  scheme        = alpha *( alpha | digit | "+" | "-" | "." )
 *
 *  authority     = server | reg_name
 *
 *  reg_name      = 1*( unreserved | escaped | "$" | "," |
 *                      ";" | ":" | "@" | "&" | "=" | "+" )
 *
 *  server        = [ [ userinfo "@" ] hostport ]
 *  userinfo      = *( unreserved | escaped |
 *                     ";" | ":" | "&" | "=" | "+" | "$" | "," )
 *
 *  hostport      = host [ ":" port ]
 *  host          = hostname | IPv4address
 *  hostname      = *( domainlabel "." ) toplabel [ "." ]
 *  domainlabel   = alphanum | alphanum *( alphanum | "-" ) alphanum
 *  toplabel      = alpha | alpha *( alphanum | "-" ) alphanum
 *  IPv4address   = 1*digit "." 1*digit "." 1*digit "." 1*digit
 *  port          = *digit
 *
 *  path          = [ abs_path | opaque_part ]
 *  path_segments = segment *( "/" segment )
 *  segment       = *pchar *( ";" param )
 *  param         = *pchar
 *  pchar         = unreserved | escaped |
 *                  ":" | "@" | "&" | "=" | "+" | "$" | ","
 *
 *  query         = *uric
 *
 *  fragment      = *uric
 *
 *  uric          = reserved | unreserved | escaped
 *  reserved      = ";" | "/" | "?" | ":" | "@" | "&" | "=" | "+" |
 *                  "$" | ","
 *  unreserved    = alphanum | mark
 *  mark          = "-" | "_" | "." | "!" | "~" | "*" | "'" |
 *                  "(" | ")"
 *
 *  escaped       = "%" hex hex
 *  hex           = digit | "A" | "B" | "C" | "D" | "E" | "F" |
 *                          "a" | "b" | "c" | "d" | "e" | "f"
 *
 *  alphanum      = alpha | digit
 *  alpha         = lowalpha | upalpha
 *
 *  lowalpha = "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" |
 *             "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" |
 *             "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z"
 *  upalpha  = "A" | "B" | "C" | "D" | "E" | "F" | "G" | "H" | "I" |
 *             "J" | "K" | "L" | "M" | "N" | "O" | "P" | "Q" | "R" |
 *             "S" | "T" | "U" | "V" | "W" | "X" | "Y" | "Z"
 *  digit    = "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" |
 *             "8" | "9"
 *
 *  HTTP-Version = "HTTP" "/" 1*DIGIT "." 1*DIGIT
 * </pre>
 */
class request_line_p {
public :

    /** Request-URI ¸ѹ¤.  */
    struct req_type {
        std::string _method;
        std::string _uri;
        std::string _version;
        std::string _reg_name;
        std::string _scheme;
        std::string _abs_path;
        std::string _query;
    };

    /**
     * ϴؿ.
     *  s  Request-URIϤη̤ t ݻޤ.
     *
     * @param s ϤԤإå.
     * @param t Ϸ̤ݻ뤿 req_type .
     * @return Ϸ̤β.
     */
    static bool parse(const std::string& s, req_type& t)
    {
        std::string tmp = s;
        return parse((std::string&)tmp, t);
    }

    static bool parse(std::string& s, req_type& t)
    {
        std::string::iterator i = s.begin();
        std::string::iterator e = s.end();

        /* Method SP Request-URI SP HTTP-Version CRLF */

        sw(i);

        std::string::iterator p = i;

        if (!token(i, e)) { i = p; return false; }

        t._method = std::string(p, i);

        if (!eq(i, ' ')) { i = p; return false; }

        p = i;
        if (eq(i, '*'))
            t._uri = std::string(p, i);
        else if (absoluteURI(i, e, t))
            t._uri = std::string(p, i);
        else if (abs_path(i, e, t)) {
            if (eq(i, '?')) {
                std::string::iterator p2 = i;
                query(i, e);
                t._query = std::string(p2, i);
            }
            t._uri = std::string(p, i);
        }
        else if (authority(i, e))
            t._uri = std::string(p, i);

        p = i;
        if (!eq(i, ' ')) { i = p; return false; }

        p = i;
        if (http_version(i, e))
            t._version = std::string(p, i);

        return true;
    }

private :

    static
    bool absoluteURI(std::string::iterator& i, std::string::iterator& e,
                     req_type& t)
    {
        /* absoluteURI = scheme ":" ( hier_part | opaque_part ) */

        std::string::iterator p = i;

        if (scheme(i ,e)) {
            t._scheme = std::string(p, i);
            if (eq(i, ':') && (hier_part(i, e, t) || opaque_part(i, e))) {
                return true;
            }
        }
        t._scheme.clear();

        i = p;
        return false;
    }

    static
    bool hier_part(std::string::iterator& i, std::string::iterator& e, req_type& t)
    {
        /* hier_part = ( net_path | abs_path ) [ "?" query ] */

        std::string::iterator p = i;

        if (net_path(i, e, t) || abs_path(i, e, t)) {
            if (eq(i, '?')) {
                if (query(i, e)) {
                    t._query = std::string(i, p);
                    return true;
                }
            }
            return true;
        }
        i = p;
        return false;
    }

    static
    bool opaque_part(std::string::iterator& i, std::string::iterator& e)
    {
        /* opaque_part = uric_no_slash *uric */

        std::string::iterator p = i;

        if (uric_no_slash(i, e)) {
            while(uric(i)) /* nothing */ ;
            return true;
        }
        i = p;
        return false;
    }

    static
    bool uric_no_slash(std::string::iterator& i, std::string::iterator& e)
    {
        /*
         * uric_no_slash = unreserved | escaped | ";" | "?" | ":" | "@" |
         *                 "&" | "=" | "+" | "$" | ","
         */

        std::string::iterator p = i;

        if (unreserved(i) || escaped(i) || eq(i, ';') || eq(i, '?') ||
            eq(i, ':') || eq(i, '@') || eq(i, '&') || eq(i, '=') ||
            eq(i, '+') || eq(i, '$') || eq(i, ','))
            return true;
        i = p;
        return false;
    }

    static
    bool net_path(std::string::iterator& i, std::string::iterator& e, req_type& t)
    {
        /* net_path = "//" authority [ abs_path ] */
        std::string::iterator p = i;

        if (eq(i, "//") && authority(i, e)) {
            abs_path(i, e, t);
            return true;
        }
        i = p;
        return false;
    }

    static
    bool abs_path(std::string::iterator& i, std::string::iterator& e, req_type& t)
    {
        /* abs_path = "/"  path_segments */

        std::string::iterator p = i;

        if (eq(i, '/') && path_segments(i, e)) {
            t._abs_path = std::string(p, i);
            return true;
        }
        i = p;
        return false;
    }

    static 
    bool scheme(std::string::iterator& i, std::string::iterator& e)
    {
        /* scheme = alpha *( alpha | digit | "+" | "-" | "." ) */

        std::string::iterator p = i;
        if (ALPHA(i)) {
            while (ALPHA(i) || DIGIT(i) ||
                   eq(i, '+') || eq(i, '-') || eq(i, ',')) /* nothing */;
            return true;
        }
        i = p;
        return false;
    }

    static
    bool authority(std::string::iterator& i, std::string::iterator& e)
    {
        /* authority = server | reg_name */

        std::string::iterator p = i;

        if (server(i, e) || reg_name(i, e))
            return true;
        i = p;
        return false;
    }

    static
    bool reg_name(std::string::iterator& i, std::string::iterator& e)
    {
        /* 
           reg_name = 1*( unreserved | escaped | "$" | "," |
                      ";" | ":" | "@" | "&" | "=" | "+" )
         */
        std::string::iterator p = i;

        if (unreserved(i) || escaped(i) || eq(i, '$') || eq(i, ',') ||
            eq(i, ';') || eq(i, ':') || eq(i, '@') || eq(i, '&') ||
            eq(i, '=') || eq(i, '+'))
        {
            while (unreserved(i) || escaped(i) || eq(i, '$') || eq(i, ',') ||
                   eq(i, ';') || eq(i, ':') || eq(i, '@') || eq(i, '&') ||
                   eq(i, '=') || eq(i, '+'))
                /* nothing */ ;
            return true;
        }
        i = p;
        return false;
    }

    static
    bool server(std::string::iterator& i, std::string::iterator& e)
    {
        /* server = [ [ userinfo "@" ] hostport ] */
        std::string::iterator p = i;

        if (userinfo(i, e)) {
            if (!eq(i, '@')) {
                i = p;
                return false;
            }
        }
        hostport(i, e);
        return true;
    }

    static
    bool userinfo(std::string::iterator& i, std::string::iterator& e)
    {
        /*
            userinfo = *( unreserved | escaped |
                     ";" | ":" | "&" | "=" | "+" | "$" | "," )
         */
        std::string::iterator p = i;

        while (unreserved(i) || escaped(i) || eq(i, ';') || eq(i, ':') ||
            eq(i, '&') || eq(i, '=') || eq(i, '+') || eq(i, '$') || eq(i, ','))
            /* nothing */ ;
        return true;
    }

    static
    bool hostport(std::string::iterator& i, std::string::iterator& e)
    {
        /* hostport = host [ ":" port ] */
        std::string::iterator p = i;

        if (!host(i, e)) return false;

        if (eq(i, ':')) {
            if (port(i, e)) return true;
            i = p;
            return false;
        }
        return true;
    }

    static
    bool host(std::string::iterator& i, std::string::iterator& e)
    {
        /* host = hostname | IPv4address */
        std::string::iterator p = i;

        if (hostname(i, e) || IPv4address(i, e))
            return true;
        i = p;
        return false;
    }

    static
    bool hostname(std::string::iterator& i, std::string::iterator& e)
    {
        /* hostname = *( domainlabel "." ) toplabel [ "." ] */
        std::string::iterator p = i;

        for (;;) {
            std::string::iterator p = i;
            if (domainlabel(i, e) && eq(i, '.')) {
                i = p;
                break;
            }
        }

        if (!toplabel(i, e)) {
            i = p;
            return false;
        }
        eq(i, '.');

        return true;
    }

    static
    bool domainlabel(std::string::iterator& i, std::string::iterator& e)
    {
        /* domainlabel = alphanum | alphanum *( alphanum | "-" ) alphanum */
        std::string::iterator p = i;

        if (ALPHANUM(i)) return true;

        if (ALPHANUM(i)) {
            while (ALPHANUM(i) || eq(i, '-')) /* nothing */ ;
            if (ALPHANUM(i))
                return true;
        }
        i = p; 
        return false;
    }

    static
    bool toplabel(std::string::iterator& i, std::string::iterator& e)
    {
        /* toplabel = alpha | alpha *( alphanum | "-" ) alphanum */
        std::string::iterator p = i;

        if (ALPHA(i)) return true;

        if (ALPHA(i)) {
            while (ALPHANUM(i) || eq(i, '-')) /* nothing */ ;
            if (ALPHANUM(i))
                return true;
        }

        i = p; 
        return false;
    }

    static
    bool IPv4address(std::string::iterator& i, std::string::iterator& e)
    {
        /* IPv4address = 1*digit "." 1*digit "." 1*digit "." 1*digit */
        std::string::iterator p = i;

        if (DIGIT(i)) {
            while (DIGIT(i)) /* nothing */ ;
            if (!eq(i, '.')) { i = p; return false; }
        } else {
            i = p; return false;
        }
        if (DIGIT(i)) {
            while (DIGIT(i)) /* nothing */ ;
            if (!eq(i, '.')) { i = p; return false; }
        } else {
            i = p; return false;
        }
        if (DIGIT(i)) {
            while (DIGIT(i)) /* nothing */ ;
        } else {
            i = p; return false;
        }

        return true;
    }

    static
    bool port(std::string::iterator& i, std::string::iterator& e)
    {
        /* port = *digit */
        while (DIGIT(i)) /* nothing */ ;
        return true;
    }

    static
    bool path(std::string::iterator& i, std::string::iterator& e, req_type& t)
    {
        /* path = [ abs_path | opaque_part ] */

        if (abs_path(i, e, t) || opaque_part(i, e)) /* nothing */ ;

        return true;
    }

    static
    bool path_segments(std::string::iterator& i, std::string::iterator& e)
    {
        /* path_segments = segment *( "/" segment ) */

        std::string::iterator p = i;

        if (segment(i, e)) {
            while (eq(i, '/') && segment(i, e)) /* nothing */ ;
            return true;
        }
        i = p;
        return false;
    }

    static
    bool segment(std::string::iterator& i, std::string::iterator& e)
    {
        /* segment = *pchar *( ";" param ) */
        while (pchar(i)) /* nothing */ ;
        while (eq(i, ';'))
            param(i);

        return true;
    }

    static
    bool param(std::string::iterator& i)
    {
        while (pchar(i)) /* nothing */ ;
        return true;
    }

    static
    bool pchar(std::string::iterator& i)
    {
        /*
           pchar = unreserved | escaped |
                   ":" | "@" | "&" | "=" | "+" | "$" | ","
         */
        std::string::iterator p = i;

        if (unreserved(i) || escaped(i) ||
            eq(i, ';') || eq(i, '@') || eq(i, '&') || eq(i, '=') ||
            eq(i, '+') || eq(i, '$') || eq(i, ','))
            return true;
        i = p;
        return false;
    }

    static
    bool query(std::string::iterator& i, std::string::iterator& e)
    {
        while (uric(i)) /* nothing */ ;
        return true;
    }

    static
    bool fragment(std::string::iterator& i, std::string::iterator& e)
    {
        while (uric(i)) /* nothing */ ;
        return true;
    }

    static
    bool uric(std::string::iterator& i)
    {
        /* uric = reserved | unreserved | escaped */

        std::string::iterator p = i;

        if (reserved(i) || unreserved(i) || escaped(i))
            return true;
        i = p;
        return false;
    }

    static
    bool reserved(std::string::iterator& i)
    {
        /*
           reserved = ";" | "/" | "?" | ":" | "@" | "&" | "=" | "+" |
                      "$" | ","
         */
        std::string::iterator p = i;

        if (eq(i, ';') || eq(i, '/') || eq(i, '?') || eq(i, ':') || 
            eq(i, '@') || eq(i, '&') || eq(i, '=') || eq(i, '+') ||
            eq(i, '$') || eq(i, ','))
            return true;

        i = p;
        return false;
    }

    static
    bool unreserved(std::string::iterator& i)
    {
        /* unreserved = alphanum | mark */
        std::string::iterator p = i;

        if (ALPHANUM(i) || mark(i))
            return true;
        i = p;
        return false;
    }

    static
    bool mark(std::string::iterator & i)
    {
        /* mark = "-" | "_" | "." | "!" | "~" | "*" | "'" | "(" | ")" */
        std::string::iterator p = i;

        if (eq(i, '-') || eq(i, '_') || eq(i, '.') || eq(i, '!') || 
            eq(i, '~') || eq(i, '*') || eq(i, '\'') || eq(i, '(') || eq(i, ')'))
            return true;
        i = p;
        return false;
    }

    static
    bool escaped(std::string::iterator & i)
    {
        /* escaped = "%" hex hex */
        std::string::iterator p = i;

        if (eq(i, '%') && hex(i) && hex(i))
            return true;
        i = p;
        return false;
    }

    static
    bool hex(std::string::iterator & i)
    {
        /* hex = digit | A | B | C | D | E | F | a | b | c | d | e | f */

        std::string::iterator p = i;

        if (DIGIT(i) ||
            eq(i, 'A') || eq(i, 'B') || eq(i, 'C') || eq(i, 'D') ||
            eq(i, 'E') || eq(i, 'F') ||
            eq(i, 'a') || eq(i, 'b') || eq(i, 'c') || eq(i, 'd') ||
            eq(i, 'e') || eq(i, 'f'))
            return true;
        i = p;
        return false;
    }

    static
    bool http_version(std::string::iterator& i, std::string::iterator& e)
    {
        std::string::iterator p = i;

        if (eq(i, "HTTP/") && DIGIT(i)) {
            while (DIGIT(i)) /* nothing */ ;

            if (eq(i, '.') && DIGIT(i)) {
                while (DIGIT(i)) /* nothing */ ;
                if (i == e || eq(i, '\r') || eq(i, '\n'))
                    return true;
            }
        }
        i = p;
        return false;
    }
};

} }  // namespace sl::http


#endif // SL_INET_HTTP_UTIL_REQUEST_LINE_P_HPP
