/*-
 * Copyright (c) 2009 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_HTTP_BASIC_PARSE_RULES_IN_BNF_HPP
#define SL_HTTP_BASIC_PARSE_RULES_IN_BNF_HPP

#include <string>

namespace sl { namespace http {

/**
 * ȾѥڡȽ.
 *
 *  i Ⱦѥڡؤ i ʤޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i ȾѥڡȽ.
 */
inline bool sw(std::string::iterator& i)
{
    while (*i == ' ') ++i;
    return true;
}

/**
 * ʸȤȽؿ. 
 * 
 *  i λؤȰ ch ƱǤ i Ŀʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @param ch ʸ.
 * @return i Ƚ.
 */
inline bool eq(std::string::iterator& i, const char ch)
{
    if (*i == ch) { ++i; return true; }
    return false;
}

/**
 * ʸȤȽؿ. 
 * 
 *  i λؤȰ s ƱǤϤʸʬ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @param s ʸ.
 * @return i Ƚ.
 */
inline bool eq(std::string::iterator& i, const char *s)
{
    while (*s != 0 && *i == *s) { ++i; ++s; }
    return *s == 0;
}

/**
 * Ƚؿ. 
 *
 * UPALPHA = <US-ASCII "A" .. "Z">
 * 
 *  i λؤ褬ʸǤ뤫Ƚꤷޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool is_UPALPHA(std::string::iterator& i)
{
    return *i >= 'A' && *i <= 'Z';
}

/**
 * Ƚؿ. 
 *
 * UPALPHA = <US-ASCII "A" .. "Z">
 * 
 *  i λؤ褬ʸǤ i Ĥʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool UPALPHA(std::string::iterator& i)
{
    if (is_UPALPHA(i)) { ++i; return true; }
    return false;
}

/**
 * Ƚؿ. 
 *
 * LOALPHA = <US-ASCII "a" .. "z">
 * 
 *  i λؤ褬ʸǤ뤫Ƚꤷޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool is_LOALPHA(std::string::iterator& i)
{
    return *i >= 'a' && *i <= 'z';
}

/**
 * Ƚؿ. 
 *
 * LOALPHA = <US-ASCII "a" .. "z">
 * 
 *  i λؤ褬ʸǤ i Ĥʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool LOALPHA(std::string::iterator& i)
{
    if (is_LOALPHA(i)) { ++i; return true; }
    return false;
}

/**
 * ʸȽؿ. 
 *
 * ALPHA = UPALPHA | LOALPHA
 * 
 *  i λؤ褬ѻǤ뤫Ƚꤷޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool is_ALPHA(std::string::iterator& i)
{
    return is_UPALPHA(i) || is_LOALPHA(i);
}

/**
 * ʸȽؿ. 
 *
 * ALPHA = UPALPHA | LOALPHA
 * 
 *  i λؤ褬ѻǤ i Ĥʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool ALPHA(std::string::iterator& i)
{
    if (is_ALPHA(i)) { ++i; return true; }
    return false;
}

/**
 * ʸȽؿ. 
 * 
 *  i λؤ褬ڤʸǤ뤫Ƚꤷޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool is_separators(std::string::iterator& i)
{
    if (*i == '(' || *i == ')' || *i == '<' || *i == '>' || *i == '@' ||
        *i == ',' || *i == ';' || *i == ':' || *i == '\\'|| *i == '"' ||
        *i == '/' || *i == '[' || *i == ']' || *i == '?' || *i == '=' ||
        *i == '{' || *i == '}' || *i == ' ' || *i == '\t')
        return true;
    return false;
}

/**
 * ʸȽؿ. 
 * 
 *  i λؤ褬ڤʸǤ i Ĥʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool separators(std::string::iterator& i)
{
    if (is_separators(i)) { ++i; return true; }
    return false;
}

/**
 * ʸȽؿ. 
 *
 * CTL = <US-ASCII 0-31 ƥå| DEL(127) >
 * 
 *  i λؤ褬ʸǤ뤫Ƚꤷޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool is_CTL(std::string::iterator& i)
{
    return *i >= 0 && *i <= 31 && *i != 127;
}

/**
 * ʸȽؿ. 
 *
 * CTL = <US-ASCII 0-31 ƥå| DEL(127) >
 * 
 *  i λؤ褬ʸǤ i Ĥʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool CTL(std::string::iterator& i)
{
    if (is_CTL(i)) { ++i; return true; }
    return false;
}

/**
 * ʸȽؿ. 
 *
 * CHAR = <US-ASCII (0-127) >
 * 
 *  i λؤ褬̾ʸǤ뤫Ƚꤷޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool is_CHAR(std::string::iterator& i)
{
    /* xxxxx */
    return *i > 0 && static_cast<unsigned char>(*i) <= 127;
}

/**
 * ʸȽؿ. 
 *
 * CHAR = <US-ASCII (0-127) >
 * 
 *  i λؤ褬̾ʸǤ i Ĥʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool CHAR(std::string::iterator& i)
{
    if (is_CHAR(i)) { ++i; return true; }
    return false;
}

/**
 * ʸȽؿ. 
 *
 * TEXT = <CTL LWS ޤह٤ƤOCTET>
 * 
 *  i λؤ褬ʸʳǤ뤫Ƚꤷޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool is_TEXT(std::string::iterator& i)
{
    return !is_CTL(i);
}

/**
 * ʸȽؿ. 
 *
 * TEXT = <CTL LWS ޤह٤ƤOCTET>
 * 
 *  i λؤ褬ʸʳǤ i Ĥʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool TEXT(std::string::iterator& i)
{
    if (is_TEXT(i)) { ++i; return true; }
    return false;
}

/**
 * ʸȽؿ. 
 *
 * DIGIT = <US-ASCII "0" .. "9">
 * 
 *  i λؤ褬ʸǤ뤫Ƚꤷޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool is_DIGIT(std::string::iterator& i)
{
    return *i == '0' || *i == '1' || *i == '2' || *i == '3' || *i == '4' ||
           *i == '5' || *i == '6' || *i == '7' || *i == '8' || *i == '9';
}

/**
 * ʸȽؿ. 
 *
 * DIGIT = <US-ASCII "0" .. "9">
 * 
 *  i λؤ褬ʸǤ i Ĥʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool DIGIT(std::string::iterator& i)
{
    if (is_DIGIT(i)) { ++i; return true; }
    return false;
}

/**
 * ʸȽؿ. 
 * 
 *  i λؤ褬ѿǤ뤫Ƚꤷޤ.
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool is_ALPHANUM(std::string::iterator& i)
{
    if (is_ALPHA(i) || is_DIGIT(i)) { return true; }
    return false;
}

/**
 * ʸȽؿ. 
 * 
 *  i λؤ褬ѿǤ i Ĥʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool ALPHANUM(std::string::iterator& i)
{
    if (is_ALPHANUM(i)) { ++i; return true; }
    return false;
}

/**
 * ʸȽؿ. 
 * 
 *  i λؤʸ󤬱ѻδ֤ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool token(std::string::iterator& i, std::string::iterator& e)
{
    std::string::iterator p = i;
    while (!is_separators(i) && !is_CTL(i) && CHAR(i))
        /* nothing */ ;
    return i != p;
}

/**
 * ʸȽؿ. 
 *
 * qvalue = ( "0" [ "." 0*3DIGIT ]) | ( "1" [ "." 0*3DIGIT ])
 * 
 *  i λؤʸBNF qvalue ǤФ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool qvalue(std::string::iterator& i, std::string::iterator& e)
{
    std::string::iterator p = i;
    if (*i == '0' || *i == '1') {
        i++;
        if (*i != '.')
            return true;
        if (eq(i, '.') && DIGIT(i) && DIGIT(i) && DIGIT(i))
            /* nothing */ ;
        return true;
    }
    i = p;
    return false;
}

/**
 * ʸȽؿ. 
 *
 * type = token
 * 
 *  i λؤʸBNF type ǤФ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool type(std::string::iterator& i, std::string::iterator& e)
{
   return token(i, e);
}

/**
 * ʸȽؿ. 
 *
 * subtype = token
 * 
 *  i λؤʸBNF subtype ǤФ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool subtype(std::string::iterator& i, std::string::iterator& e)
{
   return token(i, e);
}

/**
 * ʸȽؿ. 
 *
 * attribute = token
 * 
 *  i λؤʸBNF attribute ǤФ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool attribute(std::string::iterator& i, std::string::iterator& e)
{
   return token(i, e);
}

/**
 * ʸȽؿ. 
 *
 * qdtext = <<">  TEXT >"
 * 
 *  i λؤʸBNF qdtext ǤФ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool qdtext(std::string::iterator& i, std::string::iterator& e)
{
    /* xxxxx */
    while (*i != '"' && *i != ' ' && TEXT(i)) /* nothing */;
    return true;
}

/**
 * ʸȽؿ. 
 *
 * quoted-pair = "\" CHAR
 * 
 *  i λؤʸBNF quoted-pair ǤФ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool quoted_pair(std::string::iterator& i, std::string::iterator& e)
{
    return eq(i, '\\') && CHAR(i);
}

/**
 * ʸȽؿ. 
 *
 * quoted-string = ( <"> *( qdtext | quoted-pair ) <"> )
 * 
 *  i λؤʸBNF quoted-string ǤФ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool quoted_string(std::string::iterator& i, std::string::iterator& e)
{
    std::string::iterator p = i;
    if (eq(i, '"')) {
        for (;;) {
            std::string::iterator j = i;
            if (qdtext(i, e) || quoted_pair(i, e)) {
                if (i == j) break;
                if (i == e) return false;
            }
        }
        if (eq(i, '"')) return true;
    }
    i = p;
    return false;
}

/**
 * ʸȽؿ. 
 *
 * value            = token | quoted-string
 * 
 *  i λؤʸBNF value ǤФ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool value(std::string::iterator& i, std::string::iterator& e)
{
   /* value            = token | quoted-string */
   return token(i, e) || quoted_string(i, e);
}

/**
 * ʸȽؿ. 
 *
 * parameter        = attribute "=" value
 * 
 *  i λؤʸBNF parameter ǤФ i ʤޤ. 
 * @param i ȽоʸΥƥ졼.
 * @return i Ƚ.
 */
inline bool parameter(std::string::iterator& i, std::string::iterator& e)
{
    std::string::iterator p = i;
    if (eq(i, ';') && sw(i) && attribute(i, e) && sw(i) &&
        eq(i, '=') && sw(i) && value(i, e))
        return true;
    i = p;
    return false;
}

} } // namespace sl::http

#endif // SL_HTTP_BASIC_PARSE_RULES_IN_BNF_HPP
