/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_HTTP_UTIL_HPP
#define SL_INET_HTTP_UTIL_HPP

#include <iostream>
#include <sstream>
#include <string>
#include <functional>
#include <map>
#include <vector>

#include <sl/inet/http_util/basic_parse_rules_in_BNF.hpp>

namespace sl { namespace http {

typedef std::vector<std::pair<std::string, std::string> > parameter_map;

struct parameter_first :
    std::unary_function <parameter_map::value_type,
                         parameter_map::value_type::first_type>
{
    const parameter_map::value_type::first_type&
    operator()(const parameter_map::value_type& value) const
    {
        return value.first;
    }
};

struct parameter_second :
    std::unary_function <parameter_map::value_type,
                         parameter_map::value_type::second_type>
{
    const parameter_map::value_type::second_type&
    operator()(const parameter_map::value_type& value) const
    {
        return value.second;
    }
};

struct parameter_first_equal :
    std::unary_function <bool, parameter_map::value_type>
{
    parameter_first_equal(const std::string& s) : _value(s) { }

    bool operator()(const parameter_map::value_type& r) const
    {
        return _value == r.first;
    }

    std::string _value;
};


template <class MapType>
struct __parameter_to_map
: std::unary_function <sl::http::parameter_map::value_type,
                       typename MapType::value_type>
{
    typename MapType::value_type
    operator()(const sl::http::parameter_map::value_type& v) const
    {
        return std::make_pair(v.first, v.second);
    }
};


inline std::map<std::string, std::string>
parameter_to_map(const sl::http::parameter_map& p)
{
    std::map<std::string, std::string> m;
    std::transform(p.begin(), p.end(),
        std::inserter(m, m.end()),
            __parameter_to_map<std::map<std::string, std::string> >());
    return m;
}

inline std::multimap<std::string, std::string>
parameter_to_multimap(const sl::http::parameter_map& p)
{
    std::multimap<std::string, std::string> m;
    std::transform(p.begin(), p.end(),
        std::inserter(m, m.end()),
            __parameter_to_map<std::multimap<std::string, std::string> >());
    return m;
}

inline parameter_map parameter_split(const std::string& s)
{
    parameter_map result;

    std::string::size_type cur, pos;
    size_t length = s.length();

    for (cur = 0; cur <= length; ) {
        pos = s.find('=', cur);
        if (pos == std::string::npos) {
            result.push_back(std::make_pair(s.substr(cur), ""));
            break;
        }
        std::string name(s.substr(cur, pos - cur));
        cur = pos + 1;

        pos = s.find('&', cur);
        if (pos != std::string::npos) {
            result.push_back(std::make_pair(name,s.substr(cur,pos - cur)));
            cur = pos + 1;
        } else {
            result.push_back(std::make_pair(name, s.substr(cur)));
            break;
        }
    }
    return result;
}

/**
 *  c  Http-Request  Method ȤƵƤʸǤ뤫
 * ȽԤʤޤ.
 *
 * @param    c    ȽΥåȤȤʤʸ.
 *
 *  Method SP Request-URI SP HTTP-Version CRLF
 *
 *  Method         = "OPTIONS"                ; It is equivalent to 'token.
 *                 | "GET"                    ; It is equivalent to 'token.
 *                 | "HEAD"                   ; It is equivalent to 'token.
 *                 | "POST"                   ; It is equivalent to 'token.
 *                 | "PUT"                    ; It is equivalent to 'token.
 *                 | "DELETE"                 ; It is equivalent to 'token.
 *                 | "TRACE"                  ; It is equivalent to 'token.
 *                 | "CONNECT"                ; It is equivalent to 'token.
 *                 | extension-method
 *  extension-method = token
 */
inline bool verification_Method(const std::string& s)
{
    std::string t = s;
    std::string::iterator i = t.begin();
    std::string::iterator e = t.end();

    return token(i, e) && i == e;
}

/**
 *  c  Http-Request  HTTP-Version ȤƵƤʸǤ뤫
 * ȽԤʤޤ.
 *
 * @param    c    ȽΥåȤȤʤʸ.
 *  HTTP-Version   = "HTTP" "/" 1*DIGIT "." 1*DIGIT
 */
inline bool verification_HTTP_Version(const std::string& s)
{
    std::string t = s;
    std::string::iterator i = t.begin();
    std::string::iterator e = t.end();

    if (eq(i, "HTTP/") && DIGIT(i)) {
        while (DIGIT(i)) /* nothing */ ;

        if (eq(i, '.') && DIGIT(i)) {
            while (DIGIT(i)) /* nothing */ ;
            if (i == e || eq(i, '\r') || eq(i, '\n'))
                return true;
        }
    }
    return false;
}

inline char atoh(char c)
{
    if (('0' <= c) && (c <= '9')) return c - '0';
    if (('A' <= c) && (c <= 'F')) return c - 'A' + 10;
    if (('a' <= c) && (c <= 'f')) return c - 'a' + 10;
    return 0;
}

inline char htoa(char c)
{
    if ((0x0 <= c) && (c <= 0x9)) return c + '0';
    if ((0xa <= c) && (c <= 0xf)) return c + 'A' - 0xa;
    return 0;
}


/**
 *  s URLǥɤԤʤäʸ֤ޤ.
 *
 * @param    s    URLǥоݤʸ.
 * @return    URLǥɸʸ.
 */
inline std::string urldecode(const std::string& s)
{
    std::string result;

    std::string::const_iterator i = s.begin();
    for (; i != s.end(); i++) {
        if (*i == '%' && i+1 != s.end() && i+2 != s.end())
            result.append(1, static_cast<char>((atoh(*++i) << 4) + atoh(*++i)));
        else if (*i == '+')
            result.append(1, ' ');
        else
            result.append(1, *i);
    }
    return result;
}

/**
 *  s URL󥳡ɤԤʤäʸ֤ޤ.
 *
 * @param    s    URL󥳡оݤʸ.
 * @return    URL󥳡ɸʸ.
 */
inline std::string urlencode(const std::string& s)
{
    std::string result;

    std::string::const_iterator i = s.begin();
    for (; i != s.end(); i++) {
        if (*i == ' ')
            result.append(1, '+');
        else if (!isprint(*i)) {
            result.append(1, '%');
            result.append(1, htoa((*i & 0xf0) >> 4));
            result.append(1, htoa(*i & 0x0f));
        } else
            result.append(1, *i);
    }
    return result;
}


template <class UnaryFunction>
struct __urldecode_to
    : std::unary_function <sl::http::parameter_map::value_type,
                           sl::http::parameter_map::value_type::first_type>
{
    __urldecode_to(const UnaryFunction& f) : _f(f) { }

    template <class T>
    std::string operator()(const T& v) const
    {
        return sl::http::urldecode(_f(v));
    }
    const UnaryFunction&    _f;
};

template <class FuncType>
inline __urldecode_to<FuncType> url_decoder(const FuncType& t)
{
    return __urldecode_to<FuncType>(t);
}


template <class RequestType>
inline bool has_parameter(const RequestType& r)
{
    if (r.method() == "POST" &&
        r.msg_header("Content-Type") == "application/x-www-form-urlencoded")
        return true;
    return false;
}

template <class RequestType>
inline bool is_chunked(const RequestType& r)
{
    std::string e = r.msg_header("Transfer-Encoding");

    if (r.version() == "HTTP/1.1" &&
        e.find("identity") == std::string::npos)
        return true;
    return false;
}


} }  // namespace sl::http


#endif // SL_INET_HTTP_UTIL_HPP
