/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_HTTP_REQUEST_HPP
#define SL_INET_HTTP_REQUEST_HPP

#include <ostream>
#include <string>
#include <vector>
#include <utility>
#include <boost/algorithm/string/predicate.hpp>

#include <sl/inet/http_util.hpp>
#include <sl/inet/message.hpp>

namespace sl { namespace http {


class http_request : public message {
public :

    enum {
        empty,
        normality,
        request_too_long,
        bad_request,
        broken_connection
    };

    http_request()
        : _state(empty) { }

    virtual ~http_request() { }

    void state(int i) { _state = i; }
    int state() const { return _state; }

    void reset()
    {
        message::reset();
        _state = empty;
#if 0
        _method.clear();
        _uri.clear();
        _version.clear();
        _reg_name.clear();
        _scheme.clear();
        _userinfo.clear();
        _host.clear();
        _port.clear();
        _abs_path.clear();
        _query.clear();
#endif
        _param_map.clear();
    }

    operator std::string () const
    {
        return this->text();
    }

    //////////////////////////////////////////////////////////////////////
    // Inherited pure virtual funtions.
    //

    virtual int type() const
    {
        return 0;
    }

    virtual std::string start_line() const
    {
        return request_line();
    }

    //////////////////////////////////////////////////////////////////////
    // HTTP-Request Implement funtions.
    //
    std::string request_line() const
    {
        return _method + " " + _uri + " " + _version;
    }

    void method(const std::string& s) { _method = s; }
    std::string method() const { return _method; }

    void uri(const std::string& s) { _uri = s; }
    std::string uri() const { return _uri; }

    void version(const std::string& s) { _version = s; }
    std::string version() const { return _version; }

    void reg_name(const std::string& s) { _reg_name = s; }
    std::string reg_name() const { return _reg_name; }

    void scheme(const std::string& s) { _scheme = s; }
    std::string scheme() const { return _scheme; }

    void userinfo(const std::string& s) { _userinfo = s; }
    std::string userinfo() const { return _userinfo; }

    void host(const std::string& s) { _host = s; }
    std::string host() const { return _host; }

    void port(const std::string& s) { _port = s; }
    std::string port() const { return _port; }

    void abs_path(const std::string& s) { _abs_path = s; }
    std::string abs_path() const { return _abs_path; }

    void query(const std::string& s) { _query = s; }
    std::string query() const { return _query; }

    bool keep_alive() const
    {
        return boost::equals(msg_header("Connection"), "keep-alive", boost::is_iequal());
    }

    int content_length() const
    {
        std::string s = msg_header("Content-Length");
        if (s.empty())
            return -1;

        try {
            return boost::lexical_cast<int>(s);
        } catch (std::exception& e) {
            std::cerr << e.what() << std::endl;
        }
        return -1;
    }

    bool is_post_with_wwwform() const
    {
        if (_method == "POST" &&
            msg_header("Content-Type") == "application/x-www-form-urlencoded")
            return true;
        return false;
    }

    bool is_get_with_query() const
    {
        if ((_method == "GET" || _method == "HEAD") && !_query.empty())
            return true;
        return false;
    }

    void parameter(const parameter_map& param_map)
    {
        _param_map = param_map;
    }

    const parameter_map& parameter() const
    {
        return _param_map;
    }

private :

    int _state;
    std::string _method;
    std::string _uri;
    std::string _version;
    std::string _reg_name;
    std::string _scheme;
    std::string _userinfo;
    std::string _host;
    std::string _port;
    std::string _abs_path;
    std::string _query;
    parameter_map _param_map;
};


inline std::ostream& operator<<(std::ostream& os, const http_request& req)
{
    return os << static_cast<std::string>(req);
}

} } // namespace sl::http


#endif // SL_INET_HTTP_REQUEST_HPP
