/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_DERIVED_TYPE_POOL_HPP
#define SL_INET_DERIVED_TYPE_POOL_HPP

#include <deque>
#include <boost/thread/mutex.hpp>
#include <boost/thread/condition.hpp>

namespace sl {

template <class ObjectType>
class derived_type_pool {
public :
    struct derived_type_pool_init_args_type {
        derived_type_pool_init_args_type(size_t _1=100, size_t _2=10, size_t _3=0)
            : _max_objects(_1),
              _max_spare_objects(_2),
              _min_spare_objects(_3)
        { }

        size_t _max_objects;
        size_t _max_spare_objects;
        size_t _min_spare_objects;
    };

    typedef derived_type_pool_init_args_type init_args_type;

    derived_type_pool()
        : _max_objects(0), _spare_objects(0), _using_objects(0)
    { }

    ~derived_type_pool()
    {
        destroy();
    }

    void init(const init_args_type& args)
    {
        _max_objects = args._max_objects;

        for (size_t i = 0; i < _max_objects; i++) {
            ObjectType* c = new ObjectType();
            if (!c)
                throw std::runtime_error("derived_type_pool::init");
            _real_pool.push_back(c);
        }
    }

    void destroy()
    {
        boost::mutex::scoped_lock lock(_mutex);

        typename std::deque<ObjectType*>::iterator i = _real_pool.begin();
        for (; i != _real_pool.end(); i++)
            delete *i;
        _real_pool.clear();
    }

    ObjectType* get()
    {
        boost::mutex::scoped_lock lock(_mutex);

        ObjectType* c;

        while (_real_pool.empty())
            _cond.wait(lock);

        c = _real_pool.front();
        _real_pool.pop_front();
        ++_using_objects;
        return c;
    }

    template <class BasedType>
    void release(BasedType *b)
    {
        ObjectType* c = dynamic_cast<ObjectType*>(b);

        boost::mutex::scoped_lock lock(_mutex);
        if (!c)
            return;
        _real_pool.push_back(c);
        --_using_objects;
        _cond.notify_one();
    }

    template <class BasedType>
    void exclude(BasedType *b)
    {
        ObjectType* c = dynamic_cast<ObjectType*>(b);

        boost::mutex::scoped_lock lock(_mutex);
        if (!c)
            return;
        delete c;

        ObjectType* p = new ObjectType();
        if (!p)
            throw std::runtime_error("basic_pool::exclude");
        _real_pool.push_back(p);
        --_using_objects;
        _cond.notify_one();
    }

    size_t size()
    {
        boost::mutex::scoped_lock lock(_mutex);
        return _real_pool.size();
    }

    size_t use_count()
    {
        boost::mutex::scoped_lock lock(_mutex);
        return _using_objects;
    }

private :
    boost::mutex _mutex;
    boost::condition _cond;
    std::deque<ObjectType*> _real_pool;
    size_t _max_objects;
    size_t _spare_objects;
    size_t _using_objects;
};


} // namespace sl;

#endif // SL_INET_DERIVED_TYPE_POOL_HPP
