/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_CONNECTOR_HPP
#define SL_INET_CONNECTOR_HPP

#include <iostream>
#include <map>
#include <string>
#include <exception>
#include <vector>
#include <boost/bind.hpp>
#include <boost/thread.hpp>

#include <sl/inet/basic_pool.hpp>

namespace sl {

template <
    class Acceptor,
    class Connection,
    class Pool=sl::basic_pool<Connection>
    >
class connector {
public :

    struct connector_init_args_type {
        connector_init_args_type(size_t _1=10) : _keep_alive_max(_1) { }
        size_t _keep_alive_max;
    };

    typedef connector_init_args_type init_args_type;

    typedef Acceptor acceptor_type;
    typedef Connection connection_type;
    typedef Pool pool_type;
    typedef connector<Acceptor, Connection, Pool> this_type;

    typedef typename this_type::init_args_type connector_args;
    typedef typename Acceptor::init_args_type acceptor_args;
    typedef typename Connection::init_args_type connection_args;
    typedef typename Pool::init_args_type pool_args;

    typedef boost::function<void (Connection&)> accept_handle_type;
    typedef std::vector<boost::thread*> thread_list_type;
    typedef std::vector<connection_type*> connection_list_type;

    connector(accept_handle_type h) : _handler(h) { }
    virtual ~connector() { destroy(); }

    void init(const connector_args& connector_init,
              const acceptor_args& acceptor_init,
              const connection_args& connection_init=connection_args(),
              const pool_args& pool_init=pool_args())
    {
        boost::mutex::scoped_lock lock(_mutex);
        _is_continue = true;

        _keep_alive_max = connector_init._keep_alive_max;
        _keep_alive_cur = 0;
        _acceptor.init(acceptor_init);
        _pool.init(pool_init);
        _connection_init_value = connection_init;

        _do_join_all =
            new boost::thread(boost::bind(&this_type::do_join_all, this));
    }

    void destroy() throw()
    {
        boost::mutex::scoped_lock lock(_mutex);
        if (!_is_continue)
            return;
        _is_continue = false;
        lock.unlock();

        _acceptor.destroy();
        _pool.destroy();
        _do_join_all->join();
        delete _do_join_all;
    }

    void accept() throw()
    {    
        boost::mutex::scoped_lock lock(_mutex);
        lock.unlock();
        for (;;) {
            lock.lock();
            if (!_is_continue)
                break;
            lock.unlock();

            Connection* c = _pool.get();
            c->accept(boost::bind(&this_type::accept_h, this, _1));
            c->disconnect(boost::bind(&this_type::disconnect_h, this,_1));
            c->keep_alive(boost::bind(&this_type::keep_alive_h, this));
            c->keep_alive_rel(boost::bind(&this_type::keep_rel_h, this));
            c->init(_connection_init_value);

            try {
                _acceptor.accept(c->ref_to_socket());
            } catch (std::exception& e) {
                exclude(c);
                continue;
            }

            /*
             * thread  _connection_map.insert λޤǤå.
             *
             * thread  Connection::run Фᡢ
             * _connection_map.insert ǥåɤݻ
             * Connection::run λȡcollect_relating_thread ˤ
             * insert  _connection_map 饹åɤ򸡺Ƥޤ.
             * ( disconnect_h ---> collect_relating_thread )
             *
             *  _connection_map ˤåɤʤΤƤޤ
             * 夫鲼 _connection_map.insert ¹ԤƤޤᡢ
             * ̥åɤβϳˤʤ.
             *
             * ʤΤǡǤ å insert λޤ
             * åԤʤäƤ.
             */
               lock.lock();
            for (;;) {
                   boost::thread* p = 0;
                try {
                    p = new boost::thread(boost::bind(&Connection::run, c));
                } catch (boost::thread_resource_error& e) {
                    _cond.notify_one();
                    continue;
                }
                   _connection_map.insert(std::make_pair(c, p));
                break;
            }
            lock.unlock();
        }
    }

    void accept_handler(accept_handle_type h)
    {
        _handler = h;
    }

private :
    void accept_h(Connection *c)
    {
        _handler(*c);
    }

    void disconnect_h(Connection *c)
    {
        collect_relating_thread(c);
    }

    bool keep_alive_h()
    {
        boost::mutex::scoped_lock lock(_keep_alive_mutex);
        if (_keep_alive_cur < _keep_alive_max) {
            ++_keep_alive_cur;
            return true;
        }
        return false;
    }

    void keep_rel_h()
    {
        boost::mutex::scoped_lock lock(_keep_alive_mutex);
        if (_keep_alive_cur > 0)
            --_keep_alive_cur;
    }

    void exclude(Connection *c)
    {
        collect_relating_thread(c);
    }

    void collect_relating_thread(Connection *c)
    {
        boost::mutex::scoped_lock lock(_mutex);
        typename std::map<Connection*, boost::thread*>::iterator i =
                                                        _connection_map.find(c);
        if (i != _connection_map.end()) {
            _collected_connections.push_back(i->first);
            _collected_threads.push_back(i->second);
            _connection_map.erase(i);
            _cond.notify_one();
        }
    }

    void do_join_all()
    {
        for (;;) {
            boost::mutex::scoped_lock lock(_mutex);

            boost::xtime t;
            while (_collected_connections.empty() && _collected_threads.empty())
            {
                boost::xtime_get(&t, boost::TIME_UTC);
                t.sec += 1;
                _cond.timed_wait(lock, t);
                if (!_is_continue)
                    return;
            }

            thread_list_type::iterator i2 = _collected_threads.begin();
            for (; i2 != _collected_threads.end(); i2++) {
                (*i2)->join();
                delete *i2;
            }
            _collected_threads.clear();

            typename connection_list_type::iterator i1 =
                                        _collected_connections.begin();
            for (; i1 != _collected_connections.end(); i1++)
                _pool.release(*i1);
            _collected_connections.clear();
        }
    }

    acceptor_type _acceptor;
    pool_type _pool;
    connection_args _connection_init_value;
    size_t _keep_alive_max;
    size_t _keep_alive_cur;

    std::map<Connection*, boost::thread*> _connection_map;
    thread_list_type _collected_threads;
    connection_list_type _collected_connections;
    boost::thread* _do_join_all;
    boost::mutex _mutex;
    boost::mutex _keep_alive_mutex;
    boost::condition _cond;
    bool _is_continue;
    accept_handle_type _handler;
};


} // namespace sl

#endif // SL_INET_CONNECTOR_HPP
