/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_BASIC_CONNECTION_HPP
#define SL_INET_BASIC_CONNECTION_HPP

#include <sys/select.h>

#include <iostream>
#include <string>
#include <boost/function.hpp>

#include <sl/inet/timer.hpp>

namespace sl {


template <class SocketType, class StreamType, class Service>
class basic_connection
{
public :
    typedef SocketType socket_type;
    typedef StreamType stream_type;
    typedef typename Service::request_type         request_type;
    typedef typename Service::request_reader_type  request_reader_type;
    typedef typename Service::response_type        response_type;
    typedef typename Service::response_writer_type response_writer_type;

    struct init_args_type {
        init_args_type(time_t connection_wait_time=15,
                       time_t processing_wait_time=60,
                       time_t keep_alive_time=30)
            : _conn_time(connection_wait_time),
              _proc_time(processing_wait_time),
              _keep_alive_time(keep_alive_time)
        { }
        time_t _conn_time;
        time_t _proc_time;
        time_t _keep_alive_time;
    };

    basic_connection()
    {
    }

    virtual ~basic_connection()
    {
    }

    socket_type& ref_to_socket() { return _socket; }
    stream_type stream() { return _stream; }
    request_type& request() { return _request; }
    response_type& response() { return _response; }
    response_writer_type& writer() { return _response_writer; }

    void accept(const boost::function<void (basic_connection*)>& h)
    {
        _accept_h = h;
    }

    void disconnect(const boost::function<void (basic_connection*)>& h)
    {
        _disc_h = h;
    }

    void keep_alive(const boost::function<bool (void)>& h)
    {
        _keep_alive_h = h;
    }

    void keep_alive_rel(const boost::function<void (void)>& h)
    {
        _keep_rel_h = h;
    }

    void init(const init_args_type& args)
    {
         _init_args = args;
    }

    void destroy()
    {
        _timer.stop();
        socket_close();
    }

    void run()
    {
        _stream.reset(_socket);
        _stream.clear();
        _timer.timeout(boost::bind(&basic_connection::socket_close, this));

        /*
         * ̤ƤΥޤꤹ.
         * ꥯȻ Keep-Alive ¸ߤʤᡢ
         * ѥ᡼ _conn_time ͤѤ.
         *
         *  Keep-Alive ޤ׵ǤС׵ԤѤΥޤ
         * Keep-Alive إå¸ߤ͡⤷
         * ѥ᡼ _keep_alive_time ͤѤ.
         */
        time_t keep_alive_time = _init_args._conn_time;
        _keep_alive_possible = _keep_alive_h();

        try {
            for (;;) {
                if (!wait_data_or_timeout(_socket._socket, _stream, keep_alive_time))
                {
                    std::cerr << "!wait_data_or_timeout" << std::endl;
                    break;
                }

                _stream.rlength(-1);
                _request.reset();
                _response.reset();
                _timer.restart(_init_args._proc_time);

                request_reader_type::parse(_stream, _request);
                if (_request.state() != request_type::normality)
                    break;

                _timer.stop();

                _stream.rlength(_request.content_length());
                _request.stream(_stream);
                _response.stream(_stream);
                _response_writer.reset(_stream, _request);
                _response_writer.keep_alive(_keep_alive_possible);

                _accept_h(const_cast<basic_connection*>(this));

                if (!_keep_alive_possible)
                    break;
                if (!_response.keep_alive() && !_response.keep_alive())
                    break;

                /*
                 * Keep-Alive ֤ͭǤФꤹ.
                 *
                 * _response.keep_alive_time Ͻѥ᡼׵ᤫ
                 * Ŭڤͤ򤷤̤ʤΤǤͤФơ
                 * ͤǤкǽŪ Keep-Alive ֤ȤƻѤ.
                 */
                //keep_alive_time = _response.keep_alive_time();
                if (keep_alive_time == -1)
                    keep_alive_time = _init_args._keep_alive_time;
            }

        } catch (std::exception& e) {
            std::cerr << __FUNCTION__ << " exception:" << e.what() << std::endl;
        }

        _timer.stop();
        if (_keep_alive_possible) _keep_rel_h();

        socket_close();
        _disc_h(const_cast<basic_connection*>(this));
    }

private :

    bool wait_data_or_timeout(int sock, stream_type& stream, time_t sec)
    {
        time_t now = sec * 1000;
        for (;;) {
            if (now-- == 0)
                return false;

            if (!stream.eof())
                return true;

            fd_set readfds;
            //struct timeval timeout = { 0, 1000 };
            struct timeval timeout = { 1, 0 };

            FD_ZERO(&readfds);
            FD_SET(sock, &readfds);
            if (::select(FD_SETSIZE, &readfds, NULL, NULL, &timeout) < 0)
                return false;

            if (FD_ISSET(sock, &readfds))
                return true;
        }
    }

    /**
     * åȥѤδؿ.
     *
     * Keep-Alive ̵ꥯȤν塢ޤ
     * 㳰ȯ˼¹Ԥ.
     * ޤǡɤ߹ߤ䤽¾''˥ॢȤ
     * ¹ԤѤΥϥɥȤƤѤƤ.
     *
     * ºݤˤϥåȤΥǡɤ߹߰ʳϷ³Ƥޤ
     * åȤؤɤ߹ߤ񤭹ߤȯǲ餫Υ顼
     * ȯΤǤϤʤΤǡȤꤢåȤΥԤʤäƤ.
     */
    void socket_close()
    {
        if (_socket.alive())
            _socket.close();
    }

    socket_type _socket;
    stream_type _stream;
    request_type _request;
    response_type _response;
    response_writer_type _response_writer;

    boost::function<void (basic_connection*)> _accept_h;
    boost::function<void (basic_connection*)> _disc_h;
    boost::function<bool (void)> _keep_alive_h;
    boost::function<void (void)> _keep_rel_h;

    init_args_type _init_args;
    timer _timer;
    bool _keep_alive_possible;
};

} // namespace sl


#endif // SL_INET_BASIC_CONNECTION_HPP
