/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_ASIO_CONNECTION_HPP
#define SL_INET_ASIO_CONNECTION_HPP

#include <iostream>
#include <string>
#include <boost/asio.hpp>
#include <boost/function.hpp>

namespace sl {


template <class Service>
class asio_connection {
public :
    typedef typename Service::request_type            request_type;
    typedef typename Service::request_reader_type    request_reader_type;
    typedef typename Service::response_type            response_type;
    typedef typename Service::response_writer_type    response_writer_type;

    struct init_args_type {
        init_args_type() { }
    };

    typedef boost::function<void (asio_connection*)> handler_type;

    asio_connection()
        : _state(true)
    { }

    boost::asio::ip::tcp::iostream& ref_to_socket() { return _stream; }
    boost::asio::ip::tcp::iostream& stream() { return _stream; }
    request_type& request() { return _request; }
    response_type& response() { return _response; }

    response_writer_type writer()
    {
        return response_writer_type(_stream, _request);
    }


    void state(bool s)
    {
        boost::mutex::scoped_lock lock(_mutex);
        _state = s;
    }

    bool state()
    {
        boost::mutex::scoped_lock lock(_mutex);
        return _state;
    }

    void init(const init_args_type& args)
    { }

    void accept(const boost::function<void (asio_connection*)>& h)
    { _accept_handle = h; }

    void disconnect(const boost::function<void (asio_connection*)>& h)
    { _disconnect_handle = h; }

    void keep_alive(const boost::function<bool (void)>& h)
    { _keep_alive_h = h; }

    void keep_alive_rel(const boost::function<void (void)>& h)
    { _keep_rel_h = h; }


    void run()
    {
        try {
            for (;;) {
#if 1
                int native_socket = _stream.rdbuf()->native();
                if (!timed_wait(native_socket)) {
                    std::cerr << "!timed_wait" << std::endl;
                    break;
                }
#endif
                request_reader_type::parse(_stream, _request);
                if (_request.state() != request_type::normality)
                    break;

                _accept_handle(const_cast<asio_connection*>(this));

                if (!_request.keep_alive() || !_response.keep_alive())
                    break;

                if (!_stream.good() || !state())
                    break;
            }
            _stream.close();

        } catch (std::exception& e) {
            std::cerr << __FUNCTION__ << " exception:" << e.what() << std::endl;
        }
        _stream.clear();
        _disconnect_handle(const_cast<asio_connection*>(this));
    }

private :

    bool timed_wait(int sock)
    {
        fd_set readfds;
        struct timeval timeout = { 15, 0 };

        FD_ZERO(&readfds);
        FD_SET(sock, &readfds);
        if (select(FD_SETSIZE, &readfds, NULL, NULL, &timeout) <= 0)
            return false;

        if (FD_ISSET(sock, &readfds))
            return true;
        return false;
    }

    bool _state;

    boost::asio::ip::tcp::iostream    _stream;
    request_type    _request;
    response_type    _response;
    boost::function<void (asio_connection*)>        _accept_handle;
    boost::function<void (asio_connection*)>        _disconnect_handle;
    boost::function<bool ()> _keep_alive_h;
    boost::function<void ()> _keep_rel_h;
    boost::mutex _mutex;
};

} // sl


#endif // SL_INET_ASIO_CONNECTION_HPP
