/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_AJP13_REQUEST_HPP
#define SL_INET_AJP13_REQUEST_HPP

#include <ostream>
#include <string>
#include <boost/algorithm/string/predicate.hpp>

#include <sl/inet/ajp13/forward_request_packet.h>
#include <sl/inet/message.hpp>

namespace sl { namespace ajp13 {

class ajp13_request : public sl::http::message {
public :
    enum {
        empty,
        normality,
        request_too_long,
        bad_request,
        broken_connection
    };

    ajp13_request() : _state(empty) { }
    virtual ~ajp13_request() { }

    void state(int i) { _state = i; }
    int state() const { return _state; }

    void reset()
    {
        _state = empty;
        message::reset();
    }

    operator std::string () const
    {
        return this->text();
    }

    //////////////////////////////////////////////////////////////////////
    // Inherited pure virtual funtions.
    //

    virtual int type() const
    {
        return 0;
    }

    virtual std::string start_line() const
    {
        return request_line();
    }

    //////////////////////////////////////////////////////////////////////
    // HTTP-Request Implement funtions.
    //
    std::string request_line() const
    {
        return method() + " " + uri() + " " + version();
    }

    void method(const std::string& s) { _request_packet.method(s); }
    std::string method() const { return _request_packet.method(); }

    void uri(const std::string& s)
    {
/* XXXXX */
        _request_packet.req_uri(s);
    }

    std::string uri() const
    {
        return _request_packet.req_uri() + query();
    }

    void version(const std::string& s) { _request_packet.protocol(s); }
    std::string version() const { return _request_packet.protocol(); }

    void reg_name(const std::string& s) { _reg_name = s; }
    std::string reg_name() const { return _reg_name; }

    void scheme(const std::string& s) { _scheme = s; }
    std::string scheme() const { return _scheme; }

    void userinfo(const std::string& s) { _userinfo = s; }
    std::string userinfo() const { return _userinfo; }

    void host(const std::string& s) { _host = s; }
    std::string host() const { return _host; }

    void port(const std::string& s) { _port = s; }
    std::string port() const { return _port; }

    void abs_path(const std::string& s) { _request_packet.req_uri(s); }
    std::string abs_path() const { return _request_packet.req_uri(); }

    void query(const std::string& s) { _query = s; }
    std::string query() const
    {
        return _request_packet.attribute(attribute_code::QUERY_STRING);
    }

    bool keep_alive() const
    {
        return boost::equals(msg_header("Connection"),
                            "keep-alive", boost::is_iequal());
    }

    int content_length() const
    {
        std::string s = msg_header("Content-Length");
        if (s.empty())
            return -1;

        try {
            return boost::lexical_cast<int>(s);
        } catch (std::exception& e) {
            std::cerr << e.what() << std::endl;
        }
        return -1;
    }


    // ajp13 specific implement.
    sl::ajp13::forward_request_packet& request_packet()
    {
        return _request_packet;
    }

    void msg_header(const std::string& name, const std::string& value)
    {
        _request_packet.header(name, value);
    }

    void msg_header(const std::string& name, int value)
    {
        _request_packet.header(name, boost::lexical_cast<std::string>(value));
    }

    std::string msg_header(const std::string& name) const
    {
        return _request_packet.header(name);
    }

    std::string text() const
    {
        return _request_packet.text();
    }

private :
    int            _state;
    std::string _reg_name;
    std::string _scheme;
    std::string _userinfo;
    std::string _host;
    std::string _port;
    std::string _query;

    sl::ajp13::forward_request_packet    _request_packet;
};

inline std::ostream& operator<<(std::ostream& os, const ajp13_request& req)
{
    return os << static_cast<std::string>(req);
}

} } // namespace sl::ajp13


#endif // SL_INET_AJP13_REQUEST_HPP
