/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: forward_request_packet.cpp,v 1.1 2008/02/02 17:51:45 cvsuser Exp $
 */

#include <iostream>

#include <sl/net/ajp13/ajp13_error.h>
#include <sl/net/ajp13/function.h>
#include <sl/net/ajp13/forward_request_packet.h>
#include <sl/net/ajp13/types.h>
using namespace sl::ajp13;


forward_request_packet::forward_request_packet()
{
	clear();
}


Byte forward_request_packet::prefix_code() const
{
	return prefix_code::FORWARD_REQUEST;
}


forward_request_packet::operator ByteArray() const
{
	ByteArray	array;
	array += prefix_code();
	array += _method;
	array += _protocol;
	array += _req_uri;
	array += _remote_addr;
	array += _remote_host;
	array += _server_name;
	array += _server_port;
	array += _is_ssl;
	array += _nu_headers;

	if (_nu_headers > 0) {
		std::map<Integer, String>::const_iterator i = _headers.begin();
		for (; i != _headers.end(); i++) {
			array += i->first;
			array += i->second;
		}
		std::map<String, String>::const_iterator j = _other_headers.begin();
		for (; j != _other_headers.end(); j++) {
			array += j->first;
			array += j->second;
		}
	}

	std::map<Byte, String>::const_iterator i = _attributes.begin();
	for (; i != _attributes.end(); i++) {
		array += i->first;
		array += i->second;
	}

	std::map<String, String>::const_iterator j = _other_attributes.begin();
	for (; j != _other_attributes.end(); j++) {
		array += j->first;
		array += j->second;
	}

	array += static_cast<Byte>(0xff);
	return array;
}


packet &forward_request_packet::operator=(const ByteArray &array)
	throw(ajp13_error)
{
	const Byte *p = array;

	if (*p != prefix_code())
		throw ajp13_error("invalid prefix code");

	++p;	// skip prefix_code

	_method      = get_byte(p);    p += sizeof(Byte);
	_protocol    = get_string(p);  p += _protocol.size();
	_req_uri     = get_string(p);  p += _req_uri.size();
	_remote_addr = get_string(p);  p += _remote_addr.size();
	_remote_host = get_string(p);  p += _remote_host.size();
	_server_name = get_string(p);  p += _server_name.size();
	_server_port = get_integer(p); p += sizeof(Integer);
	_is_ssl      = get_boolean(p); p += sizeof(Boolean);
	_nu_headers = get_integer(p); p += sizeof(Integer);

	for (int i = 0; i < _nu_headers; i++)
		parse_headers(p);

	while (parse_attributes(p) > 0)
		;	/* nothing */

	return *this;
}

//
// member functions
//

// public

void forward_request_packet::method(const std::string& s)
	throw(ajp13_error)
{
	Byte	code;

	if ((code = method_code::to_code(s)))
		_method = code;
	else
		throw ajp13_error("unknown method");
}


std::string forward_request_packet::method() const
{
	return method_code::to_string(_method);
}


void forward_request_packet::protocol(const std::string& s)
{
	_protocol = s;
}


std::string forward_request_packet::protocol() const
{
	return _protocol;
}


void forward_request_packet::req_uri(const std::string& s)
{
	_req_uri = s;
}


std::string forward_request_packet::req_uri() const
{
	return _req_uri;
}


void forward_request_packet::header(const std::string& k, const std::string& s)
{
	Integer code;

	if ((code = req_header_code::to_code(k)) !=
	    req_header_code::UNKNOWN)
		_headers[code] = s;
	else
		_other_headers.insert(std::map<String, String>::value_type(k, s));

	++_nu_headers;
}

std::string forward_request_packet::header(const std::string& s) const
{
	Integer code;

	if ((code = req_header_code::to_code(s)) !=
	    req_header_code::UNKNOWN)
	{
		std::map<Integer, String>::const_iterator i = _headers.begin();
		for (; i != _headers.end(); i++)
			if (i->first == code)
				return i->second;
	} else {
		std::map<String, String>::const_iterator i = _other_headers.begin();
		for (; i != _other_headers.end(); i++)
			if (core::equals(i->first, s, core::is_iequal()))
				return i->second;
	}
	return "";
}

void
forward_request_packet::attribute(const std::string& k, const std::string& s)
{
	Integer code;

	if ((code = attribute_code::to_code(k)) != attribute_code::UNKNOWN)
		_attributes[code] = s;
	else
		_other_attributes.insert(
				std::map<String, String>::value_type(k, s));

	++_nu_headers;
}


// private

void forward_request_packet::parse_headers(const Byte * &p)
{
	Integer i = get_integer(p);

	if (req_header_code::find(i)) {
		p += sizeof(Integer);
		_headers[i] = get_string(p); p += _headers[i].size();

	} else {
		String  name  = get_string(p); p += name.size();
		String  value = get_string(p); p += value.size();
		_other_headers.insert(
						std::map<String, String>::value_type(name, value));
	}
}


bool forward_request_packet::parse_attributes(const Byte *&p)
{
	Byte b = get_byte(p);

	if (b == attribute_code::TERMINATOR)
		return false;

	if (attribute_code::find(b)) {
		p += sizeof(Byte);
		_attributes[b] = get_string(p); p += _attributes[b].size();

	} else {
		String name  = get_string(p); p += name.size();
		String value = get_string(p); p += value.size();
		_other_attributes.insert(
						std::map<String, String>::value_type(name, value));
	}

	return true;
}


Integer forward_request_packet::size() const
{
	Integer s = 0;

	s += sizeof(Byte);			//	Byte		_prefix_code;
	s += sizeof(Byte);			//	Byte		_method;
	s += _protocol.size();		//	String		_protocol;
	s += _req_uri.size();		//	String		_req_uri;
	s += _remote_addr.size();	//	String		_remote_addr;
	s += _remote_host.size();	//	String		_remote_host;
	s += _server_name.size();	//	String		_server_name;
	s += sizeof(Integer);		//	Integer		_server_port;
	s += sizeof(Boolean);		//	Boolean		_is_ssl;
	s += sizeof(Integer);		//	Integer		_nu_headers;

	// std::map<Integer, String>	_headers;
	std::map<Integer, String>::const_iterator i = _headers.begin();
	for (; i != _headers.end(); i++)
		s += sizeof(Integer) + i->second.size();

	// std::vector<String>		_other_headers;
	std::map<String, String>::const_iterator j = _other_headers.begin();
	for (; j != _other_headers.end(); j++)
		s += j->first.size() + j->second.size();

	// std::map<Byte, String>	_attributes;
	std::map<Byte, String>::const_iterator k = _attributes.begin();
	for (; k != _attributes.end(); k++)
		s += sizeof(Byte) + k->second.size();

	// std::vector<String>		_other_attributes;
	std::map<String, String>::const_iterator l = _other_attributes.begin();
	for (; l != _other_attributes.end(); l++)
		s += l->first.size() + l->second.size();

	// attribute_code::TERMINATOR 
	s += sizeof(Byte);			//	Byte

	return s;
}


void forward_request_packet::dump(std::ostream& os)
{
	os << "forward_request_packet" << std::endl;
	os << "prefix:[" << std::hex << prefix_code() << std::dec << "]" << std::endl;
	os << "method:[" << method_code::to_string(_method) << "]" << std::endl;
	os << "protocol:[" << _protocol << "]" << std::endl;
	os << "req_uri :[" << _req_uri << "]" << std::endl;
	os << "remote_addr:[" << _remote_addr << "]" << std::endl;
	os << "remote_host:[" << _remote_host << "]" << std::endl;
	os << "server_name:[" << _server_name << "]" << std::endl;
	os << "server_port:[" << _server_port << "]" << std::endl;
	os << "is_ssl:[" << _is_ssl << "]" << std::endl;
	os << "nu_headers:[" << _nu_headers << "]" << std::endl;

	std::map<Integer, String>::iterator i = _headers.begin();
	for (; i != _headers.end(); i++) {
		os << "header_name:["
		   << req_header_code::to_string(i->first) << "]" << std::endl;
		os << "header_value:[" << i->second << "]" << std::endl;
	}

	std::map<String, String>::iterator j = _other_headers.begin();
	for (; j != _other_headers.end(); j++)
		os << "header:[" << j->first << "][" << j->second << "]" << std::endl;

	std::map<Byte, String>::iterator k = _attributes.begin();
	for (; k != _attributes.end(); k++) {
		os << "attr_name:["
		   << attribute_code::to_string(k->first) << "]" << std::endl;
		os << "attr_value:[" << k->second << "]" << std::endl;
	}

	std::map<String, String>::iterator l = _other_attributes.begin();
	for (; l != _other_attributes.end(); l++)
		os << "attribute:[" << l->first << "][" << l->second << "]" << std::endl;

	os << "size:[" << this->size() << "]" << std::endl;
}


std::string forward_request_packet::text() const
{
	std::string		html;

	html.append(method_code::to_string(_method) + " ");
	html.append(_req_uri + " " + _protocol + "\n");

	std::map<Integer, String>::const_iterator i = _headers.begin();
	for (; i != _headers.end(); i++) {
		html.append(req_header_code::to_string(i->first) + ": ");
		html.append(i->second + "\n");
	}

	std::map<String, String>::const_iterator j = _other_headers.begin();
	for (; j != _other_headers.end(); j++)
		html.append(j->first + ": " + j->second + "\n");

	std::map<Byte, String>::const_iterator k = _attributes.begin();
	for (; k != _attributes.end(); k++) {
		html.append(attribute_code::to_string(k->first) + ": ");
		html.append(k->second + "\n");
	}

	j = _other_attributes.begin();
	for (; j != _other_attributes.end(); j++)
		html.append(j->first + ": " + j->second + "\n");

	return html;
}


void forward_request_packet::is_ssl(bool flag)
{
	_is_ssl = flag;
}


bool forward_request_packet::is_ssl() const
{
	return _is_ssl;
}


void forward_request_packet::remote_addr(const String& s)
{
	_remote_addr = s;
}

void forward_request_packet::remote_host(const String& s)
{
	_remote_host = s;
}

void forward_request_packet::server_name(const String& s)
{
	_server_name = s;
}

void forward_request_packet::server_port(Integer i)
{
	_server_port = i;
}


String forward_request_packet::remote_addr() const
{
	return _remote_addr;
}

String forward_request_packet::remote_host() const
{
	return _remote_host;
}

String forward_request_packet::server_name() const
{
	return _server_name;
}

Integer forward_request_packet::server_port() const
{
	return _server_port;
}


Integer forward_request_packet::content_length() const
{
	std::map<Integer, String>::const_iterator i = _headers.begin();
	for (; i != _headers.end(); i++)
		if (i->first == req_header_code::CONTENT_LENGTH)
			return atoi(i->second.data());
	return 0;
}


std::multimap<std::string, std::string> forward_request_packet::headers() const
{
	std::multimap<std::string, std::string> header_map;

	std::map<Integer, String>::const_iterator i = _headers.begin();
	for (; i != _headers.end(); i++) {
		header_map.insert(
					std::make_pair(req_header_code::to_string(i->first),
								   i->second));
	}

	std::map<String, String>::const_iterator j = _other_headers.begin();
	for (; j != _other_headers.end(); j++)
		header_map.insert(std::make_pair(j->first, j->second));

	return header_map;
}


std::map<std::string, std::string> forward_request_packet::attributes() const
{
	std::map<std::string, std::string> attributes_map;

	std::map<Byte, String>::const_iterator i = _attributes.begin();
	for (; i != _attributes.end(); i++) {
		attributes_map.insert(
					std::make_pair(attribute_code::to_string(i->first),
								   i->second));
	}

	std::map<String, String>::const_iterator j = _other_attributes.begin();
	for (; j != _other_attributes.end(); j++)
		attributes_map.insert(std::make_pair(j->first, j->second));

	return attributes_map;
}


void forward_request_packet::clear()
{
	_method = 0;
	_protocol.clear();
	_req_uri.clear();
	_remote_addr.clear();
	_remote_host.clear();
	_server_name.clear();
	_server_port = 0;
	_is_ssl = 0;
	_nu_headers = 0;

	_headers.clear();
	_other_headers.clear();

	_attributes.clear();
	_other_attributes.clear();
}
