/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_ACCEPTOR_HPP
#define SL_INET_ACCEPTOR_HPP

#if defined(__FreeBSD__)
#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <unistd.h>

#include <cctype>
#include <cerrno>
#include <csignal>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <string>

#include <sl/inet/basic_socket.hpp>

namespace sl {

/**
 * @class acceptor.
 * UnixƥòåȤΥץ󤫤饢Ԥ֤ޤǤԤޤ.
 *
 */
class acceptor {
public :
    typedef std::string init_args_type;

    /** 
     * 󥹥ȥ饯.
     * ˲⤷ޤ.
     */
    acceptor()
    {
    }

    /** 
     * ǥȥ饯.
     * дؿ destroy() ¹Ԥޤ.
     */
    virtual ~acceptor()
    {
        destroy();
    }

    /**
     * Unix ƥॳ socket/bind/listen ¹Ԥ
     * accept ǽ socket ޤ.
     *
     * @param port accept ݡֹ.
     */
    void init(const init_args_type& port)
    {
        if ((_socket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
            throw std::runtime_error("sl::acceptor::init :socket failed.");

        int ret;
        int opt = 1;
        ret = setsockopt(_socket, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
        if (ret != 0)
            throw std::runtime_error("sl::acceptor::init :setsockopt failed.");

        setsockopt(_socket, IPPROTO_TCP, TCP_NODELAY, &opt, sizeof(opt));
        if (ret != 0)
            throw std::runtime_error("sl::acceptor::init :setsockopt failed.");

        struct sockaddr_in  saddr;
        memset((char *)&saddr, 0x00, sizeof(saddr));
        saddr.sin_family = AF_INET;
        saddr.sin_addr.s_addr = INADDR_ANY;
        saddr.sin_port = htons(atoi(port.c_str()));

        if (::bind(_socket, (struct sockaddr *)&saddr, sizeof(saddr)) < 0)
            throw std::runtime_error("sl::acceptor::init :bind failed.");

#if defined(__linux__) || defined(__linux) || defined(linux)
        if(listen(_socket, SOMAXCONN) < 0)
#elif defined(__FreeBSD__)
        if(listen(_socket, -1) < 0)
#else
        if(listen(_socket, 128) < 0)
#endif
            throw std::runtime_error("sl::acceptor::init :listen failed.");
    }

    /**
     * ץ󤷤åȤ򥯥ޤ.
     */
    void destroy() throw()
    {
        ::close(_socket);
    }

    /**
     * Unix ƥॳ accept Ѥƥ饤Ȥ
     * Ԥޤ.
     * Ԥ֤򥭥󥻥뤹ˡȤƤϥʥˤ select 
     * Ԥ򤢤ƤˤƤޤ.
     *
     * @param s accept() åȤ¸.
     */
    void accept(basic_socket& s) // throw()
    {    
        struct sockaddr_in  addr;
        memset((char *)&addr, 0x00, sizeof(addr));
        socklen_t addr_len = sizeof(addr);

        for (;;) {
#if 1
            fd_set  fds;
            FD_ZERO(&fds);
            FD_SET(_socket, &fds);

            struct timeval t = { 1, 0 };

            int ret = ::select(FD_SETSIZE, &fds, NULL, NULL, &t);
            if (ret == 0)
                continue;
            else if (ret < 0)
                throw std::runtime_error("sl::acceptor::accept :select failed.");
            if (!FD_ISSET(_socket, &fds)) {
                continue;
            }
#else
            int kq;
            struct kevent kev;

            kq = kqueue();
            if ( kq == -1 ){
                perror("kqueue");
            }
            EV_SET(&kev, _socket, EVFILT_READ, EV_ADD, 0, 0, NULL);
            int ret = kevent(kq, &kev, 1, NULL, 0, NULL);
            if ( ret == -1 ){
                perror("kevent");
            }
            struct timespec    t = { 5, 0 };
            int n = kevent(kq, NULL, 0, &kev, 1, &t);
            if ( n == -1 ){
                perror("kevent");
            }
#endif
            s._socket = ::accept(_socket, (struct sockaddr *)&addr, &addr_len);
            if (s._socket != -1)
                return;

            switch (errno) {
            case EINTR :
            case EMFILE :
            case ENFILE :
            case ECONNABORTED :
                break;
            default :
                throw std::runtime_error("sl::acceptor::accept :accept failed.");
            }
        }
    }

private :
    int _socket;
};


} // namespace sl

#endif // SL_INET_ACCEPTOR_HPP
