/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_CLASS_LOADER_LOADER_HPP
#define SL_CLASS_LOADER_LOADER_HPP

#include <stdexcept>
#include <sl/config.hpp>
#include <sl/object.hpp>
#include <sl/class_loader/dl_impl.hpp>
#include <sl/class_loader/register.hpp>

namespace sl {

/**
 * @class class_loader.
 * ɥ֥(*.so/ *.dll)饯饹Υɤ
 * 󥹥󥹤¸륯饹Ǥ.
 *
 * ɤ뤵¦Υ饹˴ñʥեåŹߡ
 * 롼ˤä͡ॹڡХɤ¦ߤ̤
 * 饹ɤ뤳Ȥǽˤʤޤ.
 *
 * եåȤ sl/class_loader/register.hpp ˴ޤޤޥǤ
 * SL_REGIST_LOADABLE_CLASS(X) ɤ륯饹뤳Ȥǡ
 *  SL_REGIST_LOADABLE_CLASS ΰˤϥ͡ॹڡޤ९饹̾ꤷޤ.
 *
 * ޤɤ¦Υ롼Ȥơ
 * ֥͡ॹڡȥǥ쥯ȥѥϰפɬפפȤȤǤ.
 * Ĥޤ͡ॹڡޤ९饹̾ path::to::target ξ
 * ɤ¦Υץ¹ԥǥ쥯ȥ(Τˤϥȥǥ쥯ȥ)
 * 鸫 ./path/to/target.so ֤Ƥɬפޤ.
 * 
 *
 * ץ:
 *
 * ɤ¦Υ饹<br />
 * <pre>
 * base.hpp
 *    namespace path1 {
 *      class base {
 *      pubilc :
 *        virtual void something() { ... }           // ¿֤ɽؿ.
 *      };
 *    }
 *
 * target.hpp
 *    \#include "base.hpp"
 *    namespace path2 {
 *      class target : public base {                 // path1::base 饹.
 *      virtual void something() { ... }             // ¿֤ɽؿ򥪡С饤.
 *        :
 *        :
 *      };
 *    }
 *    SL_REGIST_LOADABLE_CLASS(path2::target);       // ѥޥ򵭽Ҥ
 * </pre>
 *
 *
 * ɤ¦Υɼ
 * <pre>
 *    \#include "base.hpp"                           // "base" Ƥ
 *    path1::base* p;
 *
 *    try {
 *      sl::class_loader loader("./path2/target.so");// path2::target ̾ǻ.
 *      p = loader.instance<path1::base>();          // path2::target 쥯饹Υݥ󥿤ݻ.
 *      p->something();                              // path2::target::something() ¹Ԥ.
 *    } catch (std::runtime_error& e) {
 *        :
 *    }
 * </pre>
 */
class class_loader {
public :

    /**
     * 󥹥ȥ饯.
     *
     * ⥸塼(*.so,*.dll) ؤΥѥ dso_name ޤɤ߹.
     * Υѥȥ⥸塼Υ͡ॹڡ()Ʊ뤫
     * ʤ2ǽˤƤ.
     *
     * @param dso_name ⥸塼ؤΥѥ.
     * @param path_eq_scope ѥȥ⥸塼Υ͡ॹڡƱե饰
     */
    class_loader(const std::string& dso_name,
                 bool path_eq_scope = true) throw(std::runtime_error)
        : _handle(0)
    {
        load(dso_name, path_eq_scope);
    }

    /**
     * @param rootdir ⥸塼븡Υ١ǥ쥯ȥ.
     * @param dso_name ⥸塼ؤΥѥ.
     * @param path_eq_scope ѥȥ⥸塼Υ͡ॹڡƱե饰
     */
    class_loader(const std::string& rootdir,
                 const std::string& dso_name,
                 bool path_eq_scope = true) throw(std::runtime_error)
        : _rootdir(rootdir + "/"), _handle(0)
    {
        load(dso_name, path_eq_scope);
    }


    /**
     * ǥȥ饯.
     *
     * ݻƤ⥸塼뤬ϥɤޤ.
     * ¼ŪˤϤΥ饹Ѥ桼¦ǤΥ⥸塼Υɤ
     * ԤʤƤϤޤ.
     */
    ~class_loader()
    {
        if (_handle)
            unload();
    }

    /**
     * ⥸塼ɤ߹߽.
     *
     * Υѥ̾ºݤΥǥ쥯ȥ+ե̾
     * ե dl_impl 饹ͳǥץ󤹤.
     *
     * Ǥ Unix Υǥ쥯ȥѥ "/xxx/yyy/zzz.so" 
     * Java ǤΥ׻Τ褦 "xxx.yyy.zzz "ƤƤ.
     *
     * @param dso_name ⥸塼ؤΥѥ.
     * @param path_eq_scope ѥȥ⥸塼Υ͡ॹڡƱե饰
     */
    void load(const std::string& dso_name, bool path_eq_scope = true)
        throw(std::runtime_error)
    {
        if (_handle)
            throw std::runtime_error("class loading failed. illegal state");

        if (dso_name.empty())
            throw std::invalid_argument("class path is empty");

        _dso_name = dso_name;

        // ʥѥ
        if (_dso_name.find('/') == std::string::npos) {
            std::string::size_type pos = 0;
            while ((pos = _dso_name.find('.', pos)) != std::string::npos)
                _dso_name.replace(pos, 1, "/");
            _dso_name.insert(0, "./");
        }

        if (_dso_name.find_last_of(sl::config::dso_suffix) + 1 != _dso_name.length())
            _dso_name.append(sl::config::dso_suffix);

        // ѥ͡ॹڡޤ९饹̾
        _class_name = _dso_name;
        if (_class_name[0] == '.' && _class_name[1] == '/')
            _class_name.erase(0, 2);

        std::string::size_type pos = 0;
        while ((pos = _class_name.find('/', pos)) != std::string::npos)
            _class_name.replace(pos, 1, "::");

        _class_name.erase(_class_name.length()-sl::config::dso_suffix.length());

        _handle = detail::dl_impl::open(_rootdir + _dso_name);
        if (!_handle)
            throw std::runtime_error("class loading failed. open error: "+ _dso_name);

        if (!path_eq_scope) {
            std::string::size_type pos = _class_name.rfind("::");
            if (pos != std::string::npos)
                _class_name.erase(0, pos);
        }
    }

    /**
     * ⥸塼Υɽ.
     *
     * åѥ饹Ǥ dl_impl::close ¹ԤԤʤ. 
     */
    void unload()
    {
        if (!_handle)
            throw std::runtime_error("class_loader::unload failed. illegal state.");

        detail::dl_impl::close(_handle);
        _handle = 0;
    }

    /**
     * 󥹥.
     *
     * ⥸塼ꤵƤѤΥܥɤ߹
     * δؿ¹Ԥˤäƥ󥹥󥹤.
     *
     * @param T* 󥹥󥹤ؤΥݥ.
     *
     */
    template <typename T>
    T* instance() throw (std::runtime_error)
    {
        if (!_handle)
            throw std::runtime_error("class_loader::instance failed. illegal state.");

        try {
            /*
             * mangle ܥ(Ǥϴؿ)
             * ǥޥ󥰥뤹ؿΰ()η T Ѥ
             * load ¹Ի ⥸塼̾ƻѤ.
             *
             * module* sl_class_loader_get_new_class(module*);
             */
            std::string ep = _class_name + "* " +
                             "sl_class_loader_get_new_class" +
                             "(" + _class_name + "*)";

            typedef T* (Creater)(T*);
            Creater* creater = (Creater*)detail::dl_impl::symbol(_handle, ep, false);
            if (!creater) {
                const char *e = dlerror();
                throw std::runtime_error(e);
            }

            T* ptr = (creater)(0);
            if (!ptr) {
                throw std::runtime_error("class_loader::instance faild "
                            "in sl_class_loader_get_new_class function.");
            }
            return ptr;

        } catch (std::exception& e) {
            throw std::runtime_error(
                std::string("class_loader::instance failed. ") + e.what());
        }
    }

    /**
     * instance() Υå.
     * ɬפ?...
     *
     * @return instance() ݥݻ sl::object 󥹥
     */
    template <typename T>
    sl::object get() throw(std::runtime_error)
    {
        T* p = instance<T>();
        return p ? sl::object(p) : sl::object();
    }

    /**
     * ּ.
     *
     * true ϸߥ⥸塼Ǥ false ɼ¹
     * ɻξ.
     * ǥХåѤʤΤ.
     *
     * @param ߤξ
     */
    bool is_loaded() const
    {
        return _handle ? true : false;
    }

private :
    class_loader() : _handle(0) { }
    class_loader(const class_loader&);

    std::string _rootdir;
    std::string _dso_name;
    std::string _class_name;
    void* _handle;
};

} // namespace sl

#endif // SL_CLASS_LOADER_LOADER_HPP
