/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <stdexcept>
#include <string>

#include <sl/inet/http_response.hpp>
#include <sl/inet/http_response_writer.hpp>
#include <sl/inet/http_util.hpp>
using namespace sl::http;


http_response_writer::http_response_writer()
    : _stream(0), _request(0)
{ }


http_response_writer::http_response_writer(std::ostream& stream,
                                           http_request& request)
{ reset(stream, request); }


void http_response_writer::reset(std::ostream& stream, http_request& request)
{
	_stream   = &stream;
	_request  = &request;
	_commited = false;
	_chunked  = false;
	_http_ver10 = false;
	_http_ver10_body.clear();
}


void http_response_writer::flush_all(http_response& res)
{
	if (!_stream || !_request)
		throw std::runtime_error("http_response_writer error.");

	if (_commited == true)
		return;

	try {
		flush_headers(res);
		flush_content(res);
		commit(res);
	} catch (std::exception& e) {
		throw std::runtime_error("ERROR: flush_all");
	}
}

void http_response_writer::flush_headers(http_response& res)
{
	if (!_stream || !_request)
		throw std::runtime_error("http_response_writer error.");
	if (_commited == true)
		return;

	if (_request->version() == "HTTP/1.1") {
		res.version("HTTP/1.1");

		if (_request->keep_alive() && _keep_alive_possible)
			res.msg_header("Connection", "Keep-Alive");
		else
			res.msg_header("Connection", "Close");

		if (res.msg_header("Content-Length").empty()) {
			res.msg_header("Transfer-Encoding", "chunked");
			_chunked = true;
		}
		*_stream << res.text_header();
	} else {

		/*
		 * ȤꤢŪ HTTP/1.0 .
		 * ޤ Connection إåˤ
		 */
		res.version("HTTP/1.0");
		if (_request->keep_alive() && _keep_alive_possible)
			res.msg_header("Connection", "keep-alive");

		/*
		 * Content-Length иۤɽϤ Message-Body 
		 * ϳꤷƤϤʤΤǥإåϤ.
		 * 
		 */
		if (!res.msg_header("Content-Length").empty()) {
			*_stream << res.text_header();
			return;
		}
		/*
		 * դ Content-Length ¸ߤʤ Message-Body 
		 * ꤹޤԤɬפ뤬
		 * Connection: keep-alive ʳ׵ᤢ³Ǥ
		 * Message-Body νλȤƻѽ뤿ᡢ
		 * Connection: close ξϥإåϤ.
		 */
		else if (!_request->keep_alive() || !_keep_alive_possible) {
			res.msg_header("Connection", "close");
			*_stream << res.text_header();
			return;
		}

		/*
		 * 嵭Ȥۤʤ Message-Body Хåե󥰤
		 * commit ׵ Content-Length إåơ
		 * ¾Υإå Message-Body λ˽Ϥ.
	 	 */
		else
			_http_ver10 = true;
	}
}

void http_response_writer::flush_content(http_response& res)
{
	if (!_stream || !_request)
		throw std::runtime_error("http_response_writer error.");
	if (_commited == true)
		return;
	if (res.msg_body().empty())
		return;
	if (_http_ver10)
		_http_ver10_body.append(res.msg_body());

	else if (_chunked)
		*_stream << std::hex << res.msg_body().length() << "\r\n"
				 << std::dec << res.msg_body() << "\r\n";
	else
		*_stream << res.msg_body();
}

void http_response_writer::commit(http_response& res)
{
	if (!_stream || !_request)
		throw std::runtime_error("http_response_writer error.");

	if (_commited == true)
		return;

	if (_chunked)
		*_stream << "\r\n0\r\n\r\n";
	else if (_http_ver10) {
		res.msg_header("Content-Length",
			boost::lexical_cast<std::string>(_http_ver10_body.length()));
		*_stream << res.text_header();
		*_stream << _http_ver10_body;
	}
	_stream->flush();
	_commited = true;
}

void http_response_writer::keep_alive(bool flag)
{
    _keep_alive_possible = flag;
}
