/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <string>
#include <stdexcept>
#include <boost/algorithm/string/trim.hpp>

#include <sl/inet/http_request_reader.hpp>
#include <sl/inet/http_request.hpp>
#include <sl/inet/http_util.hpp>
#include <sl/inet/http_util/request_line_p.hpp>
#include <sl/inet/uri_parser.hpp>
using namespace sl::http;


bool http_request_reader::is_empty_line(const std::string& line)
{
    return (line.empty() || line[0] == '\n' || (line[0] == '\r' || line[1] == '\n'));
}

/* static */
void http_request_reader::parse(std::istream& stream, http_request& req)
{
    try {
        std::string line;
        int         line_count = 0;
        char ch;
        for (;;) {
            /** ϸǸľ */
            if (!stream.good()) {
                req.state(http_request::request_too_long);
                return;
            }

            ch = stream.get();
            if (ch != '\n') {
                line.append(1, ch);
                continue;
            }

            if (req.state() != http_request::empty && is_empty_line(line))
                break;

            if (line_count == 0)
                start_line(line, req);
            else
                header_line(line, req);

            ++line_count;
            line.clear();
        }

        if (req.is_post_with_wwwform()) {
            size_t l = req.content_length();
            char *b = new char[l + 1];
            b[l] = '\0';
            stream.read(b, l);
            req.msg_body(b);
#if 0
std::cerr << "body" << std::endl << b << std::endl;
std::cerr << "body.length:" << req.msg_body().length() << std::endl;
#endif
            req.parameter(parameter_split(req.msg_body()));
            delete b;
        } else if (req.is_get_with_query())
            req.parameter(parameter_split(req.query()));

    } catch (std::overflow_error& e) {
        req.state(http_request::request_too_long);
        std::cerr << e.what() << std::endl;
        return;
    } catch (std::exception& e) {
        req.state(http_request::bad_request);
        std::cerr << e.what() << std::endl;
        return;
    }
}

/* static */
void http_request_reader::start_line(const std::string& line, http_request& req)
{
    std::string method;
    std::string uri;
    std::string version;
    std::string reg_name;
    std::string scheme;
    std::string userinfo;
    std::string host;
    std::string port;
    std::string abs_path;
    std::string query;

    boost::spirit::parse_info<> info;

#if 0
    /*
     * Method SP Request-URI SP HTTP-Version ¤ÎÊ¬²ò¼èÆÀ.
     * C ¥é¥¤¥Ö¥é¥ê¤Î¤ß¤ò»ÈÍÑ¤·¤¿ Request-Line ¤Î²òÀÏ½èÍý.
     */
    const char* l = line.c_str();
    const char* pos1;
    const char* pos2;
    const char* pos3;
    if ((pos1 = strchr(l, ' ')) == 0)
        throw std::runtime_error("Invalid Request-Line 1");

    if ((pos2 = strchr(pos1 + 1, ' ')) == 0)
        throw std::runtime_error("Invalid Request-Line 2");

    if (pos1 +1 == pos2)
        throw std::runtime_error("Invalid Request-Line 3");

    pos3 = strchr(pos2 + 1, '\r');
    if (pos3 == 0)
             pos3 = l + strlen(l);

    method.assign(l, pos1);
    uri.assign(pos1 + 1, pos2 - pos1 - 1);
    version.assign(pos2 + 1, pos3);

    if (!verification_Method(method))
        throw std::runtime_error("Invalid Method");

    if (!verification_HTTP_Version(version))
        throw std::runtime_error("Invalid HTTP-Version");

    uri_parser uri_p(reg_name,
                      scheme,
                      userinfo,
                      host,
                      port,
                      abs_path,
                      query);
    static boost::mutex m;
        {
    boost::mutex::scoped_lock l(m);
    info = boost::spirit::parse(uri.c_str(), uri_p);
    if (!info.full)
        throw std::runtime_error("uri_parser parse error.");
        }

    req.method(method);
    req.uri(uri);
    req.version(version);
    req.reg_name(reg_name);
    req.scheme(scheme);
    req.userinfo(userinfo);
    req.host(host);
    req.port(port);
    req.abs_path(abs_path);
    req.query(query);
#endif

    sl::http::request_line_p::req_type t;
    sl::http::request_line_p::parse(line, t);

    req.method(t._method);
    req.uri(t._uri);
    req.version(t._version);
    req.reg_name(t._reg_name);
    req.scheme(t._scheme);
    req.abs_path(t._abs_path);
    req.query(t._query);

#if 0
std::cerr << "method :[" << req.method()  << "]" << std::endl;
std::cerr << "uri    :[" << req.uri()     << "]" << std::endl;
std::cerr << "version:[" << req.version() << "]" << std::endl;
std::cerr << "reg_:" << req.reg_name() << std::endl;
std::cerr << "sche:" << req.scheme()   << std::endl;
std::cerr << "user:" << req.userinfo() << std::endl;
std::cerr << "host:" << req.host()  << std::endl;
std::cerr << "port:" << req.port()  << std::endl;
std::cerr << "abs_:" << req.abs_path()  << std::endl;
std::cerr << "quer:" << req.query() << std::endl;
#endif

    req.state(http_request::normality);
}

/* static */
void http_request_reader::header_line(const std::string& line, http_request& req)
{
    std::string::size_type pos = line.find(':');
    if (pos != std::string::npos) {
        std::string name  = line.substr(0, pos);
        std::string value = line.substr(pos + 1);
        boost::trim_left(name);
        boost::trim(value);
        req.msg_header(name, value);

    } else if (line[0] == ' ' || line[0] == '\t')
        req.append_last_header_value(line);
    else
        throw std::runtime_error("invalid header.");
}
