﻿/*
 * マルチプラットフォーム描画エンジン「Sherry」
 * Copyright(C) 2010-2011 SherryProject. all rights reserved.
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 *	shShader.c
 *		Copyright (C) 2010-2011 Cap5ule. all rights reserved.
 *
 *	Date		|Version	|Author		|Summary
 *	2010/11/16	|v0.01		|Cap5ule	|初回版
 */


/*---- インクルードファイル ----*/

#include "sherry.h"			//!<共通ヘッダー
#include "shApplication.h"	//!<必要なSHヘッダー
#include "shMath.h"
#include "shTexture.h"
#include "shShader.h"		//!<ヘッダー

/*---- マクロ定義 ----*/


/*---- スタティックプロトタイプ宣言 ----*/
// 説明は実装部に記載


/*---- グローバル変数 ----*/


/*---- スタティック関数 ----*/


/*---- 関数 ----*/

/*
 * int shCreateShaderFromFile(shShaderProg* spg,
 *							const char* vspath, const char* fspath,
 *							shVertexElements* ve);
 *
 *	＜引数＞
 *　　shShaderProg* spg		: シェーダプログラム変数のポインタ
 *　　char* vspath			: 頂点シェーダファイルパス
 *　　char* fspath			: フラグメントシェーダファイルパス
 *　　shVertexElements* ve	: 頂点要素配列
 *	＜戻り値＞
 *　　int					: エラーコードが返ります
 *	＜説明＞
 *　　シェーダソースからシェーダプログラムを生成します。
 *　　頂点シェーダとフラグメントシェーダのソースコードをコンパイルし、
 *　　シェーダプログラムを生成します。[現在OpenGLのみ対応]
 */
int shCreateShaderFromFile(shShaderProg* spg,
							const char* vspath, const char* fspath,
							shVertexElements* ve)
{
	//ローカル変数
	char* buffer_vs;	// バッファ
	char* buffer_fs;
	int fileSize;		// 確保時のファイルサイズ
	FILE* fp;			// ファイル構造体ポインタ

	// シェーダソースをファイルからメモリへ読み込む
	// 頂点シェーダを読み込む
	if((fp = fopen(vspath, "rt")) == 0)
		return 1;	/*ファイルオープンエラー*/
	// ファイルサイズを取得
	fseek(fp, 0L, SEEK_END);
	fileSize = ftell( fp );
	fseek(fp, 0L, SEEK_SET);
	// メモリ確保
	buffer_vs = (char*)malloc(fileSize);
	// 読み込み、閉じる
	fread(buffer_vs,fileSize,1,fp);
	fclose(fp);

	// フラグメントシェーダを読み込む
	if((fp = fopen(fspath, "rt")) == 0)
		return 1;	/*ファイルオープンエラー*/
	// ファイルサイズを取得
	fseek(fp, 0L, SEEK_END);
	fileSize = ftell( fp );
	fseek(fp, 0L, SEEK_SET);
	// メモリ確保
	buffer_fs = (char*)malloc(fileSize);
	// 読み込み、閉じる
	fread(buffer_fs,fileSize,1,fp);
	fclose(fp);

	// シェーダプログラムを作成
	if(shCreateShader(spg, buffer_vs, buffer_fs, ve) != 0)
		return 1;

	// 確保したメモリの解放
	free(buffer_vs);
	free(buffer_fs);

	return 0;
}

/*
 * int shCreateShader(shShaderProg* spg,
 *						const char* vsbuf, const char* fsbuf,
 *						shVertexElements* ve);
 *
 *	＜引数＞
 *　　shShaderProg* spg		: シェーダプログラム変数のポインタ
 *　　char* vsbuf			: 頂点シェーダバッファ
 *　　char* fsbuf			: フラグメントシェーダバッファ
 *　　shVertexElements* ve	: 頂点要素配列
 *	＜戻り値＞
 *　　int					: エラーコードが返ります
 *	＜説明＞
 *　　シェーダソースからシェーダプログラムを生成します。
 *	　頂点シェーダとフラグメントシェーダのバッファをコンパイルし、
 *	　シェーダプログラムを生成します。[現在OpenGLのみ対応]
 */
int shCreateShader(shShaderProg* spg,
					const char* vsbuf, const char* fsbuf,
					shVertexElements* ve)
{
	//ローカル変数
	GLuint vtxShader;	//シェーダ
	GLuint frgShader;
	shVertexElements* pve;	//頂点配列要素読み込み用
	int ve_count = 0;		//VEのカウント用

	// シェーダオブジェクトの作成
	vtxShader = glCreateShader(GL_VERTEX_SHADER);
	frgShader = glCreateShader(GL_FRAGMENT_SHADER);

	// バーテックスシェーダプログラムの読み込み
	glShaderSource(vtxShader, 1, (const GLchar **)&vsbuf, NULL);
	// シェーダのコンパイル
	glCompileShader(vtxShader);

	// フラグメントシェーダプログラムの読み込み
	glShaderSource(frgShader, 1, (const GLchar**)&fsbuf, NULL);
	// シェーダのコンパイル
	glCompileShader(frgShader);

	// シェーダプログラムの作成
	spg->shaderProg_gl = glCreateProgram();

	// シェーダオブジェクトのシェーダプログラムへの登録
	glAttachShader(spg->shaderProg_gl, vtxShader);
	glAttachShader(spg->shaderProg_gl, frgShader);

	glUseProgram(spg->shaderProg_gl);
	
	// 頂点属性のバインド
	pve = ve;	//読み込み用
	while(0xFF != pve->stream) {
		glBindAttribLocation(spg->shaderProg_gl, ve_count, &pve->valuename[0]);
		pve++;		//次の要素へ
		ve_count++;
	}
	// シェーダオブジェクトの削除
	glDeleteShader(vtxShader);	// アタッチしたらいらないため
	glDeleteShader(frgShader);

	// シェーダプログラムのリンク
	glLinkProgram(spg->shaderProg_gl);

	return 0;
}

/*
 *	int shDeleteShader(shShaderProg* spg)
 *
 *	＜引数＞
 *　　shShaderProg* spg	: シェーダプログラム変数のポインタ
 *	＜戻り値＞
 *　　int					: エラーコードが返ります
 *	＜説明＞
 *　　要らなくなったシェーダプログラムを削除します。
 */
int shDeleteShader(shShaderProg* spg)
{
	if(g_sh_DrawMode == OpenGL) {
		glDeleteProgram(spg->shaderProg_gl);
	} else {
		/*使用しない*/
	}

	return 0;
}

/*
 *	int shBeginShader(shShaderProg* spg)
 *
 *	＜引数＞
 *		shShaderProg* spg	: シェーダプログラム変数のポインタ
 *	＜戻り値＞
 *		int					: エラーコードが返ります
 *	＜説明＞
 *		シェーダプログラムの適用を開始します。
 *		以降に呼び出される描画コマンドは、すべて適用されます。
 */
int shBeginShader(shShaderProg* spg)
{
	if(g_sh_DrawMode == OpenGL) {
		glUseProgram(spg->shaderProg_gl);
	} else {
		/*使用しない*/
	}
	return 0;
}

/*
 *	int shEndShader(shShaderProg* spg)
 *
 *	＜引数＞
 *		shShaderProg* spg	: シェーダプログラム変数のポインタ
 *	＜戻り値＞
 *		int					: エラーコードが返ります
 *	＜説明＞
 *		シェーダプログラムの適用を解除します。
 */
int shEndShader(shShaderProg* spg)
{
	if(g_sh_DrawMode == OpenGL) {
		glUseProgram(0);
	} else {
		/*使用しない*/
	}
	return 0;
}

/*
 *	int shUseVertexAttr(shShaderProg* spg,
 *					int index, int size, const void* pointer)
 *
 *	＜引数＞
 *		shShaderProg* spg	: シェーダプログラム変数のポインタ
 *		int index			: 要素番号
 *		int size			: 要素数１～４
 *		char* name			: 頂点情報名前
 *	＜戻り値＞
 *		int					: エラーコードが返ります
 *	＜説明＞
 *		頂点情報と要素番号をバインドする
 *		情報をシェーダへ送る際の関連付けをする
 */
int shUseVertexAttr(shShaderProg* spg,
					int index, int size, const void* pointer)
{
	if(g_sh_DrawMode == OpenGL) {
		glVertexAttribPointer(index, size, GL_FLOAT, GL_FALSE, 0, pointer);
		glEnableVertexAttribArray(index);	// 頂点の性質を使用可能にする
	} else {
		/*使用しない*/
	}
	
	return 0;
}

/*
 *	void shSetTexture(shTexture* texture)
 *
 *	＜引数＞
 *		shTexture* texture	: テクスチャ構造体
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		テクスチャを設定する。
 */
void shSetTexture(shTexture* texture)
{
	if(g_sh_DrawMode == OpenGL) {
		//テクスチャをバインドする
		glBindTexture(GL_TEXTURE_2D, texture->gltex);
	} else {
		/*使用しない*/
	}
}

/*
 *	void shUniformMatrix(shShaderProg* spg, const char* param
 *						 shMatrix* matrix)
 *
 *	＜引数＞
 *　　shShaderProg* spg	: シェーダプログラム変数のポインタ
 *　　char* param		: シェーダ内パラメータ名
 *　　shMatrix* matrix	: 行列ポインタ
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		Matrix型の変数を設定する。
 */
void shUniformMatrix(shShaderProg* spg, const char* param,
					 shMatrix* value)
{
	// 行列をGPUにセットする
	if(g_sh_DrawMode == OpenGL) {
		glUniformMatrix4fv(glGetUniformLocation(spg->shaderProg_gl, param),
			1, GL_FALSE, (float*)value->m);
	} else {
		/*使用しない*/
	}
}

/*
 *	void shUniformFloat(shShaderProg* spg, const char* param
 *						 float value)
 *
 *	＜引数＞
 *　　shShaderProg* spg	: シェーダプログラム変数のポインタ
 *　　char* param		: シェーダ内パラメータ名
 *　　float value		: 渡す値
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		Float型の変数を設定する。
 */
void shUniformFloat(shShaderProg* spg, const char* param,
					 float value)
{
	//FloatをGPUにセットする
	if(g_sh_DrawMode == OpenGL) {
		glUniform1fv(glGetUniformLocation(spg->shaderProg_gl, param), 1, &value);
	} else {
		/*使用しない*/
	}
}

/*
 *	void shUniformVector2(shShaderProg* spg, const char* param
 *						  shVector2* value)
 *
 *	＜引数＞
 *　　shShaderProg* spg	: シェーダプログラム変数のポインタ
 *　　char* param		: シェーダ内パラメータ名
 *　　shVector2* value	: ベクトル２ポインタ
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		Vector2型の変数を設定する。
 */
void shUniformVector2(shShaderProg* spg, const char* param,
					  shVector2* value)
{
	//FloatをGPUにセットする
	if(g_sh_DrawMode == OpenGL) {
		glUniform2fv(glGetUniformLocation(spg->shaderProg_gl, param), 1, (float*)value->v);
	} else {
		/*使用しない*/
	}
}

/*
 *	void shUniformVector3(shShaderProg* spg, const char* param
 *						  shVector3* value)
 *
 *	＜引数＞
 *　　shShaderProg* spg	: シェーダプログラム変数のポインタ
 *　　char* param		: シェーダ内パラメータ名
 *　　shVector3* value	: ベクトル３ポインタ
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		Vector3型の変数を設定する。
 */
void shUniformVector3(shShaderProg* spg, const char* param,
					  shVector3* value)
{
	//FloatをGPUにセットする
	if(g_sh_DrawMode == OpenGL) {
		glUniform3fv(glGetUniformLocation(spg->shaderProg_gl, param), 1, (float*)value->v);
	} else {
		/*使用しない*/
	}
}
