﻿/*
 * マルチプラットフォーム描画エンジン「Sherry」
 * Copyright(C) 2010-2011 SherryProject. all rights reserved.
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 *	shDebug.c
 *		Copyright (C) 2010-2011 Cap5ule. all rights reserved.
 *
 *	Date		|Version	|Author		|Summary
 *	2010/11/16	|v0.01		|Cap5ule	|初回版
 *	2010/12/04	|v0.02		|onakah		|デバッグウィンドウ構造体定義、
 *										|初期化、削除、生成、描画の追加
 *	2010/12/21	|v0.03		|onakah		|FPSの取得関数を追加
 *	2010/12/23	|v0.04		|onakah		|デバッグGUIのリソースを埋め込み
 *	2010/12/24	|v0.05		|onakah		|ASCII文字の1文字描画を追加
 *	2010/12/25	|v0.06		|onakah		|文字列の描画を追加
 */

/*---- インクルードファイル ----*/

#include "sherry.h"				//!<共通ヘッダー
#include "shApplication.h"		//!<必要なSHヘッダー
#include "shSprite.h"
#include "shDebug.h"			//!<ヘッダー
#include "shDebugResource.h"	//!< 埋め込みテクスチャデータ

/*---- マクロ定義 ----*/


/*---- 定数定義 ----*/
#define TEXSIZE				(32)
#define DEFAULT_FONTSIZE	(32)


/*---- スタティックプロトタイプ宣言 ----*/
// 説明は実装部に記載


// 整数値を文字列へ変換する。
char* uint_to_str(char *buf, unsigned src, int base);

/*---- スタティック関数 ----*/


/*---- グローバル変数 ----*/

shTexture		g_debugTex[9];				//!<テクスチャデータ
shTexture		g_debugTex[9];				//!< テクスチャデータ
shTexture		g_asciiTex;					//!< ASCIIフォントテクスチャ
short			g_windowFlag[SH_DW_MAX];	//!< ウィンドウが存在するかのフラグ
shDWList*		g_pListIndex;				//!< ウィンドウリストの先頭ポインタ
int				g_fontSize;					//!< デバッグフォントサイズ
float			g_fontScale;				//!< デバッグフォントのデフォルトサイズからの拡縮倍率

/*---- 関数 ----*/


/*
 *	int shInitDebug(void)
 *
 *	＜戻り値＞
 *	　int			: エラーコードが返ります。
 *	＜説明＞
 *　　デバッグシステムの初期化。
 *	　デバッグシステムを使用する初期準備を行います。
 */
int shInitDebug(void)
{
	// 先頭ポインタはNULL
	int i;
	unsigned int a, b;
	int r;
	char bits[32][32][3];
	g_pListIndex = NULL;

	// 各ウィンドウフラグを初期化
	for(i = 0; i < SH_DW_MAX; i ++){
		g_windowFlag[i] = 0;
	}

	for (a = 0 ; a < TEXSIZE ; a++) {
		r = (a * 0xFF) / (int)TEXSIZE;
		for (b = 0 ; b < TEXSIZE ; b++) {
			bits[a][b][0] = (GLubyte)r;
			bits[a][b][1] = (GLubyte)(( b * 0xFF ) / TEXSIZE);
			bits[a][b][2] = (GLubyte)~r;
		}
	}

	// デバッグフォントサイズの初期値
	shFontSize(DEFAULT_FONTSIZE);

	// 埋め込みデータよりテクスチャの作成
	if( shCreateTextureFromData(&g_debugTex[0], win_base_w, win_base_h, (unsigned char*)win_base_image) ){
		return 1;
	}
	if( shCreateTextureFromData(&g_debugTex[1], win_pop_w, win_pop_h, (unsigned char*)win_pop_image) ){
		return 1;
	}
	if( shCreateTextureFromData(&g_debugTex[2], win_cross_w, win_cross_h, (unsigned char*)win_cross_image) ){
		return 1;
	}
	if( shCreateTextureFromData(&g_debugTex[3], win_white_w, win_white_h, (unsigned char*)win_white_image) ){
		return 1;
	}
	if( shCreateTextureFromData(&g_debugTex[4], win_line1_w, win_line1_h, (unsigned char*)win_line1_image) ){
		return 1;
	}
	if( shCreateTextureFromData(&g_debugTex[5], win_line2_w, win_line2_h, (unsigned char*)win_line2_image) ){
		return 1;
	}
	if( shCreateTextureFromData(&g_debugTex[6], win_line3_w, win_line3_h, (unsigned char*)win_line3_image) ){
		return 1;
	}
	if( shCreateTextureFromData(&g_debugTex[7], win_line4_w, win_line4_h, (unsigned char*)win_line4_image) ){
		return 1;
	}
	if( shCreateTextureFromData(&g_debugTex[8], win_line5_w, win_line5_h, (unsigned char*)win_line5_image) ){
		return 1;
	}
	if( shCreateTextureFromData(&g_asciiTex, ascii_w, ascii_h, (unsigned char*)ascii_image )){
		return 1;
	}

	return 0;
}

/*
 *	int shExitDebug(void)
 *
 *	＜戻り値＞
 *	　なし
 *	＜説明＞
 *　　デバッグシステムの終了処理。
 *	　デバッグシステムのリソースの解放を行います。
 */
void shExitDebug(void)
{
	shDWList*	pList = g_pListIndex;

	// メモリから解放
	do{
		if(pList != NULL){
			shDWList* tmp;
			tmp = pList->pNext;
			free(pList);
			pList->pNext = tmp;
			break;
		}else{
			break;
		}
	}while(1);

	// テクスチャの削除
	shDeleteTexture(&g_debugTex[0]);
	shDeleteTexture(&g_debugTex[1]);
	shDeleteTexture(&g_debugTex[2]);
	shDeleteTexture(&g_debugTex[3]);
	shDeleteTexture(&g_debugTex[4]);
	shDeleteTexture(&g_debugTex[5]);
	shDeleteTexture(&g_debugTex[6]);
	shDeleteTexture(&g_debugTex[7]);
	shDeleteTexture(&g_debugTex[8]);
	shDeleteTexture(&g_asciiTex);
}

/*
 *	int shCreateDebugWindow(float x, float y, float width, float height)
 *
 *	＜引数＞
 *	　float x			: 左上のX座標
 *	　float y			: 左上のY座標
 *	　float width		: ウィンドウの横幅（最低１６）
 *	　float height	: ウィンドウの縦幅（最低１６）
 *	＜戻り値＞
 *	　int				: ウィンドウの番号を返します。
 *						  生成に失敗した場合には-1を返します。
 *	＜説明＞
 *	　デバッグシウィンドウを生成します。
 */
int shCreateDebugWindow(float x, float y, float width, float height)
{
	int i;
	shDWList*	pList = g_pListIndex;		// リストカウンター

	// 1個目のみIndexへポインタをセット
	if(g_pListIndex == NULL){
		g_pListIndex = (shDWList*)malloc(sizeof(shDWList));
		g_pListIndex->no = 0;
		g_pListIndex->pNext = NULL;
		pList = g_pListIndex;
		g_windowFlag[0] = 1;
	}
	else{
		// ウィンドウの空きを調べる
		for(i = 0; i < SH_DW_MAX; i ++){
			if(g_windowFlag[i] == 0){
				g_windowFlag[i] = 1;
				// リストの最後尾を検索
				do{
					// 最後尾に追加します
					if(pList->pNext == NULL){
						pList->pNext = (shDWList*)malloc(sizeof(shDWList));
						pList = pList->pNext;
						pList->no = i;
						pList->pNext = NULL;
						break;
					// 最後尾でない場合は次のリストへ
					}else{
						pList = pList->pNext;
					}
				}while(1);
				break;
			}

			// ウィンドウの空きが無かった場合
			if(i == SH_DW_MAX-1){
				return -1;
			}
		}
	}

	// 値を設定
	pList->data.x		= x;
	pList->data.y		= y;
	pList->data.width	= width;
	pList->data.height	= height;
	pList->data.windowState = SH_DW_STATE_MAXIMIZED;	// 初期値は最大

	return pList->no;
}

/*
 *	void shDeleteDebugWindow(int num)
 *
 *	＜引数＞
 *　　float num		: ウィンドウ番号
 *	＜戻り値＞
 *	　なし
 *	＜説明＞
 *	　指定された番号のデバッグウィンドウを削除します。
 */
void shDeleteDebugWindow(int num)
{
	shDWList*	pList = g_pListIndex;		// リストカウンター
	shDWList*	pPrev = NULL;				// 再連結用
	shDWList*	pNext = NULL;				// 再連結用

	while(1)
	{
		if(pList == NULL){
			break;
		}
		if(pList->no == num){
			// 次のポインタを所持
			pNext = pList->pNext;
			free(pList);
			g_windowFlag[num] = 0;

			// 再連結
			if(pPrev != NULL){
				pPrev->pNext = pNext;
			}else{
				g_pListIndex = pNext;		// 先頭時
			}
			return;
		}

		// ポインタを保存し次へ
		pPrev = pList;
		pList = pList->pNext;
	}
}

/*
 *	void shDebugDraw()
 *
 *	＜引数＞
 *		なし
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *　　デバッグ情報を描画。
 *	　設定されているすべてのデバッグ情報を描画します。
 */
void shDebugDraw(void)
{
	shDWList*	pList = g_pListIndex;
	float zPos = -0.1f;

	if(g_pListIndex == NULL)
		return;

	while(1){
		// 拡縮ボタン
		shSetSpritePosition(pList->data.x+8.0f,pList->data.y+8.0f,zPos);
		shSetSpriteSize(16.0f,16.0f);
		shSetSpriteTexture(&g_debugTex[1]);
		shSpriteDraw();
		// 閉じるボタン
		shSetSpritePosition(pList->data.x+8.0f,pList->data.y+pList->data.height-8.0f,zPos);
		shSetSpriteTexture(&g_debugTex[2]);
		shSpriteDraw();
		// バー
		shSetSpritePosition(pList->data.x+8.0f,pList->data.y+pList->data.height/2,zPos);
		shSetSpriteSize(16.0f,pList->data.height-32.0f);
		shSetSpriteTexture(&g_debugTex[0]);
		shSpriteDraw();

		// ウィンドウ最大時のみ内部を描画する
		if(pList->data.windowState == SH_DW_STATE_MAXIMIZED){
			// ─上部
			shSetSpritePosition(pList->data.x+pList->data.width/2,pList->data.y+8.0f,zPos);
			shSetSpriteSize(pList->data.width-32.0f,16.0f);
			shSetSpriteTexture(&g_debugTex[8]);
			shSpriteDraw();
			// ┐
			shSetSpritePosition(pList->data.x+pList->data.width-8.0f,pList->data.y+8.0f,zPos);
			shSetSpriteSize(16.0f,16.0f);
			shSetSpriteTexture(&g_debugTex[7]);
			shSpriteDraw();
			// │右側
			shSetSpritePosition(pList->data.x+pList->data.width-8.0f,pList->data.y+pList->data.height/2,zPos);
			shSetSpriteSize(16.0f,pList->data.height-32.0f);
			shSetSpriteTexture(&g_debugTex[6]);
			shSpriteDraw();
			// ┘
			shSetSpritePosition(pList->data.x+pList->data.width-8.0f,pList->data.y+pList->data.height-8.0f,zPos);
			shSetSpriteSize(16.0f,16.0f);
			shSetSpriteTexture(&g_debugTex[5]);
			shSpriteDraw();
			// ─下部
			shSetSpritePosition(pList->data.x+pList->data.width/2,pList->data.y+pList->data.height-8.0f,zPos);
			shSetSpriteSize(pList->data.width-32.0f,16.0f);
			shSetSpriteTexture(&g_debugTex[4]);
			shSpriteDraw();
			// 背景
			shSetSpritePosition(pList->data.x+pList->data.width/2,pList->data.y+pList->data.height/2,zPos);
			shSetSpriteSize(pList->data.width-32.0f,pList->data.height-32.0f);
			shSetSpriteTexture(&g_debugTex[3]);
			shSpriteDraw();
		}

		if(pList->pNext == NULL){
			return;
		}
		pList = pList->pNext;
	}
}

/*
 *	void shPrintf(int _x, int _y, char * fmt, ...)
 *
 *	＜引数＞
 *　　int _x		: X座標
 *　　int _y		: Y座標
 *　　char * fmt	: 文字列
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　指定位置に文字列を描画します。
 *　　winAPIのprintfに近い動作をします。現在 %d %x %s に対応しています。
 */
void shPrintf(int _x, int _y, char * fmt, ...)
{
	char **arg = (char **) &fmt;
	char c;
	arg++;

	while((c = *fmt++) != 0){
		if(c != '%'){
			shPrintChar(_x, _y, c);
		}
		else
		{
			char buf[64];
			char *p;
			c = *fmt++;
			switch(c){
			case 'd':
				if( 0 > *((int *) arg) ) {
					shPrintChar(_x, _y, '-');
					*((int *) arg) *= -1;
				}
				p = uint_to_str(buf, *((unsigned *) arg++), 10);
				goto print_s;
			case 'x':
				shPrintChar(_x, _y, '0');
				_x += g_fontSize/2;
				shPrintChar(_x, _y, 'x');
				p = uint_to_str(buf, *((unsigned *) arg++), 16);
				goto print_s;
			case 's':
				p = *arg++;
				// ラベル
				print_s:
				shPrintf(_x, _y, p);		// 再起
				_x += (g_fontSize/2)*(strlen(p)-1);
				break;
			default:
				shPrintString(_x, _y, &c);
			}
		}
		_x += g_fontSize/2;
	}
}

/*
 *	void shFontSize(int size)
 *
 *	＜引数＞
 *　　int size	: フォントサイズ
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　フォントサイズを設定する。
 */
void shFontSize(int size)
{
	g_fontSize = size;
	g_fontScale = ((float)g_fontSize/DEFAULT_FONTSIZE);
}

/*
 *	void shPrintString(int _x, int _y, char* str)
 *
 *	＜引数＞
 *　　int _x			: 描画するX座標
 *　　int _y			: 描画するY座標
 *　　char* str			: 描画する文字列
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　ASCII文字列を描画します。
 */
void shPrintString(int _x, int _y, char* str)
{
	int i = 0;

	while(str[i] != '\0')
	{
		shPrintChar(_x, _y, str[i]);
		i ++;
	}
}

/*
 *	void shPrintChar(int _x, int _y, char inChar)
 *
 *	＜引数＞
 *		int _x			: 描画するX座標
 *		int _y			: 描画するY座標
 *		char inChar		: 描画する文字
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		ASCII文字を1文字描画します。
 */
void shPrintChar(int _x, int _y, char inChar)
{
	unsigned char	buf;
	int				tmpX;
	int				tmpY;
	float			zPos = -0.1f;

	buf = inChar & 0xff;
	tmpX = buf % 16;
	tmpY = (buf /16) % 16;

//	shSetTextureWarp(GL_CLAMP, GL_CLAMP);
//	shSetSpriteTexCoord((float)tmpX/16.0f, (float)tmpY/16.0f, (float)(tmpX+1)/16.0f, (float)(tmpY+1)/16.0f);
	shSetSpriteTexCoord((float)tmpX/16.0f, (float)tmpY/16.0f, (float)1.0f/16.0f, (float)1.0f/16.0f);
	shSetSpritePosition((float)_x, (float)_y, zPos);
	shSetSpriteSize(DEFAULT_FONTSIZE/2 * g_fontScale, DEFAULT_FONTSIZE * g_fontScale);
	shSetSpriteTexture(&g_asciiTex);
	shSpriteDraw();

	// UVを元に戻す
	shSetSpriteTexCoord(0.0f, 0.0f, 1.0f, 1.0f);
}

/*
 *	char* uint_to_str(char *buf, unsigned src, int base)
 *
 *	＜引数＞
 *		******
 *	＜戻り値＞
 *		char*			: ***
 *	＜説明＞
 *		******
 */
char* uint_to_str(char *buf, unsigned src, int base)
{
	char *p = buf;
	char *p1, *p2;

	do{
		*p++ = "0123456789ABCDEF"[src%base];
	}while(src /= base);

	// Terminate BUF
	*p = 0;

	// Reverse BUF
	for(p1=buf, p2=p-1; p1 < p2; p1++, p2--) {
		char tmp = *p1;
		*p1 = *p2;
		*p2 = tmp;
	}

	return buf;
}