﻿/*
 * マルチプラットフォーム描画エンジン「Sherry」
 * Copyright(C) 2010-2011 SherryProject. all rights reserved.
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 *	shApplication_win32.c
 *		Copyright (C) 2010-2011 Cap5ule. all rights reserved.
 * 
 *	Date		|Version	|Author		|Summary
 *	2010/11/16	|v0.01		|Cap5ule	|初回版
 */

// Windowsのみ
#ifdef WIN32

/*---- インクルードファイル ----*/

#include "sherry.h"				//!<共通ヘッダー
#include "shApplication.h"		//!<ヘッダー

/*---- マクロ定義 ----*/


/*---- スタティックプロトタイプ宣言 ----*/
// 説明は実装部に記載

// ウィンドウプロシージャ
static LRESULT CALLBACK shWndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam);
// 更新ループ処理
static unsigned int CALLBACK shUpdateThread( void* pValue );
// ウィンドウの初期化・ウィンドウクラス登録
static int shWndInit(const HINSTANCE hInstance);
// OpenGLとGLEWの初期化を行う
static int shInitOpenGL(int width, int height);

/*---- グローバル変数 ----*/

// ウィンドウとコンテキスト
HWND			g_sh_hWnd;				//!<ウィンドウハンドル
HDC				g_sh_hDC;				//!<デバイスコンテキスト
HGLRC			g_sh_hRC;				//!<OpenGLレンダリングコンテキスト
shDrawMode		g_sh_DrawMode;			//!<描画モード
// 描画スレッド（メインスレッド）
void			(*g_sh_DrawFunc)();		//!<スレッド内で呼び出される関数のポインタ
// 更新スレッド
HANDLE			g_sh_hUdThread;			//!<スレッドハンドル
void			(*g_sh_UpdateFunc)();	//!<スレッド内で呼び出される関数のポインタ
int				g_sh_UdFrameRate;
// 描画設定
unsigned int	g_sh_clsFlag;			//!<クリアターゲットフラグ
// システム変数
unsigned long	g_sh_DrawPreTime;		//!<描画：前フレームの時間
float			g_sh_DrawFPS;			//!<描画：現在のFPS
unsigned long	g_sh_UpdatePreTime;		//!<更新：前フレームの時間
float			g_sh_UpdateFPS;			//!<更新：現在のFPS
// 入力系
int				g_sh_MButton;			//!<内部入力 押されているかどうか
float			g_sh_MCursor[2];		//!<内部入力 0:X,1:Y

/*---- スタティック関数 ----*/

/*
 *	static LRESULT CALLBACK shWndProc(HWND hWnd, UINT uMsg,	
 *										WPARAM wParam, LPARAM lParam)
 * 
 *	メインのウィンドウプロシージャ
 *	※ プロシージャです。情報は割愛します。
 */
static LRESULT CALLBACK shWndProc(HWND hWnd, UINT uMsg,
									WPARAM wParam, LPARAM lParam)
{
	switch(uMsg)
	{
	case WM_DESTROY:
		PostQuitMessage(0);	//終了メッセージ
		return 0;
	case WM_MOUSEMOVE:	// マウスカーソルが移動
		// 移動先の座標を取得
		g_sh_MCursor[0] = (float)LOWORD( lParam );	//X
		g_sh_MCursor[1] = (float)HIWORD( lParam );	//Y
		return 0;
	case WM_LBUTTONDOWN:// マウスカーソルの左ボタン押下
		g_sh_MButton = 1;
		return 0;
	case WM_LBUTTONUP:// マウスカーソルの左ボタン離す
		g_sh_MButton = 0;
		return 0;
	case WM_KEYDOWN:
		/*キー入力：F12キーで終了*/
		switch( wParam ) {
		case VK_F12:
			PostQuitMessage(0);	//終了メッセージ
			return 0;
		}
		break;
	default: break;
	}

	return DefWindowProc(hWnd, uMsg, wParam, lParam);
}

/*
 *	static unsigned int CALLBACK shUpdateThread( void* pValue )	
 *
 *	＜引数＞
 *　　void* pValue	: NULL
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　更新処理を行うためのマルチスレッドループを提供します。
 */
static unsigned int CALLBACK shUpdateThread( void* pValue )
{
	// ローカル変数
	unsigned int dwTime;
	unsigned int dwNowTime;

	// タイマー精度を精密化
	timeBeginPeriod(1);

	// 無限ループを行う
	dwTime = timeGetTime() + g_sh_UdFrameRate;
	while(TRUE)
	{
		// フレームレートを固定する
		if (dwTime > timeGetTime()) {
			Sleep(1);
		continue;
		}
		dwTime = timeGetTime() + g_sh_UdFrameRate;
		// 更新関数を呼び出す
		if(g_sh_UpdateFunc != NULL) {
			g_sh_UpdateFunc();
			//FPSを測る
			dwNowTime = timeGetTime();
			g_sh_UpdateFPS = 1000.0f / (dwNowTime - g_sh_UpdatePreTime);
			g_sh_UpdatePreTime = dwNowTime;
		}
	}

	timeEndPeriod(1);

	return 0;
}

/*
 *	static int shWndInit(const HINSTANCE hInstance)
 *
 *	＜引数＞
 *	　HINSTANCE hInstance：アプリケーションのインスタンス
 *	＜戻り値＞
 *	　必ず0が返ります。
 *	＜説明＞
 *	　ウィンドウの生成に必要なウィンドウクラスを登録します。
 */
static int shWndInit(const HINSTANCE hInstance)
{
	// ウィンドウクラス構造体
	WNDCLASSEX wcex;

	// 設定
	wcex.cbSize			= sizeof(WNDCLASSEX);
	wcex.style			= 0;
	wcex.lpfnWndProc	= shWndProc;
	wcex.cbClsExtra		= 0;
	wcex.cbWndExtra		= 0;
	wcex.hInstance		= hInstance;
	wcex.hIcon			= NULL;
	wcex.hCursor        = LoadCursor (NULL, IDC_ARROW);
	wcex.hbrBackground	= (HBRUSH) GetStockObject (BLACK_BRUSH);
	wcex.lpszMenuName	= NULL;
	wcex.lpszClassName	= WINDOWCLASS_NAME;
	wcex.hIconSm		= NULL;

	RegisterClassEx(&wcex);		//クラス登録

	return 0;
}

/*
 *	static int shInitOpenGL(int width, int height)
 *
 *	＜引数＞
 *	　int width		：OpenGLの描画横幅
 *	　int height	：OpenGLの描画縦幅
 *	＜戻り値＞
 *	　int			： 初期化に成功した場合は、0が返ります。
 *	＜説明＞
 *	　OpenGLとGLEWを初期化し、描画を可能にします。
 */
static int shInitOpenGL(int width, int height)
{
	// ローカル変数
	GLenum glslErr;		//エラー変数
	int pixelFormat;

	// ピクセルフォーマットの設定
	static PIXELFORMATDESCRIPTOR pfd = {
		sizeof(PIXELFORMATDESCRIPTOR),	// この構造体のサイズ
		1,								// バージョン番号
		PFD_DRAW_TO_WINDOW |			// Windowのサポート
		PFD_SUPPORT_OPENGL |			// OpenGLのサポート
		PFD_DOUBLEBUFFER,				// ダブルバッファリングのサポート
		PFD_TYPE_RGBA,					// ＲＧＢＡタイプ
		24,								// カラービット深度
		0, 0, 0, 0, 0, 0,				// color bits ignored 
		0,								// no alpha buffer 
		0,								// shift bit ignored 
		0,								// no accumulation buffer 
		0, 0, 0, 0,						// accum bits ignored 
		32,								// 32-bit z-buffer 
		0,								// no stencil buffer 
		0,								// no auxiliary buffer 
		PFD_MAIN_PLANE,					// main layer 
		0,								// reserved 
		0, 0, 0							// layer masks ignored 
	};

	// 引くセルフォーマットを選択
	pixelFormat = ChoosePixelFormat( g_sh_hDC, &pfd );
	// セットする
	SetPixelFormat( g_sh_hDC, pixelFormat, &pfd );

	// レンダリングコンテキスト
	g_sh_hRC = wglCreateContext( g_sh_hDC );	//コンテキスト作成
	wglMakeCurrent( g_sh_hDC, g_sh_hRC );	//デバイスコンテキストとレンダリングコンテキストを紐付け

	// OpenGLデフォルト設定
	glClearColor(0, 0, 0, 1);	// デフォルト背景色
	g_sh_clsFlag = GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT;	// デフォルトクリアターゲットフラグ
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_TEXTURE_2D);

	// ビューポートの設定
	glViewport(0, 0, width , height);

	// 行列変換（使用しないためデフォルトにする）
	glMatrixMode(GL_PROJECTION);	// 投影変換
	glLoadIdentity();
	glMatrixMode(GL_MODELVIEW);		// モデルビュー変換
	glLoadIdentity();

	// GLEW初期化（GLSLを使うため）
	glslErr = glewInit();
	if (glslErr != GLEW_OK)
		return 1;	/*GLEWが使えない*/

	return 0;
}

/*---- 関数 ----*/

/*
 *	int shInitCreateSherry(HINSTANCE hInst, const char* title,
 *							 int width, int height, int cmd)
 *
 *	＜引数＞
 *　　HINSTANCE hInst	: モジュールのハンドル
 *　　char* title		: ウィンドウタイトル
 *　　int	width		: ウィンドウ幅サイズ
 *　　int height		: ウィンドウ縦サイズ
 *　　shDrawMode mode : 描画モード
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　描画エンジン「Sherry」を使用するための初期化とウィンドウの作成を
 *　　行います。必ず必要となる。
 */
int shInitCreateSherry(HINSTANCE hInst, const char* title,
					  int width, int height, shDrawMode mode)
{
	// ローカル変数
	int realWidth, realHeight;
	int windowLeft, windowTop;

	// 実際のサイズにするため
	RECT clSize= { 0, 0, width, height };	//クライアントサイズ
	//DWORD style = WS_VISIBLE | WS_POPUP;
	DWORD style = WS_VISIBLE | WS_SYSMENU | WS_BORDER | WS_CAPTION | WS_OVERLAPPED | WS_MINIMIZEBOX;
	AdjustWindowRect(&clSize, style, FALSE);	//スタイルよりサイズ取得

	// 調整
	realWidth = clSize.right - clSize.left;		//クライアントサイズ横幅
	realHeight = clSize.bottom - clSize.top;	//クライアントサイズ縦幅
	windowLeft = (GetSystemMetrics(SM_CXSCREEN) - realWidth) / 2;
	windowTop = (GetSystemMetrics(SM_CYSCREEN) - realHeight) / 2;

	// ウィンドウデータ初期化
	shWndInit(hInst);	//ウィンドウクラス生成

	// ウィンドウ作成
	g_sh_hWnd = CreateWindow(WINDOWCLASS_NAME, title,
		style, windowLeft, windowTop, realWidth, realHeight,
		NULL, NULL, hInst, NULL);
	if(g_sh_hWnd == NULL)
		return 1;		/*ウィンドウ生成に失敗*/

	// デバイスコンテキストを取得する
	g_sh_hDC = GetDC(g_sh_hWnd);

	// ウィンドウを表示し更新する
	ShowWindow(g_sh_hWnd, SW_SHOW);
	UpdateWindow(g_sh_hWnd);

	// 描画するＡＰＩを選択し、初期化を行う
	switch(mode) {
		case OpenGL: shInitOpenGL(width, height);	// OpenGLで初期化する
			break;
		default:
			return 1;	/*使用できないＡＰＩ*/
	}
	g_sh_DrawMode = mode;

	// 更新ループスレッドを生成
	g_sh_hUdThread = (HANDLE)_beginthreadex(NULL, 0, shUpdateThread, NULL, 0, NULL);
	if(g_sh_hUdThread == NULL)
		return 1;		/*スレッド生成に失敗*/

	// 変数の初期化
	g_sh_DrawFunc = NULL;
	g_sh_UpdateFunc = NULL;
	g_sh_UdFrameRate = 60;
	g_sh_DrawPreTime = timeGetTime();
	g_sh_DrawFPS = 0.0f;
	g_sh_UpdatePreTime = timeGetTime();
	g_sh_UpdateFPS = 0.0f;
	g_sh_MButton = 0;
	g_sh_MCursor[0] = g_sh_MCursor[1] = 0.0f;

	return 0;
}

/*
 *	int shExitSherry()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　描画エンジン「Sherry」の終了をさせます。
 *　　描画エンジン「Sherry」を使用した際に使ったメモリを解放します。
 *　　アプリケーションの終了時に必ず呼んでください。
 */
int shExitSherry()
{
	CloseHandle(g_sh_hUdThread);	// スレッドを閉じる

	// 解放
	wglMakeCurrent(g_sh_hDC, 0);	// OpenGL系
	wglDeleteContext(g_sh_hRC);

	// デバイスコンテキストを解放する
	ReleaseDC(g_sh_hWnd, g_sh_hDC);

	return 0;
}

/*
 *	void shMainSherry()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　描画エンジン「Sherry」メインループ関数です。
 *　　この関数を呼び出すことで、プログラムはイベントの待ち受け状態になります。
 */
void shMainSherry()
{
	// ローカル変数
	MSG msg;
	unsigned int dwNowTime;

	// タイマー精度を精密化
	timeBeginPeriod(1);

	// ウィンドウ管理ループ
	PeekMessage(&msg, NULL, 0U, 0U, PM_REMOVE);
	while(msg.message != WM_QUIT)
	{
		// メッセージ受け取り
		if(PeekMessage(&msg, NULL, 0U, 0U, PM_REMOVE)) {
			TranslateMessage(&msg);	//Windows API いつもの処理
			DispatchMessage(&msg);
		}

		// 描画関数を呼び出す
		if(g_sh_DrawFunc != NULL) {
			g_sh_DrawFunc();
			//FPSを測る
			dwNowTime = timeGetTime();
			g_sh_DrawFPS = 1000.0f / (dwNowTime - g_sh_DrawPreTime);
			g_sh_DrawPreTime = dwNowTime;
		}

		// 画面の表示処理
		if(g_sh_DrawMode == OpenGL) {
			glFlush();				// OpenGLコマンドのフラッシュ
			SwapBuffers(g_sh_hDC);	// バッファ入れ替え -> 画面表示
		} else {
			/*使用不可*/
		}

		Sleep(1);
	}

	timeEndPeriod(1);
}

/*
 *	int shSetUpdateThread(int framerate, void(*thd)())
 *
 *	＜引数＞
 *　　int	framerate	: 更新間隔（フレームレート）
 *　　void(*thd)()	: 更新用関数のポインタ：使用しない場合は[NULL]
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　更新関数を別スレッドとして動作します。
 *　　そうすることで一定のアニメーションを可能にします。
 */
int shSetUpdateThread(int framerate, void(*thd)())
{
	// 更新間隔
	g_sh_UdFrameRate = framerate;

	// 関数アドレスを入れる
	g_sh_UpdateFunc = thd;

	return 0;
}

/*
 *	int shSetDrawThread(void(*thd)())
 *
 *	＜引数＞
 *　　void(*thd)()	: 描画用関数のポインタ：使用しない場合は[NULL]
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　描画関数を別スレッドとして動作します。
 *　　そうすることで最高のパフォーマンスの提供を可能にします。
 */
int shSetDrawThread(void(*thd)())
{
	// 関数アドレスを入れる
	g_sh_DrawFunc = thd;

	return 0;
}

/*
 *	void shChangeScreenMode()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　ウィンドウモードとフルスクリーンモードを切り替えます。
 *　　ウィンドウモードの場合に呼び出すとフルスクリーンモードに変わります。
 *　　逆の場合も同じです。
 */
void shChangeScreenMode()
{
	/*未実装*/
}

/*
 *	HWND shGetWindowHandle()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　HWND		: ウィンドウハンドル
 *	＜説明＞
 *　　ウィンドウハンドルを取得します。
 */
HWND shGetWindowHandle()
{
	return g_sh_hWnd;
}

/*
 *	void shDCP()
 *
 *	＜引数＞
 *		char* filePath	: 変換するファイルパス
 *	＜戻り値＞
 *		char*			: 変換されたコンテンツのパスを返す
 *	＜説明＞
 *　　DeviceContentsPathの省略形
 *　　デバイス等のハードウェア内部パスの違いをなくし、
 *　　コンテンツパスへとつなげる統一性を図る関数です。
 */
char* shDCP(const char* filePath)
{
	// ファイルパスを無変換で返す
	return (char*)filePath;
}

/*
 *	float shGetUpdateFPS()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　float			: 平均更新回数
 *	＜説明＞
 *　　１秒間あたりの更新回数を取得する。
 */
float shGetUpdateFPS()
{
	return g_sh_UpdateFPS;
}

/*
 *	float shGetDrawFPS()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　float			: 平均描画回数
 *	＜説明＞
 *　　１秒間あたりの描画回数を取得する。
 */
float shGetDrawFPS()
{
	return g_sh_DrawFPS;
}

/*
 *	int shGetCursorPosX(float* x, float* y)
 *
 *	＜引数＞
 *　　float* x		: カーソルのＸ座標を入れる
 *　　float* y		: カーソルのＹ座標を入れる
 *	＜戻り値＞
 *　　int			: 押されている場合は１が返り、押されてない場合は０が返ります。
 *	＜説明＞
 *　　基本共通入力取得用関数。
 *　　ＰＣの場合は、カーソル位置を取得でき、
 *　　スマートフォンの場合は、押されている位置を取得できる。※実用に使うべきでない。
 */
int shGetCursorPos(float* x, float* y)
{
	//カーソル情報を代入
	(*x) = g_sh_MCursor[0];
	(*y) = g_sh_MCursor[1];
	//押下判定
	return g_sh_MButton;
}


#endif	/*WIN32*/
