﻿///////////////////////////////////////////////////////////////////////////////
// This file is part of BookmarkManager.
///////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2009-2011 bis5 <bis5@users.sourceforge.jp>
// All rights reserved.
///////////////////////////////////////////////////////////////////////////////
// BookmarkManager is one of modules for SharpWebBrowser.
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the Microsoft Public Licence (Ms-PL) as published by 
// Microsoft Corp.
//
// You should have received a copy of the Microsoft Public License along 
// with this program. 
// If not, see <http://www.microsoft.com/opensource/licenses.mspx>
//
///////////////////////////////////////////////////////////////////////////////
//
// File information
// Name: ManageDialog.cs, ManageDialog.Designer.cs
// Author: bis5
// Module: BookmarkManager
//
///////////////////////////////////////////////////////////////////////////////

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace Bis5Products.SharpWebBrowser {
    public partial class ManageDialog : Form {
        private BookmarkManager mgr;

        public ManageDialog(BookmarkManager manager) {
            InitializeComponent();
            mgr = manager;
            mgr.lang.AppendDelegate(SetLang);
            SetLang();
            this.treeView1.NodeMouseClick += new TreeNodeMouseClickEventHandler(treeView1_NodeMouseClick);
        }

        void treeView1_NodeMouseClick(object sender, TreeNodeMouseClickEventArgs e) {
            treeView1.SelectedNode = e.Node;
        }

        private void SetLang() {
            button1.Text = mgr.lang.Get("Generic_OK");
            button2.Text = mgr.lang.Get("Generic_Cancel");
            button3.Text = mgr.lang.Get("Generic_Apply");
            lbName.Text = mgr.lang.Get("BookmarkManager_lbName");
            lbUrl.Text = mgr.lang.Get("BookmarkManager_lbUrl");
            lbLocation.Text = mgr.lang.Get("BookmarkManager_lbLocation");
        }

        private void RefreshList() {
            treeView1.Nodes.Clear();

            var rnode = new TreeNode(mgr.lang.Get("BookmarkManager_RootNode"));

            var tlist = new List<BookmarkItem>();
            foreach (var list in mgr.bm.Values)
                foreach (var item in list)
                    tlist.Add(item);

            var plist = new List<BookmarkItem>();
            var alist = new List<BookmarkItem>();
            var ilist = new List<BookmarkItem>();
            foreach (var item in tlist)
                if (item.Type == BookmarkItemType.Dir) {
                    if (item.HasChild)
                        plist.Add(item);
                    else
                        alist.Add(item);
                } else if (item.Type == BookmarkItemType.Item)
                    ilist.Add(item);

            var inodes = new List<NodeWithID>();
            foreach (var item in ilist) {
                var node = new NodeWithID(item.Name, item.Id);
                node.ForeColor = Color.Blue;// REMIND:アイコン決定まで
                node.ParentID = item.ParentId;
                node.Type = BookmarkItemType.Item;
                setNodeContext(ref node);
                inodes.Add(node);
            }
            var anodes = new List<NodeWithID>();
            foreach (var item in alist) {
                var node = new NodeWithID(item.Name, item.Id);
                node.ParentID = item.ParentId;
                node.Type = BookmarkItemType.Dir;
                setNodeContext(ref node);
                anodes.Add(node);
            }
            var pnodes = new List<NodeWithID>();
            foreach (var item in plist) {
                var node = new NodeWithID(item.Name, item.Id);
                node.ParentID = item.ParentId;
                setNodeContext(ref node);
                node.Type = BookmarkItemType.Dir;
                pnodes.Add(node);
            }

            foreach (var node in pnodes) {
                foreach (var child in inodes)
                    if (child.ParentID == node.ID)
                        node.Nodes.Add(child);
                foreach (var child in anodes)
                    if (child.ParentID == node.ID)
                        node.Nodes.Add(child);
                foreach (var child in pnodes)
                    if (child.ParentID == node.ID)
                        node.Nodes.Add(child);
                if (node.ParentID == -1)
                    rnode.Nodes.Add(node);
            }
            foreach (var node in anodes)
                if (node.ParentID == -1)
                    rnode.Nodes.Add(node);
            foreach (var node in inodes)
                if (node.ParentID == -1)
                    rnode.Nodes.Add(node);

            var rexp = new ToolStripMenuItem(mgr.lang.Get("BookmarkManager_Expand"));
            rexp.Font = new Font(rexp.Font, FontStyle.Bold);
            rnode.ContextMenuStrip = new ContextMenuStrip();
            rnode.ContextMenuStrip.Items.Add(rexp);

            treeView1.Nodes.Add(rnode);
            rnode.Expand();
        }

        bool isDirOnly = false;
        private void RefreshList(bool dirOnly) {
            isDirOnly = dirOnly;
            if (!dirOnly) {
                RefreshList();
                return;
            }
            treeView1.Nodes.Clear();

            var rnode = new TreeNode(mgr.lang.Get("BookmarkManager_RootNode"));

            var tlist = new List<BookmarkItem>();
            foreach (var list in mgr.bm.Values)
                foreach (var item in list)
                    tlist.Add(item);

            var plist = new List<BookmarkItem>();
            var alist = new List<BookmarkItem>();

            foreach (var item in tlist)
                if (item.Type == BookmarkItemType.Dir) {
                    if (item.HasChild)
                        plist.Add(item);
                    else
                        alist.Add(item);
                } else if (item.Type == BookmarkItemType.Item)
                    continue;


            var anodes = new List<NodeWithID>();
            foreach (var item in alist) {
                var node = new NodeWithID(item.Name, item.Id);
                node.ParentID = item.ParentId;
                anodes.Add(node);
            }
            var pnodes = new List<NodeWithID>();
            foreach (var item in plist) {
                var node = new NodeWithID(item.Name, item.Id);
                node.ParentID = item.ParentId;
                pnodes.Add(node);
            }

            foreach (var node in pnodes) {
                foreach (var child in anodes)
                    if (child.ParentID == node.ID)
                        node.Nodes.Add(child);
                foreach (var child in pnodes)
                    if (child.ParentID == node.ID)
                        node.Nodes.Add(child);
                if (node.ParentID == -1)
                    rnode.Nodes.Add(node);
            }
            foreach (var node in anodes)
                if (node.ParentID == -1)
                    rnode.Nodes.Add(node);

            var rexp = new ToolStripMenuItem(mgr.lang.Get("BookmarkManager_Expand"));
            rexp.Font = new Font(rexp.Font, FontStyle.Bold);
            rnode.ContextMenuStrip = new ContextMenuStrip();
            rnode.ContextMenuStrip.Items.Add(rexp);

            treeView1.Nodes.Add(rnode);
            rnode.Expand();
        }

        private void setNodeContext(ref NodeWithID node) {
            treeView1.ContextMenuStrip = new ContextMenuStrip();
            var col = new ToolStripItemCollection(treeView1.ContextMenuStrip, new ToolStripItem[] { });
            if (node.Type == BookmarkItemType.Dir) {
                var expand = new ToolStripMenuItem();
                expand.Text = mgr.lang.Get("BookmarkManager_Expand");
                expand.Font = new Font(expand.Font, FontStyle.Bold);
                expand.Click += new EventHandler(expand_Click);
                var sp = new ToolStripSeparator();
                var edit = new ToolStripMenuItem();
                edit.Text = mgr.lang.Get("BookmarkManager_Edit");
                edit.Click += new EventHandler(edit_Click);
                var remove = new ToolStripMenuItem();
                remove.Text = mgr.lang.Get("BookmarkManager_Remove");
                remove.Click += new EventHandler(remove_Click);
                node.ContextMenuStrip = new ContextMenuStrip();
                node.ContextMenuStrip.Items.AddRange(new ToolStripItem[] { expand, sp, edit, remove });
            } else if (node.Type == BookmarkItemType.Item) {
                var edit = new ToolStripMenuItem();
                edit.Text = mgr.lang.Get("BookmarkManager_Edit");
                edit.Click += new EventHandler(edit_Click);
                var remove = new ToolStripMenuItem();
                remove.Text = mgr.lang.Get("BookmarkManager_Remove");
                remove.Click += new EventHandler(remove_Click);
                node.ContextMenuStrip = new ContextMenuStrip();
                node.ContextMenuStrip.Items.AddRange(new[] { edit, remove });
            }
        }

        void remove_Click(object sender, EventArgs e) {
            if (MessageBox.Show(mgr.lang.Get("BookmarkManager_RemoveConfirm"), mgr.lang.Get("strName"),
                MessageBoxButtons.YesNo, MessageBoxIcon.Exclamation) == DialogResult.Yes) {
                NodeWithID tg = (NodeWithID)treeView1.SelectedNode;
                if (tg.Type == BookmarkItemType.Item)
                    mgr.RemoveItem(tg.ID);
                else
                    mgr.RemoveDirectory(tg.ID);

                if (runtype == ManageType.Dir)
                    RefreshList(true);
                else
                    RefreshList();
                treeView1.SelectedNode = treeView1.Nodes[0];
            }
        }

        void edit_Click(object sender, EventArgs e) {
            runmode = ManageMode.Edit;
        }

        void expand_Click(object sender, EventArgs e) {
            treeView1.SelectedNode.Expand();
        }

        class NodeWithID : TreeNode {
            public int ID { get; set; }
            public int ParentID { get; set; }
            public BookmarkItemType Type { get; set; }
            public NodeWithID() { }
            public NodeWithID(string name, int id) {
                base.Text = name;
                base.Name = (ID = id).ToString();
            }
            public NodeWithID(int id) {
                base.Name = (ID = id).ToString();
            }
        }

        public DialogResult ShowDialog(ManageMode mode, ManageType type) {
            runmode = mode;
            runtype = type;
            if (type == ManageType.Dir) {
                RefreshList(true);
                tbUrl.Enabled = false;
            } else {
                RefreshList();
                tbUrl.Enabled = true;
            }
            switch (mode) {
                case ManageMode.AddNew:
                    RefreshList(true);
                    this.Text = mgr.lang.Get("BookmarkManager_CreateNew");
                    treeView1.SelectedNode = null;
                    if (target != null) {
                        tbName.Text = target.Name;
                        tbUrl.Text = target.Url;
                        //TODO:新規追加時の仮想パスの設定
                        //ルートのままでいいかも
                    } else
                        tbName.Text = mgr.lang.Get("BookmarkManager_NewDirectory");
                    treeView1.BeforeSelect -= new TreeViewCancelEventHandler(treeView1_BeforeSelect);
                    treeView1.AfterSelect -= new TreeViewEventHandler(treeView1_AfterSelect);
                    treeView1.AfterSelect += new TreeViewEventHandler(treeView1_ChangeDirectory);
                    break;
                case ManageMode.Edit:
                    this.Text = mgr.lang.Get("BookmarkManager_EditItem");
                    treeView1.SelectedNode = treeView1.Nodes[target.Id.ToString()];//treeView1.AfterChangedが発生してればおｋ
                    break;
                case ManageMode.Manage:
                default:
                    this.Text = mgr.lang.Get("BookmarkManager_ManageBookmark");
                    treeView1.SelectedNode = treeView1.Nodes[0];
                    break;
            }
            var res = this.ShowDialog();
            target = null;
            this.Dispose();
            return res;
        }

        void treeView1_ChangeDirectory(object sender, TreeViewEventArgs e) {
            tbLocation.Text = treeView1.SelectedNode.FullPath;
        }

        BookmarkItem target;
        ManageMode runmode;
        ManageType runtype;
        public DialogResult ShowDialog(ManageMode mode, ManageType type, BookmarkItem item) {
            target = item;
            runmode = mode;
            runtype = type;
            if (type == ManageType.Item)
                return this.ShowDialog(mode, ManageType.Item);
            else // if(type == ManageType.Dir)
                return this.ShowDialog(mode, ManageType.Dir);
        }

        /// <summary>
        /// OK
        /// </summary>
        void button1_Click(object sender, System.EventArgs e) {
            button3_Click(sender, e);
            this.DialogResult = DialogResult.OK;
            this.Close();
        }

        /// <summary>
        /// Cancel
        /// </summary>
        void button2_Click(object sender, System.EventArgs e) {
            this.DialogResult = DialogResult.Cancel;
            this.Close();
        }

        /// <summary>
        /// Apply
        /// </summary>
        void button3_Click(object sender, System.EventArgs e) {
            if (runmode == ManageMode.Edit || runmode == ManageMode.Manage) {
                if (treeView1.SelectedNode.Text == mgr.lang.Get("BookmarkManager_RootNode"))
                    return;
                if (runtype == ManageType.Item)
                    mgr.Set(((NodeWithID)treeView1.SelectedNode).ID, tbName.Text, tbUrl.Text, -2);
                else
                    mgr.Set(((NodeWithID)treeView1.SelectedNode).ID, tbName.Text, null, -2);
            } else if (runmode == ManageMode.AddNew) {
                var item = new BookmarkItem();
                item.Name = tbName.Text;
                if (runtype == ManageType.Item) {
                    item.Url = tbUrl.Text;
                    item.Type = BookmarkItemType.Item;
                } else
                    item.Type = BookmarkItemType.Dir;
                try {
                    item.ParentId = ((NodeWithID)treeView1.SelectedNode).ID;
                } catch (InvalidCastException ex) {
                    item.ParentId = -1;
                }
                mgr.Add(item);
            }
            if (runtype == ManageType.Dir)
                RefreshList(true);
            else
                RefreshList();
        }

        /// <summary>
        /// 項目情報の更新
        /// </summary>
        private void treeView1_AfterSelect(object sender, TreeViewEventArgs e) {
            tbLocation.Text = treeView1.SelectedNode.FullPath;

            if (treeView1.SelectedNode.Text == mgr.lang.Get("BookmarkManager_RootNode")) {
                tbName.Text = "";
                tbUrl.Text = "";
                tbUrl.Enabled = false;
                return;
            }

            var item = mgr.GetItems()[((NodeWithID)treeView1.SelectedNode).ID];
            tbName.Text = item.Name;
            if (item.Type == BookmarkItemType.Dir) {
                tbUrl.Enabled = false;
                tbUrl.Text = "";
            } else {
                tbUrl.Text = item.Url;
                tbUrl.Enabled = true;
            }
        }

        /// <summary>
        /// 洗濯されたアイテムが編集されていた際に変更を保存するかどうかを確認
        /// </summary>
        void treeView1_BeforeSelect(object sender, System.Windows.Forms.TreeViewCancelEventArgs e) {
            if (treeView1.SelectedNode == null || treeView1.SelectedNode.Text == mgr.lang.Get("BookmarkManager_RootNode"))
                return;
            var item = mgr.GetItems()[((NodeWithID)treeView1.SelectedNode).ID];
            if (tbName.Text != item.Name || tbUrl.Text != item.Url) {
                if (MessageBox.Show(mgr.lang.Get("BookmarkManager_ModifyConfirm"), mgr.lang.Get("strName"),
                    MessageBoxButtons.YesNo, MessageBoxIcon.Information)
                    == DialogResult.Yes) {
                    button3_Click(sender, e);
                    if (isDirOnly)
                        RefreshList(true);
                    else
                        RefreshList();
                }
            }
        }

    }

    public enum ManageMode { AddNew, Edit, Manage }
    public enum ManageType { Dir, Item }
}