package jp.ac.titech.sharp4k.cuten.test;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.lang.reflect.Field;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import jp.ac.titech.sharp4k.cuten.Apk;
import jp.ac.titech.sharp4k.cuten.DownloadActivity;
import jp.ac.titech.sharp4k.cuten.HttpAPIClient;
import jp.ac.titech.sharp4k.cuten.HttpResponseListener;
import jp.ac.titech.sharp4k.cuten.Lecture;
import jp.ac.titech.sharp4k.cuten.LectureActivity;
import jp.ac.titech.sharp4k.cuten.LectureFolder;
import jp.ac.titech.sharp4k.cuten.LectureFolderActivity;
import jp.ac.titech.sharp4k.cuten.SQLHelper;
import jp.ac.titech.sharp4k.cuten.Task;
import jp.ac.titech.sharp4k.cuten.Teacher;
import jp.ac.titech.sharp4k.cuten.util.XmlUtility;
import android.app.Activity;
import android.app.Instrumentation.ActivityMonitor;
import android.content.Context;
import android.content.Intent;
import android.database.sqlite.SQLiteDatabase;
import android.util.JsonWriter;
import android.util.SparseArray;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ExpandableListAdapter;
import android.widget.ExpandableListView;
import android.widget.ImageButton;

import com.google.inject.AbstractModule;

public class LectureFolderActivityTest extends
		ActivityTestCaseWithRobo<LectureFolderActivity> {
	private static class NormalModule extends AbstractModule {
		protected void configure() {
			bind(HttpAPIClient.class).to(NormalHttpAPIClient.class);
		}
	}

	private static class NormalHttpAPIClient extends NullHttpAPIClient {
		static List<Lecture> dummyLectures;
		static SparseArray<List<Apk>> dummyApksForLecture;
		static List<Integer> lastIds = new ArrayList<Integer>();

		static {
			dummyLectures = new ArrayList<Lecture>();
			dummyApksForLecture = new SparseArray<List<Apk>>();
			Teacher t = new Teacher(1000, "tea1", null);
			for (int i = 0; i < 3; i++) {
				// testFolderSorted() のために、あえて id に対して講義名が降順になるようにする
				Lecture l = new Lecture(1000 + i, "lec" + (2 - i), t);
				dummyLectures.add(l);
				List<Apk> apks = new ArrayList<Apk>();
				for (int j = 0; j < 3; j++) {
					int k = 3 * i + j;
					Task task = new Task(1000 + k, "task" + k, l);
					Apk apk = new Apk(task.getId(), 1, "com.example.apk" + k,
							task);
					apks.add(apk);
				}
				dummyApksForLecture.put(l.getId(), apks);
			}
		}

		@Override
		public void getLectures(List<Lecture> lectures,
				HttpResponseListener listener) {
			StringWriter sw = new StringWriter();
			JsonWriter w = new JsonWriter(sw);
			lastIds.clear();
			try {
				w.beginArray();
				for (Lecture l1 : lectures) {
					lastIds.add(l1.getId());
					for (Lecture l2 : dummyLectures) {
						if (l1.getId() == l2.getId()) {
							ToJSON.write(w, l2,
									dummyApksForLecture.get(l2.getId()));
							break;
						}
					}
				}
				w.endArray();
				w.close();
				listener.preExec();
				listener.onSuccess(sw.toString());
				listener.postExec();
			} catch (IOException ex) {
				assert false;
			}
		}
	}

	private static final String FOLDER_FILE_NAME = "backup_"
			+ XmlUtility.fileName;
	private Activity activity;
	private ExpandableListView folderList;
	private ImageButton updateButton, createFolderBtn, moveButton;

	public LectureFolderActivityTest() {
		super(LectureFolderActivity.class, new NormalModule());
	}

	@Override
	protected void setUp() throws Exception {
		super.setUp();
		Context ctx = getInstrumentation().getTargetContext();
		try {
			copyFile(ctx, FOLDER_FILE_NAME, XmlUtility.fileName);
			ctx.deleteFile(XmlUtility.fileName);
		} catch (FileNotFoundException ex) {
			// ignore
		}
		setupDB(ctx);

		activity = getActivity();
		folderList = (ExpandableListView) activity
				.findViewById(jp.ac.titech.sharp4k.cuten.R.id.folderList);
		updateButton = (ImageButton) activity
				.findViewById(jp.ac.titech.sharp4k.cuten.R.id.updateBtn);
		moveButton = (ImageButton) activity
				.findViewById(jp.ac.titech.sharp4k.cuten.R.id.moveBtn);
		createFolderBtn = (ImageButton) activity
				.findViewById(jp.ac.titech.sharp4k.cuten.R.id.createFolderBtn);
	}

	private static void setupDB(Context ctx) {
		SQLiteDatabase db = new SQLHelper(ctx).getWritableDatabase();
		for (Lecture l : NormalHttpAPIClient.dummyLectures) {
			l.save(db);
			l.getTeacher().save(db, ctx);
		}
		NormalHttpAPIClient.dummyLectures.get(0).saveToSelected(db);
		NormalHttpAPIClient.dummyLectures.get(2).saveToSelected(db);
		db.close();
	}

	@Override
	protected void tearDown() throws Exception {
		try {
			activity.deleteFile(XmlUtility.fileName);
			copyFile(activity, XmlUtility.fileName, FOLDER_FILE_NAME);
		} catch (FileNotFoundException ex) {
			// ignore
		}
		activity.deleteFile(FOLDER_FILE_NAME);
		super.tearDown();
	}

	private static void copyFile(Context ctx, String dstName, String srcName)
			throws IOException {
		FileInputStream fis = ctx.openFileInput(srcName);
		FileOutputStream fos = ctx
				.openFileOutput(dstName, Context.MODE_PRIVATE);
		FileChannel in = fis.getChannel();
		FileChannel out = fos.getChannel();
		in.transferTo(0, in.size(), out);
	}

	public void testFolderListExists() throws Exception {
		assertNotNull(folderList);
	}

	public void testFolderListIsNotEmpty() throws Exception {
		assertFalse(folderList.getExpandableListAdapter().isEmpty());
	}

	public void testFolderListHasOneDefaultFolder() throws Exception {
		ExpandableListAdapter adapter = folderList.getExpandableListAdapter();
		assertEquals(adapter.getGroupCount(), 1);
		LectureFolder folder = (LectureFolder) adapter.getGroup(0);
		assertTrue(folder.isDefault());
	}

	public void testCreateFolderButtonCreatesNewFolder() throws Exception {
		activity.runOnUiThread(new Runnable() {
			@Override
			public void run() {
				createNewFolderFromUI("folder");
			}
		});
		waitUI();
		assertEquals(folderList.getExpandableListAdapter().getGroupCount(), 2);
	}

	public void testCreatedFolderShouldBeTheSecondLast() throws Exception {
		for (int i = 0; i < 3; i++) {
			final String folderName = "folder" + i;
			activity.runOnUiThread(new Runnable() {
				@Override
				public void run() {
					createNewFolderFromUI(folderName);
				}
			});
			waitUI();

			ExpandableListAdapter adapter = folderList
					.getExpandableListAdapter();
			int N = adapter.getGroupCount();
			LectureFolder folder = (LectureFolder) adapter.getGroup(N - 2);
			assertEquals(folder.getTitle(), folderName);
		}
	}

	public void testFolderWithExistingNameShouldNotBeCreated() throws Exception {
		final String folderName = "folder with the same name";
		activity.runOnUiThread(new Runnable() {
			@Override
			public void run() {
				createNewFolderFromUI(folderName);
			}
		});
		waitUI();
		int prev = folderList.getExpandableListAdapter().getGroupCount();
		activity.runOnUiThread(new Runnable() {
			@Override
			public void run() {
				createNewFolderFromUI(folderName);
			}
		});
		waitUI();
		int next = folderList.getExpandableListAdapter().getGroupCount();
		assertEquals(prev, next);
	}

	public void testUpdateTaskInfo() throws Exception {
		assertNotNull(updateButton);
		final List<Lecture> dummyLectures = NormalHttpAPIClient.dummyLectures;
		final SparseArray<List<Apk>> dummyApksForLecture = NormalHttpAPIClient.dummyApksForLecture;

		SQLiteDatabase db = new SQLHelper(activity).getReadableDatabase();
		for (Lecture l : dummyLectures) {
			assertTrue(l.findTasks(db).isEmpty());
		}
		click(updateButton);

		List<Lecture> selected = Lecture.findSelectedAll(db);
		List<Integer> sentIds = new ArrayList<Integer>();
		for (Lecture l : selected) {
			sentIds.add(l.getId());
		}
		assertEquals(sentIds, NormalHttpAPIClient.lastIds);

		for (Lecture l : selected) {
			List<Task> tasks = l.findTasks(db);
			List<Apk> dummyApks = dummyApksForLecture.get(l.getId());
			assertEquals(dummyApks.size(), tasks.size());
			for (int i = 0; i < tasks.size(); i++) {
				Task task = tasks.get(i);
				Task dummyTask = dummyApks.get(i).getTask();
				assertEquals(dummyTask.getId(), task.getId());
				assertEquals(dummyTask.getName(), task.getName());
			}
		}
		db.close();
	}

	// SEE #29552
	public void testSortedFolders() throws Exception {
		ExpandableListAdapter adapter = folderList.getExpandableListAdapter();
		assertEquals(1, adapter.getGroupCount());
		LectureFolder folder = (LectureFolder) adapter.getGroup(0);
		List<String> names = new ArrayList<String>();
		List<String> sorted = new ArrayList<String>();
		for (Lecture l : folder.getLectures()) {
			names.add(l.getName());
			sorted.add(l.getName());
		}
		Collections.sort(sorted);
		assertEquals(sorted, names);
	}

	public void testStartDownloadActivity() throws Exception {
		assertNotNull(moveButton);
		ActivityMonitor monitor = new ActivityMonitor(
				DownloadActivity.class.getName(), null, true);
		getInstrumentation().addMonitor(monitor);
		assertEquals(0, monitor.getHits());
		click(moveButton);
		getInstrumentation().waitForMonitorWithTimeout(monitor, 1000);
		assertEquals(1, monitor.getHits());
	}

	public void testStartLectureActivity() throws Exception {
		activity.runOnUiThread(new Runnable() {
			@Override
			public void run() {
				folderList.expandGroup(0);
				folderList.setSelectedChild(0, 0, true);
			}
		});
		waitUI();
		final int pos = folderList.getSelectedItemPosition();
		final View v = folderList.getSelectedView();
		Lecture lecture = (Lecture) folderList.getSelectedItem();

		ActivityMonitor monitor = new ActivityMonitor(
				LectureActivity.class.getName(), null, false);
		getInstrumentation().addMonitor(monitor);
		assertEquals(0, monitor.getHits());

		activity.runOnUiThread(new Runnable() {
			@Override
			public void run() {
				folderList.performItemClick(v, pos, -1);
			}
		});
		waitUI();

		Activity invoked = getInstrumentation().waitForMonitorWithTimeout(
				monitor, 1000);
		assertEquals(1, monitor.getHits());
		assertNotNull(invoked);
		assertTrue(invoked instanceof LectureActivity);
		Intent intent = invoked.getIntent();
		int lectureId = intent.getIntExtra(LectureFolderActivity.LECTURE_KEY,
				-1);
		assertEquals(lecture.getId(), lectureId);
		invoked.finish();
	}

	// UI スレッドから呼ぶこと!!
	private void createNewFolderFromUI(final String folderName) {
		createFolderBtn.performClick();

		View[] vs = getDecorViews();
		if (vs.length < 2) {
			return;
		}
		// vs[0] には普通の Panda のルート View が入っているはず。
		// vs[1] はフォルダ作成ボタンが押されたときに表示されるダイアログのルート View が入っているはず。
		View v = vs[1];

		List<View> xs = findViewsOf(EditText.class, v);
		if (xs.isEmpty() || !(xs.get(0) instanceof EditText)) {
			return;
		}
		final EditText editText = (EditText) xs.get(0);

		xs = findViewsOf(Button.class, v);
		if (xs.isEmpty()) {
			return;
		}
		final Button btn = (Button) xs.get(0);

		editText.setText(folderName);
		btn.performClick();
	}

	/*
	 * 現在のトップレベルに表示されている Window の View の配列を返す。ダイアログが表示されている状態で無理矢理ダイアログの View
	 * を得るために使っている。
	 */
	private static View[] getDecorViews() {
		try {
			Class<?> c = Class.forName("android.view.WindowManagerImpl");
			Field viewsField = c.getDeclaredField("mViews");
			Field singletonField = c.getDeclaredField("sWindowManager");
			viewsField.setAccessible(true);
			singletonField.setAccessible(true);
			Object instance = singletonField.get(null);
			return (View[]) viewsField.get(instance);
		} catch (Exception e) {
			return null;
		}
	}

	/*
	 * v をルートとする View のうち、klass を継承しているクラスのインスタンスのみからなるリストを返す。
	 */
	private static List<View> findViewsOf(Class<? extends View> klass, View v) {
		List<View> a = new ArrayList<View>();
		if (v instanceof ViewGroup) {
			ViewGroup vg = (ViewGroup) v;
			for (int i = 0; i < vg.getChildCount(); i++) {
				a.addAll(findViewsOf(klass, vg.getChildAt(i)));
			}
		} else {
			if (klass.isAssignableFrom(v.getClass())) {
				a.add(v);
			}
		}
		return a;
	}
}
