package jp.co.epson.watch.plaWasabi.security.spi;

import java.io.IOException;
import java.security.Principal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.login.LoginException;
import javax.security.auth.spi.LoginModule;
import javax.sql.DataSource;

import jp.co.epson.watch.plaWasabi.security.RolePrincipal;
import jp.co.epson.watch.plaWasabi.security.UserPrincipal;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.sun.security.auth.module.Krb5LoginModule;

/**
 * 認証をKerobros、ロール取得をデータベースから行うログインモジュール。
 * Kerberos認証を行うために、JVM引数に-Djava.security.krb5.realm=<i>レルム</i>
 * -Djava.security.krb5.kdc=<i>KDCサーバ</i>を指定して起動すること。<br/>
 * このログインモジュールのオプションは、<br/>
 * <b>dsJndiName</b>: ロールテーブルを持つデータソース名。<br/>
 * <b>rolesQuery</b>: ロールを取得するSQL:
 * "select Role, RoleGroup from Roles where userID=?"<br/>
 * 
 */
public class Krb5DbRoleLoginModule implements LoginModule {
  private static Log log = LogFactory.getLog(Krb5DbRoleLoginModule.class);

  private Subject subject;
  private CallbackHandler callbackHandler;

  /** The JNDI name of the DataSource to use */
  private String dsJndiName;
  /** The sql query to obtain the user roles */
  private String rolesQuery = "select Role, RoleGroup from Roles where PrincipalID=?";

  private Principal principal;
  private List rolePrincipals;

  private Krb5LoginModule krb5LoginModule;

  /**
   * @param options
   *          - dsJndiName: The name of the DataSource of the database
   *          containing the Principals, Roles tables rolesQuery: The prepared
   *          statement query, equivalent to:
   *          "select Role, RoleGroup from Roles where PrincipalID=?"
   */
  public void initialize(Subject subject, CallbackHandler callbackHandler, Map sharedState,
      Map options) {
    this.subject = subject;
    this.callbackHandler = callbackHandler;
    krb5LoginModule = new Krb5LoginModule();
    krb5LoginModule.initialize(subject, callbackHandler, sharedState, options);
    dsJndiName = (String) options.get("dsJndiName");
    if (dsJndiName == null)
      dsJndiName = "java:comp/env/BeamsPooledDS";

    Object tmp = options.get("rolesQuery");
    if (tmp != null)
      rolesQuery = tmp.toString();
  }

  /*
   * (non-Javadoc)
   * 
   * @see javax.security.auth.spi.LoginModule#abort()
   */
  public boolean abort() throws LoginException {
    return krb5LoginModule.abort();
  }

  /*
   * (non-Javadoc)
   * 
   * @see javax.security.auth.spi.LoginModule#commit()
   */
  public boolean commit() throws LoginException {
    boolean ret = krb5LoginModule.commit();
    if (ret) {
      if (principal == null) {
        return false;
      }

      if (!subject.getPrincipals().contains(principal)) {
        subject.getPrincipals().add(principal);
      }
      Iterator it = this.rolePrincipals.iterator();
      while (it.hasNext()) {
        Principal rolePrincipal = (Principal) it.next();
        subject.getPrincipals().add(rolePrincipal);
      }
    }
    return ret;
  }

  /*
   * (non-Javadoc)
   * 
   * @see javax.security.auth.spi.LoginModule#logout()
   */
  public boolean logout() throws LoginException {
    return krb5LoginModule.logout();
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.jboss.security.auth.spi.AbstractServerLoginModule#getRoleSets()
   */
  protected boolean authorize(String userId) throws LoginException {
    log.info("****************** getRoleSets called ***************************");
    Connection conn = null;
    PreparedStatement rolePs = null;
    ResultSet roleRs = null;
    rolePrincipals = new ArrayList();
    try {
      InitialContext ctx = new InitialContext();
      DataSource ds = (DataSource) ctx.lookup(dsJndiName);
      conn = ds.getConnection();
      // Get the user role names
      rolePs = conn.prepareStatement(rolesQuery);
      try {
        rolePs.setString(1, userId);
      } catch (ArrayIndexOutOfBoundsException ignore) {
        // The query may not have any parameters so just try it
      }
      roleRs = rolePs.executeQuery();
      while (roleRs.next()) {
        String name = roleRs.getString(1);
        try {
          Principal p = new RolePrincipal(name);
          log.trace("Assign user to role " + name);
          rolePrincipals.add(p);
        } catch (Exception e) {
          log.debug("Failed to create principal: " + name, e);
          throw new LoginException(e.toString());
        }
      }
    } catch (NamingException ex) {
      throw new LoginException(ex.toString(true));
    } catch (SQLException ex) {
      log.error("SQL failure", ex);
      throw new LoginException(ex.toString());
    } finally {
      if (roleRs != null) {
        try {
          roleRs.close();
        } catch (SQLException e) {
        }
      }
      if (rolePs != null) {
        try {
          rolePs.close();
        } catch (SQLException e) {
        }
      }
      if (conn != null) {
        try {
          conn.close();
        } catch (Exception ex) {
        }
      }
    }
    return (!rolePrincipals.isEmpty());
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.jboss.security.auth.spi.UsernamePasswordLoginModule#getUsersPassword()
   */
  protected String getUsersPassword() throws LoginException {
    return "";
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.jboss.security.auth.spi.UsernamePasswordLoginModule#validatePassword
   * (java.lang.String, java.lang.String)
   */
  protected boolean authenticate() throws FailedLoginException {
    try {
      return krb5LoginModule.login();
    } catch (Exception e) {
      throw new FailedLoginException(e.getMessage());
    }
  }

  public boolean login() throws LoginException {
    if (callbackHandler == null) {
      throw new LoginException("No CallbackHandler specified");
    }
    Callback callbacks[] = new Callback[2];
    callbacks[0] = new NameCallback("Username: ");
    callbacks[1] = new PasswordCallback("Password: ", false);

    String username = null;
    try {
      callbackHandler.handle(callbacks);
      username = ((NameCallback) callbacks[0]).getName();
      if (username == null) {
        return false;
      }
    } catch (NullPointerException e) {
      return false;
    } catch (IOException e) {
      throw new LoginException(e.toString());
    } catch (UnsupportedCallbackException e) {
      throw new LoginException(e.toString());
    }

    if (!authenticate()) {
      return false;
    }

    if (!this.authorize(username)) {
      return false;
    }
    principal = new UserPrincipal(username);
    return true;

  }

}
