const { app, ipcMain, protocol, session, BrowserWindow, BrowserView, Menu, nativeImage, clipboard, dialog, Notification } = require('electron');
const path = require('path');
const fs = require('fs');
const url = require('url');
const os = require('os');

const localShortcut = require("electron-localshortcut");

const Config = require('electron-store');
const config = new Config({
    defaults: {
        design: {
            homeButton: false,
            darkTheme: false,
            theme: 'default'
        },
        homePage: {
            defaultPage: 'my://newtab',
            defaultEngine: 'Google',
            searchEngines: [
                {
                    name: 'Google',
                    url: 'https://www.google.com/search?q=%s'
                },
                {
                    name: 'Bing',
                    url: 'https://www.bing.com/search?q=%s'
                },
                {
                    name: 'Yahoo! Japan',
                    url: 'https://search.yahoo.co.jp/search?p=%s'
                },
                {
                    name: 'goo',
                    url: 'https://search.goo.ne.jp/web.jsp?MT=%s'
                },
                {
                    name: 'Google Translate',
                    url: 'https://translate.google.com/?text=%s'
                },
                {
                    name: 'Youtube',
                    url: 'https://www.youtube.com/results?search_query=%s'
                },
                {
                    name: 'Twitter',
                    url: 'https://www.twitter.com/search?q=%s'
                },
                {
                    name: 'GitHub',
                    url: 'https://github.com/search?q=%s'
                }
            ]
        },
        adBlocker: true,
        window: {
            isCustomTitlebar: true,
            isMaximized: false,
            bounds: {
                width: 1100,
                height: 680
            }
        }
    },
});

const Datastore = require('nedb');
let db = {};

db.history = new Datastore({
    filename: path.join(app.getPath('userData'), 'Files', 'History.db'),
    autoload: true,
    timestampData: true
});

const { loadFilters, updateFilters, removeAds } = require('./AdBlocker');
const WindowManager = require('./WindowManager');
const windowManager = new WindowManager();

let loginCallback;
let mainWindow;
let subWindow;

let views = [];
let tabCount = 0;

getBaseWindow = (width = 1100, height = 680, minWidth = 320, minHeight = 600, x, y, frame = false) => {
    return new BrowserWindow({
        width, height, minWidth, minHeight, x, y, 'titleBarStyle': 'hidden', frame, fullscreenable: true,
        webPreferences: {
            nodeIntegration: true,
            webviewTag: true,
            plugins: true,
            experimentalFeatures: true,
            contextIsolation: false,
        }
    });
}

/*
showFilterDialog = () => {
    if (!hasFile()) {
        const dialogResult = dialog.showMessageBox({
            type: 'info',
            buttons: ['Ok'],
            title: '本当にデータをリセットしてよろしいですか？',
            message: '"続行"を押した場合データのリセット後アプリが再起動します。',
            defaultId: 0,
            cancelId: 1
        });

        if (dialogResult === 0) {
            showFilterDialog();
        }
    }
}
*/

registerProtocols = () => {
    protocol.registerFileProtocol('my', (request, callback) => {
        const parsed = url.parse(request.url);

        if (parsed.hostname.endsWith('.css') || parsed.hostname.endsWith('.js')) {
            return callback({
                path: path.join(app.getAppPath(), 'pages', parsed.hostname),
            });
        } else {
            return callback({
                path: path.join(app.getAppPath(), 'pages', `${parsed.hostname}.html`),
            });
        }
    }, (error) => {
        if (error) console.error('Failed to register protocol: ' + error);
    });
}

module.exports = class Application {
    loadWindow = () => {
        protocol.registerSchemesAsPrivileged([
            { scheme: 'my', privileges: { standard: true, bypassCSP: true, secure: true } }
        ]);

        app.on('ready', () => {
            loadFilters();
            registerProtocols();
            
            Menu.setApplicationMenu(null);

            const { width, height, x, y } = config.get('window.bounds');
            mainWindow = getBaseWindow(config.get('window.isMaximized') ? 1110 : width, config.get('window.isMaximized') ? 680 : height, 320, 600, x, y, !config.get('window.isCustomTitlebar'));

            config.get('window.isMaximized') && mainWindow.maximize();

            const startUrl = process.env.ELECTRON_START_URL || url.format({
                pathname: path.join(__dirname, '/../build/index.html'), // 警告：このファイルを移動する場合ここの相対パスの指定に注意してください
                protocol: 'file:',
                slashes: true,
            });

            mainWindow.loadURL(startUrl);

            localShortcut.register(mainWindow, 'CmdOrCtrl+Shift+I', () => {
                if (mainWindow.getBrowserView() == undefined) return;

                const view = mainWindow.getBrowserView();
                if (view.webContents.isDevToolsOpened()) {
                    view.webContents.devToolsWebContents.focus();
                } else {
                    view.webContents.openDevTools();
                }
            });

            localShortcut.register(mainWindow, 'CmdOrCtrl+R', () => {
                if (mainWindow.getBrowserView() == undefined) return;

                const view = mainWindow.getBrowserView();
                view.webContents.reload();
            });

            localShortcut.register(mainWindow, 'CmdOrCtrl+Shift+R', () => {
                if (mainWindow.getBrowserView() == undefined) return;

                const view = mainWindow.getBrowserView();
                view.webContents.reloadIgnoringCache();
            });

            mainWindow.on('closed', () => {
                mainWindow = null;
            });

            ['resize', 'move'].forEach(ev => {
                mainWindow.on(ev, () => {
                    config.set('window.isMaximized', mainWindow.isMaximized());
                    config.set('window.bounds', mainWindow.getBounds());
                })
            });

            mainWindow.on('maximize', this.fixBounds);
            mainWindow.on('unmaximize', this.fixBounds);
            mainWindow.on('enter-full-screen', this.fixBounds);
            mainWindow.on('leave-full-screen', this.fixBounds);
            mainWindow.on('enter-html-full-screen', this.fixBounds);
            mainWindow.on('leave-html-full-screen', this.fixBounds);

            // mainWindow.webContents.openDevTools();

            windowManager.addWindow();
        });

        app.on('window-all-closed', () => {
            if (process.platform !== 'darwin') {
                app.quit();
            }
        });

        app.on('activate', () => {
            if (mainWindow === null) {
                Menu.setApplicationMenu(null);

                const { width, height, x, y } = config.get('window.bounds');
                mainWindow = getBaseWindow(config.get('window.isMaximized') ? 1110 : width, config.get('window.isMaximized') ? 680 : height, 320, 600, x, y, !config.get('window.isCustomTitlebar'));

                config.get('window.isMaximized') && mainWindow.maximize();

                const startUrl = process.env.ELECTRON_START_URL || url.format({
                    pathname: path.join(__dirname, '/../build/index.html'), // 警告：このファイルを移動する場合ここの相対パスの指定に注意してください
                    protocol: 'file:',
                    slashes: true,
                });

                mainWindow.loadURL(startUrl);

                mainWindow.on('closed', () => {
                    mainWindow = null;
                });

                ['resize', 'move'].forEach(ev => {
                    mainWindow.on(ev, () => {
                        config.set('window.isMaximized', mainWindow.isMaximized());
                        config.set('window.bounds', mainWindow.getBounds());
                    })
                });

                mainWindow.on('maximize', this.fixBounds);
                mainWindow.on('unmaximize', this.fixBounds);
                mainWindow.on('enter-full-screen', this.fixBounds);
                mainWindow.on('leave-full-screen', this.fixBounds);
                mainWindow.on('enter-html-full-screen', this.fixBounds);
                mainWindow.on('leave-html-full-screen', this.fixBounds);

                registerProtocols();
            }
        });

        app.on('login', (e, webContents, request, authInfo, callback) => {
            e.preventDefault();

            subWindow = getBaseWindow(320, 230, 320, 230);
            subWindow.setParentWindow(mainWindow);
            subWindow.setMovable(false);
            subWindow.setResizable(false);
            subWindow.setMinimizable(false);
            subWindow.setMaximizable(false);
            const startUrl = process.env.ELECTRON_START_URL || url.format({
                pathname: path.join(__dirname, '/../build/index.html'), // 警告：このファイルを移動する場合ここの相対パスの指定に注意してください
                protocol: 'file:',
                slashes: true,
                hash: '/auth',
            });

            subWindow.loadURL(startUrl);
            loginCallback = callback;
        });

        ipcMain.on('authorization', (event, arg) => {
            loginCallback(arg.username, arg.password);
            subWindow.close();
        });

        ipcMain.on('data-history-get', (e, args) => {
            db.history.find({}).sort({ createdAt: -1 }).exec((err, docs) => {
                e.sender.send('data-history-get', { historys: docs });
            });
        });

        ipcMain.on('data-history-clear', (e, args) => {
            db.history.remove({}, { multi: true });
        });

        ipcMain.on('browserview-add', (e, args) => {
            this.addView(args.url, args.isActive);
        });

        ipcMain.on('browserview-remove', (e, args) => {
            this.removeView(args.id);
        });

        ipcMain.on('browserview-select', (e, args) => {
            this.selectView(args.id);
        });

        ipcMain.on('browserview-get', (e, args) => {
            let datas = [];
            for (var i = 0; i < views.length; i++) {
                const url = views[i].view.webContents.getURL();
                datas.push({ id: views[i].id, title: views[i].view.webContents.getTitle(), url: url, icon: url.startsWith('my://') ? undefined : `http://www.google.com/s2/favicons?domain=${url}` });
            }
            e.sender.send('browserview-get', { views: datas });
        });

        ipcMain.on('browserview-goBack', (e, args) => {
            views.filter(function (view, i) {
                if (view.id == args.id) {
                    let webContents = views[i].view.webContents;
                    if (webContents.canGoBack())
                        webContents.goBack();
                }
            });
        });

        ipcMain.on('browserview-goForward', (e, args) => {
            views.filter(function (view, i) {
                if (view.id == args.id) {
                    let webContents = views[i].view.webContents;
                    if (webContents.canGoForward())
                        webContents.goForward();
                }
            });
        });

        ipcMain.on('browserview-reload', (e, args) => {
            views.filter(function (view, i) {
                if (view.id == args.id) {
                    let webContents = views[i].view.webContents;
                    webContents.reload();
                }
            });
        });

        ipcMain.on('browserview-stop', (e, args) => {
            views.filter(function (view, i) {
                if (view.id == args.id) {
                    let webContents = views[i].view.webContents;
                    webContents.stop();
                }
            });
        });

        ipcMain.on('browserview-goHome', (e, args) => {
            views.filter(function (view, i) {
                if (view.id == args.id) {
                    let webContents = views[i].view.webContents;
                    webContents.loadURL(config.get('homePage.defaultPage'));
                }
            });
        });

        ipcMain.on('browserview-loadURL', (e, args) => {
            views.filter(function (view, i) {
                if (view.id == args.id) {
                    let webContents = views[i].view.webContents;
                    webContents.loadURL(args.url);
                }
            });
        });

        ipcMain.on('browserview-loadFile', (e, args) => {
            views.filter(function (view, i) {
                if (view.id == args.id) {
                    let webContents = views[i].view.webContents;
                    webContents.loadFile(args.url);
                }
            });
        });

        ipcMain.on('clear-browsing-data', () => {
            const ses = session.defaultSession;
            ses.clearCache((err) => {
                if (err) log.error(err);
            });

            ses.clearStorageData({
                storages: [
                    'appcache',
                    'cookies',
                    'filesystem',
                    'indexdb',
                    'localstorage',
                    'shadercache',
                    'websql',
                    'serviceworkers',
                    'cachestorage',
                ],
            });

            config.clear();
            db.history.remove({}, { multi: true });
        });

    }

    fixBounds = () => {
        if (mainWindow.getBrowserView() == undefined) return;

        const view = mainWindow.getBrowserView();
        const { width, height } = mainWindow.getContentBounds();

        view.setAutoResize({ width: true, height: true });
        if (mainWindow.isFullScreen()) {
            view.setBounds({
                x: 0,
                y: 0,
                width: width,
                height: height,
            });
        } else {
            view.setBounds({
                x: 1,
                y: 73 + 1,
                width: width - 2,
                height: mainWindow.isMaximized() ? height - 73 : (height - 73) - 2,
            });
        }
        view.setAutoResize({ width: true, height: true });
    }

    updateNavigationState = (id, view) => {
        mainWindow.webContents.send('update-navigation-state', {
            id: id,
            canGoBack: view.webContents.canGoBack(),
            canGoForward: view.webContents.canGoForward(),
        });
    }

    addView = (url, isActive) => {
        const id = tabCount++;
        this.addTab(id, url, isActive);
    }

    removeView = (id) => {
        views.filter((view, i) => {
            if (id == view.id) {
                views.splice(i, 1);
            }
        });
    }

    selectView = (id) => {
        views.filter((view, i) => {
            if (id == view.id) {
                mainWindow.setBrowserView(views[i].view);
                mainWindow.setTitle(views[i].view.webContents.getTitle());
                mainWindow.webContents.send('browserview-set', { id: id });
                this.fixBounds();
            }
        });
    }

    getViews = () => {
        let datas = [];
        for (var i = 0; i < views.length; i++) {
            const url = views[i].view.webContents.getURL();
            datas.push({ id: views[i].id, title: views[i].view.webContents.getTitle(), url: url, icon: url.startsWith('my://') ? undefined : `http://www.google.com/s2/favicons?domain=${url}` });
        }
        mainWindow.webContents.send('browserview-get', { views: datas });
    }

    addTab = (id, url = config.get('homePage.defaultPage'), isActive = true) => {
        const view = new BrowserView({
            webPreferences: {
                nodeIntegration: false,
                contextIsolation: false,
                plugins: true,
                experimentalFeatures: true,
                safeDialogs: true,
                safeDialogsMessage: '今後このページではダイアログを表示しない',
                preload: require.resolve('./Preload')
            }
        });

        view.webContents.on('did-start-loading', () => {
            mainWindow.webContents.send('browserview-start-loading', { id: id });
        });
        view.webContents.on('did-stop-loading', () => {
            mainWindow.webContents.send('browserview-stop-loading', { id: id });
        });

        view.webContents.on('did-finish-load', (e) => {
            mainWindow.setTitle(view.webContents.getTitle());
            mainWindow.webContents.send('browserview-load', { id: id, title: view.webContents.getTitle(), url: view.webContents.getURL() });

            this.updateNavigationState(id, view);
        });

        view.webContents.on('did-start-navigation', (e) => {
            const url = view.webContents.getURL();

            if (config.get('adBlocker'))
                removeAds(url, view.webContents);

            this.updateNavigationState(id, view);
        });

        view.webContents.on('page-title-updated', (e) => {
            mainWindow.setTitle(view.webContents.getTitle());
            mainWindow.webContents.send('browserview-load', { id: id, title: view.webContents.getTitle(), url: view.webContents.getURL() });

            if (!view.webContents.getURL().startsWith('my://'))
                db.history.insert({ title: view.webContents.getTitle(), url: view.webContents.getURL() });

            this.updateNavigationState(id, view);
        });

        view.webContents.on('page-favicon-updated', (e, favicons) => {
            mainWindow.setTitle(view.webContents.getTitle());
            if (favicons.length > 0) {
                mainWindow.webContents.send('browserview-load', { id: id, title: view.webContents.getTitle(), url: view.webContents.getURL(), favicon: favicons[0] });
            } else {
                mainWindow.webContents.send('browserview-load', { id: id, title: view.webContents.getTitle(), url: view.webContents.getURL(), favicon: favicons[0] });
            }

            this.updateNavigationState(id, view);
        });

        view.webContents.on('did-change-theme-color', (e, color) => {
            mainWindow.setTitle(view.webContents.getTitle());
            mainWindow.webContents.send('browserview-load', { id: id, title: view.webContents.getTitle(), url: view.webContents.getURL(), color: color });
        });

        view.webContents.on('new-window', (e, url) => {
            e.preventDefault();
            this.addView(url, true);
        });

        view.webContents.on('context-menu', (e, params) => {
            const menu = Menu.buildFromTemplate(
                [
                    ...(params.linkURL !== '' ?
                        [
                            {
                                label: '新しいタブで開く',
                                click: () => {
                                    this.addView(params.linkURL, true);
                                }
                            },
                            {
                                label: '新しいウィンドウで開く',
                                enabled: false,
                                click: () => { view.webContents.openDevTools(); }
                            },
                            {
                                label: 'シークレットウィンドウで開く',
                                enabled: false,
                                click: () => { view.webContents.openDevTools(); }
                            },
                            { type: 'separator' },
                            {
                                label: 'リンクをコピー',
                                accelerator: 'CmdOrCtrl+C',
                                click: () => {
                                    clipboard.clear();
                                    clipboard.writeText(params.linkURL);
                                }
                            },
                            { type: 'separator' }
                        ] : []),
                    ...(params.hasImageContents ?
                        [
                            {
                                label: '新しいタブで画像を開く',
                                click: () => {
                                    this.addView(params.srcURL, true);
                                }
                            },
                            {
                                label: '画像をコピー',
                                click: () => {
                                    const img = nativeImage.createFromDataURL(params.srcURL);

                                    clipboard.clear();
                                    clipboard.writeImage(img);
                                }
                            },
                            {
                                label: '画像アドレスをコピー',
                                click: () => {
                                    clipboard.clear();
                                    clipboard.writeText(params.srcURL);
                                }
                            },
                            { type: 'separator' }
                        ] : []),
                    ...(params.isEditable ?
                        [
                            {
                                label: '元に戻す',
                                accelerator: 'CmdOrCtrl+Z',
                                enabled: params.editFlags.canUndo,
                                click: () => { view.webContents.undo(); }
                            },
                            {
                                label: 'やり直す',
                                accelerator: 'CmdOrCtrl+Y',
                                enabled: params.editFlags.canRedo,
                                click: () => { view.webContents.redo(); }
                            },
                            { type: 'separator' },
                            {
                                label: '切り取り',
                                accelerator: 'CmdOrCtrl+X',
                                enabled: params.editFlags.canCut,
                                click: () => { view.webContents.cut(); }
                            },
                            {
                                label: 'コピー',
                                accelerator: 'CmdOrCtrl+C',
                                enabled: params.editFlags.canCopy,
                                click: () => { view.webContents.copy(); }
                            },
                            {
                                label: '貼り付け',
                                accelerator: 'CmdOrCtrl+V',
                                enabled: params.editFlags.canPaste,
                                click: () => { view.webContents.paste(); }
                            },
                            { type: 'separator' },
                            {
                                label: 'すべて選択',
                                accelerator: 'CmdOrCtrl+A',
                                enabled: params.editFlags.canSelectAll,
                                click: () => { view.webContents.selectAll(); }
                            },
                            { type: 'separator' }
                        ] : []),
                    ...(params.selectionText !== '' && !params.isEditable ?
                        [
                            {
                                label: 'コピー',
                                accelerator: 'CmdOrCtrl+C',
                                click: () => { view.webContents.copy(); }
                            },
                            {
                                label: `Googleで「${params.selectionText}」を検索`,
                                click: () => {
                                    this.addView(`https://www.google.co.jp/search?q=${params.selectionText}`, true);
                                }
                            },
                            { type: 'separator' }
                        ] : []),
                    {
                        label: '戻る',
                        accelerator: 'Alt+Left',
                        enabled: view.webContents.canGoBack(),
                        click: () => { view.webContents.goBack(); }
                    },
                    {
                        label: '進む',
                        accelerator: 'Alt+Right',
                        enabled: view.webContents.canGoForward(),
                        click: () => { view.webContents.goForward(); }
                    },
                    {
                        label: '再読み込み',
                        accelerator: 'CmdOrCtrl+R',
                        click: () => { view.webContents.reload(); }
                    },
                    { type: 'separator' },
                    {
                        label: 'ページの保存',
                        accelerator: 'CmdOrCtrl+S',
                        enabled: !view.webContents.getURL().startsWith('my://'),
                        click: () => {
                            view.webContents.savePage(`${app.getPath('downloads')}/${view.webContents.getTitle()}.html`, 'HTMLComplete', (err) => {
                                if (!err) console.log('Page Save successfully');
                            });
                        }
                    },
                    {
                        label: '印刷',
                        accelerator: 'CmdOrCtrl+P',
                        enabled: !view.webContents.getURL().startsWith('my://'),
                        click: () => { view.webContents.print(); }
                    },
                    { type: 'separator' },
                    {
                        label: 'デベロッパーツール',
                        accelerator: 'CmdOrCtrl+Shift+I',
                        enabled: !view.webContents.getURL().startsWith('my://'),
                        click: () => { if (view.webContents.isDevToolsOpened()) { view.webContents.devToolsWebContents.focus(); } else { view.webContents.openDevTools(); } }
                    }
                ]
            );

            menu.popup();
        });

        view.webContents.on('before-input-event', (e, input) => {
            if ((input.control || input.meta) && input.shift && input.key == 'I') {
                e.preventDefault();
                if (view.webContents.isDevToolsOpened()) {
                    view.webContents.devToolsWebContents.focus();
                } else {
                    view.webContents.openDevTools();
                }
            } else if ((input.control || input.meta) && input.key == 'R') {
                e.preventDefault();
                view.webContents.reload();
            } else if ((input.control || input.meta) && input.shift && input.key == 'R') {
                e.preventDefault();
                view.webContents.reloadIgnoringCache();
            }
        });

        view.webContents.session.on('will-download', (event, item, webContents) => {
            item.on('updated', (e, state) => {
                if (state === 'interrupted') {
                    console.log('Download is interrupted but can be resumed')
                } else if (state === 'progressing') {
                    if (item.isPaused()) {
                        console.log('Download is paused')
                    } else {
                        console.log(`Received bytes: ${item.getReceivedBytes()}`)
                    }
                }
            });

            item.once('done', (e, state) => {
                if (state === 'completed') {
                    console.log('Download successfully')
                } else {
                    console.log(`Download failed: ${state}`)
                }
            });
        });

        view.webContents.loadURL(url);

        views.push({ id: id, view: view });

        if (isActive) {
            mainWindow.webContents.send('browserview-set', { id: id });
            mainWindow.setBrowserView(view);
        }
        this.fixBounds();

        this.getViews();
    }

    loadExtension = (id) => {
        console.log(os.homedir())
        const extensionDir = path.resolve(os.homedir(), 'AppData\\Local\\Google\\Chrome\\User Data\\Default\\Extensions');

        const versions = fs.readdirSync(`${extensionDir}/${id}`).sort();
        const version = versions.pop();

        BrowserWindow.addExtension(`${extensionDir}/${id}/${version}`);
    }
};