<?php
/**
 * ユーティリティクラス
 *
 * ユーティリティファンクション群
 *
 * @package   risoluto
 * @author    Yuta Hayakawa <haya@hyec.org>
 * @license   http://opensource.org/licenses/bsd-license.php new BSD license
 * @copyright (C) 2008-2009 Yuta Hayakawa / All Rights Reserved.
 */

  /**
   * セッションクラス
   */
  require_once( RISOLUTO_FUNC . 'risoluto_session.php' );

  class RisolutoUtils
  {
    //------------------------------------------------------//
    // クラス変数定義
    //------------------------------------------------------//
    /**
     * クラスインスタンスを保持する変数
     * @access private
     * @var    object
     */
    private static $obj_instance;
    /**
     * セッションクラスインスタンスを保持する変数
     * @access private
     * @var    object
     */
    private $obj_sess;

    //------------------------------------------------------//
    // クラスメソッド定義
    //------------------------------------------------------//
    /**
     * コンストラクタメソッド
     *
     * コントローラのコンストラクタメソッド
     *
     * @param     void なし
     * @return    void なし
     */
    private function __construct()
    {
    } // end of function:__construct()

    /**
     * クローンメソッド
     *
     * コントローラのクローンメソッド
     *
     * @param     void なし
     * @return    void なし
     */
    public function __clone()
    {
    } // end of function:__clone()

    /**
     * シングルトンメソッド
     *
     * コントローラのインスタンスをシングルトンパターンで生成する
     *
     * @param     void なし
     * @return    object インスタンス
     */
    public static function singleton()
    {
        if ( ! isset( self::$obj_instance ) )
        {
            $tmp_myself = __CLASS__;
            self::$obj_instance = new $tmp_myself;
        } // end of if

        return self::$obj_instance;
    }

    /**
     * スーパバイザセッション判定メソッド
     *
     * 現在のセッションがスーパバイザセッションかを判定する
     *
     * @param     void なし
     * @return    boolean(true:スーパバイザセッション/false:スーパバイザセッションではない)
     */
    public function is_Supervisor()
    {

      //-- ローカル変数 --//
      $retval   = false;
      if ( empty( $this->obj_sess ) )
      {
        $this->obj_sess = RisolutoSession::singleton();
      } // end of if
      $sessinfo = $this->obj_sess->sessLoad( 'login_groupid' );

      if ( $sessinfo === '0' )
      {
        $retVal = true;
      } // end of if

      return $retVal;

    } // end of function:is_Supervisor()

    /**
     * アドミンセッション判定メソッド
     *
     * 現在のセッションがアドミンセッションかを判定する
     *
     * @param     void なし
     * @return    boolean(true:アドミンセッション/false:アドミンセッションではない)
     */
    public function is_Admin()
    {

      //-- ローカル変数 --//
      $retval   = false;
      if ( empty( $this->obj_sess ) )
      {
        $this->obj_sess = RisolutoSession::singleton();
      } // end of if
      $sessinfo = $this->obj_sess->sessLoad( 'login_groupid' );

      if ( $sessinfo === '1' || $this->is_Supervisor() )
      {
        $retVal = true;
      } // end of if

      return $retVal;

    } // end of function:is_Admin()

    /**
     * ユーザセッション判定メソッド
     *
     * 現在のセッションがユーザセッションかを判定する
     *
     * @param     void なし
     * @return    boolean(true:ユーザセッション/false:ユーザセッションではない)
     */
    public function is_User()
    {

      //-- ローカル変数 --//
      $retval   = false;
      if ( empty( $this->obj_sess ) )
      {
        $this->obj_sess = RisolutoSession::singleton();
      } // end of if
      $sessinfo = $this->obj_sess->sessLoad( 'login_groupid' );

      if ( $sessinfo === '2' || $this->is_Admin() || $this->is_Supervisor() )
      {
        $retVal = true;
      } // end of if

      return $retVal;

    } // end of function:is_User()

    /**
     * ゲストセッション判定メソッド
     *
     * 現在のセッションがゲストセッションかを判定する
     *
     * @param     void なし
     * @return    boolean(true:ゲストセッション/false:ゲストセッションではない)
     */
    public function is_Guest()
    {

      //-- ローカル変数 --//
      $retval   = false;
      if ( empty( $this->obj_sess ) )
      {
        $this->obj_sess = RisolutoSession::singleton();
      } // end of if
      $lastcage = $this->obj_sess->sessLoad( 'lastcage' );
      $lastact  = $this->obj_sess->sessLoad( 'lastact'  );

      if ( $this->is_User() || $this->is_Admin() || $this->is_Supervisor() )
      {
        $retVal = true;
      } // end of if
      else
      if (
                ! empty( $lastcage ) and $lastcage != 'firsttime' and $lastcage != 'reset' and $lastcage != 'unknown'
           and  ! empty( $lastact  ) and $lastact  != 'firsttime' and $lastact  != 'reset' and $lastact  != 'unknown'
         )
      {
         $retVal = true;
      } // end of if

      return $retVal;

    } // end of function:is_Guest()

    /**
     * Risoluto内画面リダイレクトメソッド
     *
     * 指定された画面へリダイレクトする
     *
     * @param     string $actionId リダイレクト先のAct
     * @param     string $cageId   リダイレクト先のCage
     * @param     array  $getKey   リダイレクト時に付与するGETパラメタのキー部（配列指定）
     * @param     array  $getVal   リダイレクト時に付与するGETパラメタのバリュー部（配列指定）
     * @access    public
     * @return    void なし
     */
    public function redirectTo( $actionId = null, $cageId = null, $getKey = null, $getVal = null )
    {

      // 必要なインスタンスを生成
      $obj_conf = new RisolutoConf();
      $obj_conf->parse( RISOLUTO_CONF . 'risoluto.ini' );

      // 基本情報取得
      $scheme = ( $_SERVER[ 'SERVER_PORT' ] == $obj_conf->get( 'PORT', 'secureport' ) ? 'https://' : 'http://' );
      $host   =   $_SERVER[ 'HTTP_HOST'   ];
      $self   =   $_SERVER[ 'PHP_SELF'    ];

      // Cage及びActを構成
      unset( $param );
      if ( !empty( $actionId ) )
      {
        $param  = "?act=$actionId";
      } // end of if

      if ( !empty( $cageId ) )
      {
        if ( empty( $param ) )
        {
          $param  = "?cage=$cageId";
        } // end of if
        else
        {
          $param  = $param . "&cage=$cageId";
        } // end of else
      } // end of if

      // 他のパラメタが指定されていたら、それをGetパラメタの形に生成
      if ( ! empty( $getKey ) and ! empty( $getVal ) )
      {
        $otherParam = null;
        $tmp_keys   = explode( ',', $getKey );
        $tmp_vals   = explode( ',', $getVal );

        // 両方の要素数が合致しない場合は処理を行わない
        if ( count( $tmp_keys ) == count( $tmp_vals ) )
        {
          for ( $i = 0; $i < count( $tmp_keys ); $i++ )
          {
            $otherParam .= '&' . $tmp_keys[ $i ] . '=' . $tmp_vals[ $i ];
          } // end of for
        } // end of if
      } // end of if

      // ヘッダを出力する
      header("Location: $scheme$host$self$param$otherParam");

    } // end of function:redirectTo()

    /**
     * ファイル／ディレクトリステータスチェックメソッド
     *
     * 指定されたファイルやディレクトリのステータスをチェックする
     *
     * @param     array  $target チェック対象の情報が格納された連想配列
     * @access    public
     * @return    array  チェック結果が格納された連想配列
     */
    public function statChecker( $target )
    {
      // ローカル変数の初期化
      $result = array(
                       'path'     => 'unknown'
                     , 'required' => 'unknown'
                     , 'real'     => 'unknown'
                     , 'result'   => 'unknown'
                     );

      // 引数が配列でない場合は即時return
      if ( ! is_array( $target ) )
      {
        return $result;
      } // end of if

      // 共通情報はまとめてセット
      $result[ 'path'     ] = $target[ 'path' ];
      $result[ 'required' ] = $target[ 'stat' ];

      // 対象が存在しない場合は「missing」をセットし、即時return
      clearstatcache();
      if ( ! file_exists( $target[ 'path' ] ) )
      {
        // 結果をセット
        $result[ 'real'   ] = 'missing';
        $result[ 'result' ] = 'NG';

        return $result;
      } // end of if

      // キャッシュステータスのクリア
      clearstatcache();
      // 判定を実施（defaultは書かない）
      // 評価項目を増やすにはここにcaseを追加してください
      switch( $target[ 'stat' ] )
      {
        // 読込可
        case 'readable':
          if ( is_readable( $target[ 'path' ] ) and ! is_writable( $target[ 'path' ] ) )
          {
            // 結果をセット
            $result[ 'real'   ] = 'readable';
            $result[ 'result' ] = 'OK';
          } // end of if
          else
          {
            // 結果をセット
            $result[ 'real'   ] = 'writable';
            $result[ 'result' ] = 'NG';
          } // end of else
          break;

        // 書込可
        case 'writable':
          if ( is_writable( $target[ 'path' ] ) )
          {
            // 結果をセット
            $result[ 'real'   ] = 'writable';
            $result[ 'result' ] = 'OK';
          } // end of if
          else
          {
            // 結果をセット
            $result[ 'real'   ] = 'readable';
            $result[ 'result' ] = 'NG';
          } // end of else
          break;
      } // end of switch

      return $result;

    } // end of function:statChecker()

    /**
     * ファイル／ディレクトリ操作メソッド
     *
     * 指定されたファイルやディレクトリに対し、作成/コピー/移動/削除等を行う
     *
     * @param     string  $operation   処理内容を示す文字列(make/copy/move/unlink/mkdir/rmdir)
     * @param     string  $target      対象となるパス
     * @param     string  $destination コピー又は移動先となるパス
     * @access    public
     * @return    boolean 処理結果（true: 成功 / false: 失敗）
     */
    public function fileOperator( $operation, $target, $destination = null )
    {
      // エラーフラグの初期化
      $result = false;

      // operationの内容によって、処理を分ける
      switch( $operation )
      {
        // make
        case 'make' :
          if ( ! @touch( $target ) )
          {
            $result = true;
          } // end of if
          break;

        // copy
        case 'copy' :
          if ( ! @copy( $target, $destination ) )
          {
            $result = true;
          } // end of if
          break;

        // move
        case 'move' :
          if ( ! @rename( $target, $destination ) )
          {
            $result = true;
          } // end of if
          break;

        // unlink
        case 'unlink' :
          if ( ! @unlink( $target ) )
          {
            $result = true;
          } // end of if
          break;

        // mkdir
        case 'mkdir' :
          if ( ! @mkdir( $target, 0777, true ) )
          {
            $result = true;
          } // end of if
          break;

        // rmdir
        case 'rmdir' :
          if ( ! @rmdir( $target ) )
          {
            $result = true;
          } // end of if
          break;

      } // end of switch

      return $result;

    } // end of function:fileOperator()

    /**
     * cuser/muser向け識別子生成メソッド
     *
     * 各種テーブルのcuserまたはmuserにセットするための識別子を生成する
     *
     * @param     void なし
     * @return    string 識別子（何らかの理由で生成できない場合は、null）
     */
    public function answerMyId()
    {

      //-- ローカル変数 --//
      $retval   = false;
      if ( empty( $this->obj_sess ) )
      {
        $this->obj_sess = RisolutoSession::singleton();
      } // end of if
      $currentcage = str_replace( DIRECTORY_SEPARATOR, '.', $this->obj_sess->sessLoad( 'currentcage' ) );
      $currentact  = $this->obj_sess->sessLoad( 'currentact'  );

      // 値が取得できなければ、nullを返す
      if ( empty( $currentcage ) or empty( $currentact ) )
      {
        return null;
      } // end of if

      // 値が取得できないか、現在のcageまたはactが特別な値なら、nullを返す
      if (
              empty( $currentcage ) or $currentcage == 'firsttime' or $currentcage == 'reset' or $currentcage == 'unknown'
           or empty( $currentact  ) or $currentact  == 'firsttime' or $currentact  == 'reset' or $currentact  == 'unknown'
         )
      {
        return null;
      } // end of if

      return substr( $currentcage . '/' . $currentact, 0, 255);

    } // end of function:answerMyId()

    /**
     * serviceStatus()
     *
     * サービスの状態を判定する
     *
     * @param     void なし
     * @return    integer ( 0:オープン/1:既存セッションのみ許可/2:ユーザセッションのみ許可/3:管理者セッションのみ許可/4:スーパバイザセッションのみ許可 )
     */
    public function serviceStatus()
    {

      //-- ローカル変数 --//
      $retval = 0;

      // 必要なインスタンスを生成
      $obj_conf = new RisolutoConf();
      $obj_conf->parse( RISOLUTO_CONF . 'risoluto.ini' );

      // 各サービスストップ関連のファイルが存在するかどうかをチェック
      clearstatcache();
      if ( file_exists( RISOLUTO_SYSROOT . $obj_conf->get( 'SERVICESTOP', 'ServiceStop' ) ) )
      {
        // サービスストップ（＝スーパバイザセッションのみ許可）
        $retVal = 4;
      } // end of if
      else
      if ( file_exists( RISOLUTO_SYSROOT . $obj_conf->get( 'SERVICESTOP', 'AdminOnly' ) ) )
      {
        // 管理者のみ許可
        $retVal = 3;
      } // end of else if
      else
      if ( file_exists( RISOLUTO_SYSROOT . $obj_conf->get( 'SERVICESTOP', 'UserOnly' ) ) )
      {
        // ユーザセッションのみ許可
        $retVal = 2;
      } // end of else if
      else
      if ( file_exists( RISOLUTO_SYSROOT . $obj_conf->get( 'SERVICESTOP', 'CurrentSessionOnly' ) ) )
      {
        // 既存セッションのみ許可
        $retVal = 1;
      } // end of else if
      else
      {
        // オープン（デフォルト）
        $retVal = 0;
      } // end of else

      return $retVal;

    } // end of function:serviceStatus()

    /**
     * 自動リンク設定メソッド
     *
     * 引数で指定された文字列中の特定部分をリンクに変換する
     *
     * @param     string    $val 対象となる文字列
     * @return    string    変換後の文字列
     */
    public function autoUrlLink( $val )
    {

      // 一度、一時変数へ格納する
      $tmp_val = $val;

      // 文字列中の「http」又は「https」で始まる部分を、<a>タグに変換する
      $tmp_val = preg_replace( "/(http|https)\:\/\/[[:alnum:]-_.!~*'();\/?:@&=+$,%#]+/i"
                              , "<a href='$0' target='_blank'>$0</a>"
                              , $tmp_val );
      // タグの途中で改行が入っている場合、取り除く
      $tmp_val = preg_replace( "/(\r|\n|\r\n)'>/i",    "'>",   $tmp_val );
      $tmp_val = preg_replace( "/(\r|\n|\r\n)<\/a>/i", "</a>", $tmp_val );

      // <a>タグ以外のタグを消し去る
      $tmp_val = strip_tags( $tmp_val, "<a>" );

      return $tmp_val;
    }  // end of function:autoUrlLink()

  }  // end of class:RisolutoUtils
?>
