/*****************************************************************************
 *    Licensed to the Apache Software Foundation (ASF) under one
 *    or more contributor license agreements.  See the NOTICE file
 *    distributed with this work for additional information
 *    regarding copyright ownership.  The ASF licenses this file
 *    to you under the Apache License, Version 2.0 (the
 *    "License"); you may not use this file except in compliance
 *    with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing,
 *    software distributed under the License is distributed on an
 *    "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *    KIND, either express or implied.  See the License for the
 *    specific language governing permissions and limitations
 *    under the License.
 *
 * R}hCC^v^
 * $Id: MyConsole.c 27 2012-11-16 00:29:24Z xeerda $
 *****************************************************************************/
#include "rbtos.h"
#include "rbtDevice.h"

#define MAX_CHARS					(128)
#define RESERVE_FOR_ARGV			(8)
#define DEFAULT_WIDTH				(80)


static RBTOS_WCHAR_T buff[MAX_CHARS + RESERVE_FOR_ARGV];
static short nPosX, nPosY;
static short nIndex, nLength;

static const char PS1[]= "$> ";
#define PS1_LENGTH	(3)

typedef struct _CommandTableT {
	const char *cmdname;
	int (*func)(int argc, RBTOS_WCHAR_T **argv);
	const char *helpmsg;
} CommandTableT;


static int xCMD_clear(int argc, RBTOS_WCHAR_T **argv)
{
	con_putstring("\033[2J\033[0;0H");
	return 0;
}

static int xCMD_echo(int argc, RBTOS_WCHAR_T **argv)
{
	int i;
	for (i = 1; i < argc; i++) {
		con_printf("%S", argv[i]);
		if (i < (argc - 1)) {
			con_putchar(' ');
		}
	}
	con_putstring("\r\n");
	return 0;
}


static RBTOS_ThreadT test_thread1;
static RBTOS_ThreadT test_thread2;
#define TESTTHREAD_STACK_DEPTH	160
static RBTOS_STACK_ELEM_T  TestThread1Stack[TESTTHREAD_STACK_DEPTH];
static RBTOS_STACK_ELEM_T  TestThread2Stack[TESTTHREAD_STACK_DEPTH];
static RBTOS_EventT evtTest1 = RBTOS_INITIALIZED_EVENT_VALUE;
static RBTOS_EventT evtTest2 = RBTOS_INITIALIZED_EVENT_VALUE;
static RBTOS_LockT lockTest = RBTOS_INITIALIZED_LOCK_VALUE;
typedef struct _TEST1_ARG_T {
	unsigned fAlive:1;
	unsigned fReqQuit:1;
	long count;
	const char *name;
	RBTOS_EventT *evtSelf;
	RBTOS_EventT *evtTgt;
} TEST1_ARG_T;
static long test_total;


void test_thread_func(void *arg)
{
	TEST1_ARG_T *param = (TEST1_ARG_T*) arg;
	con_printf("> %s started.\n", param->name);
	while (param->fReqQuit == 0) {
		RBTOS_Lock_enter(&lockTest);
		long n = test_total;
		param->count++;
		RBTOS_Event_set(param->evtTgt);
		test_total = n + 1;
		RBTOS_Lock_leave(&lockTest);
		RBTOS_Event_wait(param->evtSelf, 0);
	}
	con_printf("> %s Done.\n", param->name);
	param->fAlive = 0;
}

static int xCMD_test1(int argc, RBTOS_WCHAR_T **argv)
{
	TEST1_ARG_T arg1, arg2;
	test_total = 0;
	arg1.count = 0;
	arg1.fAlive = 1;
	arg1.fReqQuit = 0;
	arg1.evtSelf = &evtTest1;
	arg1.evtTgt = &evtTest2;
	arg1.name = "Test Thread #1";
	arg2.count = 0;
	arg2.fAlive = 1;
	arg2.fReqQuit = 0;
	arg2.evtSelf = &evtTest2;
	arg2.evtTgt = &evtTest1;
	arg2.name = "Test Thread #2";
	RBTOS_Thread_sleep(100);
	RBTOS_Thread_create(&test_thread1, test_thread_func, &arg1, 100,
						TestThread1Stack, sizeof(TestThread1Stack), "TEST THREAD1");
	RBTOS_Thread_create(&test_thread2, test_thread_func, &arg2, 101,
						TestThread2Stack, sizeof(TestThread2Stack), "TEST THREAD2");
	unsigned long prevTick = getCurTick();
	#define TEST1_TICKS	1000
	RBTOS_Thread_sleep(TEST1_TICKS);
	arg1.fReqQuit = 1;
	arg2.fReqQuit = 1;
	con_printf("> waiting thread finished...\n");
	while (1) {
		if ((arg1.fAlive == 0) && (arg2.fAlive == 0)) {
			break;
		}
		RBTOS_Event_set(&evtTest1);
		RBTOS_Event_set(&evtTest2);
		RBTOS_Thread_sleep(5);
	}
	unsigned long elapsed = getCurTick() - prevTick;
	con_printf("> done, total_count: %ld\n", test_total);
	con_printf("- elapsed time: %lu (ticks)\n", elapsed);
	con_printf("- %s count: %ld\n", arg1.name, arg1.count);
	con_printf("- %s count: %ld\n", arg2.name, arg2.count);
	if ((arg1.count + arg2.count) == test_total) {
		con_printf("- performance: %ld thread switches per second.\n",
				   test_total * TEST1_TICKS / elapsed);
		con_printf("- Result: <Passed>\n");
		return 0;
	} else {
		con_printf("- Result: <Falied>\n");
		return -1;
	}
}


static int xCMD_help(int argc, RBTOS_WCHAR_T **argv);


static const CommandTableT CommandTable[] = {
	{	"clear",		xCMD_clear,	" clear the terminal screen" },
	{	"echo",			xCMD_echo,		"display a line of text" },
	{	"help",			xCMD_help,		"list of commands" },
	{	"test1",		xCMD_test1,		"OS Thread Test #1" },
};

static int xCMD_help(int argc, RBTOS_WCHAR_T **argv)
{
	int i;
	for (i = 0; i < (sizeof(CommandTable) / sizeof(CommandTable[0])); i++) {
		con_printf("%s:\n"
					"    %s\n", CommandTable[i].cmdname, CommandTable[i].helpmsg);
	}
	return 0;
}

static RBTOS_WCHAR_T **parseCmdLine(int *pargc)
{
	RBTOS_WCHAR_T *cmdline = buff;
	unsigned len = nLength;
	RBTOS_WCHAR_T **argv = (RBTOS_WCHAR_T **) &buff[((len + 1) + 1) & ~1];
	unsigned maxargc = (MAX_CHARS + RESERVE_FOR_ARGV - (((len + 1) + 1) & ~1)) / 2;
	
	enum { MD_WhiteSpace, MD_Char, MD_DoubleQuad } mode = MD_WhiteSpace;
	int argc = 0;
	unsigned i;
	for (i = 0; i < len; i++) {
		RBTOS_WCHAR_T c = cmdline[i];
		unsigned fIsWhiteSpace = ((c == ' ') || (c == '\t'));
		unsigned fIsDoubleQuad = (c == '\"');
		if (mode == MD_WhiteSpace) {
			if (!fIsWhiteSpace) {
				if (argc == maxargc) {
					break;
				}
				if (fIsDoubleQuad) {
					mode = MD_DoubleQuad;
					argv[argc] = cmdline + i + 1;
				} else {
					mode = MD_Char;
					argv[argc] = cmdline + i;
				}
				argc++;
			}
			
		} else if (mode == MD_Char) {
			if (fIsDoubleQuad) {
				memmove(cmdline + i, cmdline + i + 1, sizeof(RBTOS_WCHAR_T) * (len - i - 1));
				i--;
				len--;
				mode = MD_DoubleQuad;
			} else {
				if (fIsWhiteSpace) {
					cmdline[i] = '\0';
					mode = MD_WhiteSpace;
				}
			}
			
		} else if (mode == MD_DoubleQuad) {
			if (fIsDoubleQuad) {
				memmove(cmdline + i, cmdline + i + 1, sizeof(RBTOS_WCHAR_T) * (len - i - 1));
				i--;
				len--;
				mode = MD_Char;
			}
		}
	}
	if (mode != MD_WhiteSpace) {
		cmdline[len] = '\0';
	}
	*pargc = argc;
	return argv;
}


static int getcharsize(RBTOS_WCHAR_T ch)
{
	if ((ch < 0xA7) || ((ch >= 0xFF61) && (ch <= 0xFF9F))) {
		return 1;
	}
	return 2;
}

static void escMvUp(int n)
{
	if (n != 0) {
		con_printf("\033[%dA", n);
	}
}

static void escMvRight(int n)
{
	if (n != 0) {
		con_printf("\033[%dC", n);
	}
}

static void escMvLeft(int n)
{
	if (n != 0) {
		con_printf("\033[%dD", n);
	}
}

static void escClsToLineEnd(void)
{
	con_putstring("\033[K");
}

static void escNextLine(void)
{
	con_putstring("\r\n");
}


static void escMvLineTop(void)
{
	con_putchar('\r');
}

static void initLine(void)
{
	nPosY = nIndex = nLength = 0;
	con_putstring(PS1);
	nPosX = PS1_LENGTH;
}


static void layout(void)
{
	int tx, ty, i;
	unsigned crn;
	escClsToLineEnd();
	con_putstring("\0337");
	tx = nPosX;
	ty = nPosY;
	crn = 0;
	for (i = nIndex; i < nLength; i++) {
		unsigned ch = buff[i];
		int cw = getcharsize(ch);
		if ((tx + cw) < DEFAULT_WIDTH) {
			con_putwchar(ch);
			tx += cw;
		} else {
			if ((tx + cw) > DEFAULT_WIDTH) {
				escClsToLineEnd();
				escNextLine();
				con_putwchar(ch);
				tx = cw;
			} else {
				con_putwchar(ch);
				escNextLine();
				tx = 0;
			}
			escClsToLineEnd();
			crn++;
			ty++;
		}
	}
	if (crn != 0) {
		escMvLineTop();
		escMvUp(crn);
		escMvRight(nPosX);
	} else {
		con_putstring("\0338");
	}
}


static void evChar(int ch)
{
	if (nLength < MAX_CHARS) {
		int cw;
		if (nIndex != nLength) {
			memmove(buff + nIndex + 1, buff + nIndex,
					sizeof(unsigned short) * (nLength - nIndex));
		}
		buff[nIndex] = ch;
		nLength++;
		nIndex++;
		cw = getcharsize(ch);
		if ((nPosX + cw) < DEFAULT_WIDTH) {
			con_putwchar(ch);
			nPosX += cw;
		} else {
			if ((nPosX + cw) > DEFAULT_WIDTH) {
				escClsToLineEnd();
				escNextLine();
				con_putwchar(ch);
				nPosX = cw;
			} else {
				con_putwchar(ch);
				escNextLine();
				nPosX = 0;
			}
			nPosY++;
		}
		if (nIndex != nLength) {
			layout();
		}
	}
}


static void evDelete(void)
{
	if ((nLength > 0) && (nIndex < nLength)) {
		nLength--;
		memmove(buff + nIndex, buff + nIndex + 1,
				sizeof(unsigned short) * (nLength - nIndex));
		layout();
	}
}

static int Utl_cmpWideStrWithUTF8(const RBTOS_WCHAR_T *ws, const char *s)
{
	unsigned phase = 0;
	unsigned uch = 0;
	#define UTF8_PROC		1
	#define UTF8_2BYTES_2ND	2
	#define UTF8_3BYTES_2ND	3
	#define UTF8_3BYTES_3RD	4
	while (1) {
		int df;
		unsigned wch;
		unsigned ch = *s++;
		if (phase == 0) {
			if ((ch & 0xE0) == 0xC0) {
				uch = (ch & 0x1F) << 6;
				phase = UTF8_2BYTES_2ND;
				continue;
			} else if ((ch & 0xF0) == 0xE0) {
				uch = (ch & 0x0F) << 12;
				phase = UTF8_3BYTES_2ND;
				continue;
			}
		} else if ((ch & 0xC0) == 0x80) {
			if (phase == UTF8_3BYTES_2ND) {
				uch |= ((ch & 0x3F) << 6);
				phase = UTF8_3BYTES_3RD;
				continue;
			} else {
				ch = uch | (ch & 0x3F);
				phase = 0;
			}
		}
		wch = *ws++;
		df = (int) wch - (int) ch;
		if (df != 0) {
			return df;
		}
		if (wch == 0) {
			return 0;
		}
	}
	#undef UTF8_PROC
	#undef UTF8_2BYTES_2ND
	#undef UTF8_3BYTES_2ND
	#undef UTF8_3BYTES_3RD
}



static void evEnter(void)
{
	int px, i, argc;
	RBTOS_WCHAR_T **argv;
	escNextLine();
	px = nPosX;
	for (i = nIndex; i < nLength; i++) {
		int cw = getcharsize(buff[i]);
		if ((px + cw) < DEFAULT_WIDTH) {
			px += cw;
		} else {
			px = ((px + cw) == DEFAULT_WIDTH) ? 0 : cw;
			escNextLine();
		}
	}
	
	argv = parseCmdLine(&argc);
	
	if (argc > 0) {
		for (i = 0; i < sizeof(CommandTable) / sizeof(CommandTable[0]); i++) {
			if (Utl_cmpWideStrWithUTF8(argv[0], CommandTable[i].cmdname) == 0) {
				int ret = CommandTable[i].func(argc, argv);
				if (ret != 0) {
					con_printf("\"%S\" was finished with return code %d\n", argv[0], ret);
				}
				break;
			}
		}
		if (i == (sizeof(CommandTable) / sizeof(CommandTable[0]))) {
			con_printf("\"%S\" is wrong command.\n", argv[0]);
		}
	}
	initLine();
}


static void evMoveLeft(void)
{
	if (nIndex > 0) {
		nIndex--;
		if (nPosX > 0) {
			int cw = getcharsize(buff[nIndex]);
			escMvLeft(cw);
			nPosX -= cw;
		} else {
			int i;
			int tx = PS1_LENGTH;
			for (i = 0; i < nIndex; i++) {
				int cw = getcharsize(buff[i]);
				if ((tx + cw) < DEFAULT_WIDTH) {
					tx += cw;
				} else {
					tx = ((tx + cw) == DEFAULT_WIDTH) ? 0 : cw;
				}
			}
			escMvUp(1);
			escMvRight(tx);
			nPosX = tx;
			nPosY--;
		}
	}
}


static void evBackSpace(void)
{
	if (nIndex > 0) {
		if ((nIndex == nLength) && (nPosX > 0)) {
			int cw;
			nIndex--;
			nLength--;
			cw = getcharsize(buff[nIndex]);
			nPosX -= cw;
			escMvLeft(cw);
			escClsToLineEnd();
		} else {
			evMoveLeft();
			evDelete();
		}
	}
}


static void evMoveRight(void)
{
	if (nIndex < nLength) {
		int cw = getcharsize(buff[nIndex]);
		nIndex++;
		if ((nPosX + cw) < DEFAULT_WIDTH) {
			escMvRight(cw);
			nPosX += cw;
			if (nPosX == (DEFAULT_WIDTH - 1)) {
				if (nIndex < nLength) {
					cw = getcharsize(buff[nIndex]);
					if (cw != 1) {
						escNextLine();
						nPosX = 0;
						nPosY++;
					}
				}
			}
		} else {
			escNextLine();
			nPosX = 0;
			nPosY++;
		}
	}
}


static void evMoveLineTop(void)
{
	escMvUp(nPosY);
	escMvLineTop();
	escMvRight(PS1_LENGTH);
	nPosX = PS1_LENGTH;
	nIndex = nPosY = 0;
}


static void evMoveLineEnd(void)
{
	unsigned mr = 0;
	for (; nIndex < nLength; nIndex++) {
		unsigned fNextLine = 0;
		int cw = getcharsize(buff[nIndex]);
		if ((nPosX + cw) >= DEFAULT_WIDTH) {
			fNextLine = 1;
		} else {
			mr += cw;
			nPosX += cw;
			if (nPosX == (DEFAULT_WIDTH - 1)) {
				if ((nIndex + 1) < nLength) {
					cw = getcharsize(buff[nIndex + 1]);
					if (cw > 1) {
						fNextLine = 1;
					}
				}
			}
		}
		if (fNextLine) {
			mr = nPosX = 0;
			nPosY++;
			escNextLine();
		}
	}
	escMvRight(mr);
}


static unsigned phase = 0;
static unsigned uch = 0;
static unsigned fIsPrevCR = 0;

void ozCmdConsoleTask(void)
{
	initLine();
	while (1) {
		unsigned ch = (unsigned) con_getchar();
		if (ch == '\r') {
			evEnter();
			fIsPrevCR = 1;
			phase = 0;
		} else {
			unsigned isPrevCR = fIsPrevCR;
			fIsPrevCR = 0;
			if (ch == '\n') {
				if (isPrevCR) {
					evEnter();
				}
				phase = 0;
			} else if (ch == 0x1B) { // ESC
				phase = 1;
			} else if (ch == 0x7F) { // DEL
				evDelete();
				phase = 0;
			} else if (ch == 0x08) { // BS
				evBackSpace();
				phase = 0;
			} else {
				
				#define UTF8_PROC		0x100
				#define UTF8_2BYTES_2ND	0x100
				#define UTF8_3BYTES_2ND	0x101
				#define UTF8_3BYTES_3RD	0x102
				
				if (phase >= UTF8_PROC) {
					if ((ch & 0xC0) != 0x80) {
						phase = 0;
					} else {
						if (phase == UTF8_3BYTES_2ND) {
							uch |= ((ch & 0x3F) << 6);
							phase = UTF8_3BYTES_3RD;
						} else {
							uch |= (ch & 0x3F);
							evChar(uch);
							phase = 0;
						}
						continue;
					}
				}
				
				if (phase == 0) {
					if ((ch >= ' ') && (ch <= 0x7E)) {
						evChar(ch);
					} else if ((ch & 0xE0) == 0xC0) {
						uch = (ch & 0x1F) << 6;
						phase = UTF8_2BYTES_2ND;
					} else if ((ch & 0xF0) == 0xE0) {
						uch = (ch & 0x0F) << 12;
						phase = UTF8_3BYTES_2ND;
					}
					
				} else if (phase == 1) {
					phase = ((ch == '[') ? '[' : 0);
					
				} else if (phase == 0x5B) {
					if ((ch >= '0') && (ch <= '9')) {
						phase = ch;
					} else {
						if (ch == 'D') { // LEFT
							evMoveLeft();
						} else if (ch == 'C') { // RIGHT
							evMoveRight();
						}
						phase = 0;
					}
					
				} else if ((phase >= '0') && (phase <= '9')) {
					if (ch == 0x7E) {
						if (phase == '4') {
							evMoveLineEnd();
						} else if (phase == '1') {
							evMoveLineTop();
						}
					}
					phase = 0;
				}
			}
		}
	}
}
