/*!
  \file
  \brief 設定ファイルの読み出しと保存管理

  \author Satofumi KAMIMURA

  $Id: GenerateConfigHandler.cpp 263 2009-01-30 14:48:10Z satofumi $
*/

#include "GenerateConfigHandler.h"
#include "PartsConfig.h"
#include <QStringList>
#include <QFile>
#include <QTextStream>

using namespace std;


struct GenerateConfigHandler::pImpl
{
  vector<PartsConfig> parts_configs_;
  string file_;


  bool load(const char* file)
  {
    QFile data(file);
    if (! data.open(QFile::ReadOnly)) {
      return false;
    }

    vector<PartsConfig> configs;

    // UTF-8 の文字列を扱いたいため、QTextStream を使う
    QTextStream fin(&data);
    fin.setCodec("UTF-8");
    PartsConfig current_parts;
    while (true) {
      // 単語毎に読み出しを行い、単語に応じた変数に情報を格納する
      // !!!

      QString line = fin.readLine();
      if (line.isNull()) {
        // !!! この判定条件を、ループ条件に変更したい
        break;
      }
      if ((line.size() <= 0) || (line[0] == '#')) {
        continue;
      }

      QStringList tokens = line.split("\t");

      size_t tokens_n = tokens.size();
      QString first_token = tokens.at(0);

      if (tokens_n == 1) {
        if (first_token.size() <= 0) {
          continue;
        }

        if (first_token[0] == '\\') {
          // \<name> 読み出し後は、それまでの情報を格納してから次の情報を扱う
          storePartsInformation(configs, current_parts);
          current_parts.type = first_token.mid(1).toStdString();
          setDefaultTags(current_parts);
        }
      } else if (tokens_n == 2) {
        QString second_token = tokens.at(1);

        if (! first_token.compare("font")) {
          current_parts.font_name = second_token.toStdString();

        } else if (! first_token.compare("size")) {
          current_parts.font_size = second_token.toULong();

        } else if (! first_token.compare("fore")) {
          current_parts.fore_color = second_token.toStdString();

        } else if (! first_token.compare("back")) {
          current_parts.back_color = second_token.toStdString();

        } else if (! first_token.compare("offset")) {
          current_parts.offset = second_token.toULong();
        }
      } else if (! first_token.compare("tag")) {
        // タグ文字の格納
        tokens.pop_front();
        current_parts.tags.clear();

        while (! tokens.isEmpty()) {
          current_parts.tags.push_back(tokens.front());
          tokens.pop_front();
        }
      }
    }

    // 最後に読み出していた情報を格納する
    storePartsInformation(configs, current_parts);
    swap(parts_configs_, configs);

    file_ = file;
    return true;
  }


  void storePartsInformation(vector<PartsConfig>& configs,
                             const PartsConfig& parts)
  {
    if (! parts.type.empty()) {
      configs.push_back(parts);
    }
  }


  void setDefaultTags(PartsConfig& parts)
  {
    string parts_type = parts.type;

    if (! parts_type.compare("segment")) {
      parts.tags.clear();

      pushbackNumbers(parts);
      parts.tags.push_back(" ");
      parts.tags.push_back(":");
      parts.tags.push_back("A");
      parts.tags.push_back("P");

    } else if (! parts_type.compare("number")) {
      parts.tags.clear();
      pushbackNumbers(parts);

    } else if (! parts_type.compare("week")) {
      parts.tags.clear();
      pushbackWeekTags(parts);

    } else if (! parts_type.compare("separator")) {
      parts.tags.clear();
      pushbackSeparatorTags(parts);
    }
  }


  void pushbackNumbers(PartsConfig& parts)
  {
    // !!! ループにすべき
    parts.tags.push_back("0");
    parts.tags.push_back("1");
    parts.tags.push_back("2");
    parts.tags.push_back("3");
    parts.tags.push_back("4");
    parts.tags.push_back("5");
    parts.tags.push_back("6");
    parts.tags.push_back("7");
    parts.tags.push_back("8");
    parts.tags.push_back("9");
  }

  void pushbackWeekTags(PartsConfig& parts)
  {
    parts.tags.push_back("Sun");
    parts.tags.push_back("Mon");
    parts.tags.push_back("Tue");
    parts.tags.push_back("Wed");
    parts.tags.push_back("Thu");
    parts.tags.push_back("Fri");
    parts.tags.push_back("Sat");
  }


  void pushbackSeparatorTags(PartsConfig& parts)
  {
    parts.tags.push_back("h");
    parts.tags.push_back("m");
  }


  bool saveAs(const char* file)
  {
    QFile data(file);
    if (! data.open(QFile::WriteOnly)) {
      return false;
    }
    QTextStream fout(&data);
    fout.setCodec("UTF-8");

    fout << "# Parts Generator config file" << endl
         << '#' << endl
         << endl;

    for (vector<PartsConfig>::iterator it = parts_configs_.begin();
         it != parts_configs_.end(); ++it) {

      fout << '\\' << it->type.c_str() << endl
           << "font\t" << it->font_name.c_str() << endl
           << "style\t" << it->font_style_name.c_str() << endl
           << "size\t" << it->font_size << endl
           << "fore\t" << it->fore_color.c_str() << endl
           << "back\t" << it->back_color.c_str() << endl
           << "offset\t" << it->offset << endl
           << "tag";

      for (vector<QString>::iterator tags_it = it->tags.begin();
           tags_it != it->tags.end(); ++tags_it) {
        fout << '\t' << *tags_it;
      }
      fout << endl
           << endl;
    }

    file_ = file;
    return true;
  }
};


GenerateConfigHandler::GenerateConfigHandler(void) : pimpl(new pImpl)
{
}


GenerateConfigHandler::~GenerateConfigHandler(void)
{
}


void GenerateConfigHandler::initialize(void)
{
  const char* tags[] = {
    "segment", "number", "week", "separator"
  };
  size_t n = sizeof(tags) / sizeof(tags[0]);

  vector<PartsConfig> configs;
  for (size_t i = 0; i < n; ++i) {
    PartsConfig current_parts;
    current_parts.type = tags[i];
    pimpl->setDefaultTags(current_parts);
    pimpl->storePartsInformation(configs, current_parts);
  }
  swap(pimpl->parts_configs_, configs);
}


size_t GenerateConfigHandler::size(void)
{
  return pimpl->parts_configs_.size();
}


PartsConfig& GenerateConfigHandler::getPartsConfig(size_t index)
{
  static PartsConfig dummy;

  if (index >= size()) {
    return dummy;
  }

  return pimpl->parts_configs_[index];
}


bool GenerateConfigHandler::load(const char* file)
{
  return pimpl->load(file);
}


bool GenerateConfigHandler::save(void)
{
  if (! pimpl->file_.empty()) {
    return saveAs(pimpl->file_.c_str());

  } else {
    return false;
  }
}


bool GenerateConfigHandler::saveAs(const char* file)
{
  return pimpl->saveAs(file);
}
