/*!
  \file
  \brief RrdaEncoder のテスト

  \author Satofumi KAMIMURA

  $Id: TestRrdaEncoder.cpp 261 2009-01-22 06:48:10Z satofumi $
*/

#include <algorithm>
#include "TestRrdaEncoder.h"
#include "RrdaEncoder.h"
#include "RrdaResource.h"

using namespace std;


CPPUNIT_TEST_SUITE_REGISTRATION(TestRrdaEncoder);
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(TestRrdaEncoder, "TestRrdaEncoder");


namespace {
  bool isExist(vector<string>& resources, const char* fileName) {

    return (find(resources.begin(), resources.end(),
                      fileName) != resources.end());
  }


  // !!! CppUnit っぽいメッセージが出力できるようにする
  bool cmp(auto_ptr<RrdaResource>& resource, const char* fileName) {

    ifstream fin(fileName);
    if (! fin.is_open()) {
      return false;
    }

    size_t actual_size = 0;
    char* actual_ch = reinterpret_cast<char*>(resource->get(&actual_size));

    size_t index = 0;

    while (! fin.eof()) {
      char expected_ch;
      fin.read(&expected_ch, 1);

      // テキストのデコード時に最後の１文字の改行を除去されていので、
      // 最後の１文字だけは比較対象から除外
      if (index != actual_size) {

        if (expected_ch != *actual_ch) {

          // !!! CPPUNIT_ASSERT_MESSAGE() を使うようにすべき
          fprintf(stderr, "[%d: %02x %02x]\n", index,
                  static_cast<unsigned char>(expected_ch),
                  static_cast<unsigned char>(*actual_ch));
          return false;
        }
      }

      // !!! for ループにすべきかも
      ++actual_ch;
      ++index;
    }

    return true;
  }
};


void TestRrdaEncoder::getResourceNamesTest(void) {

  RrdaEncoder encoder("../theme/lm_clocktheme_01.dat", "LMTC");
  vector<string> names;
  encoder.getResourceNames(names);
  CPPUNIT_ASSERT(! names.empty());

  // リソースがあるかのテスト
  CPPUNIT_ASSERT(isExist(names, "config"));
  CPPUNIT_ASSERT(isExist(names, "base"));
  CPPUNIT_ASSERT(isExist(names, "digital"));
  CPPUNIT_ASSERT(isExist(names, "hari_h"));
  CPPUNIT_ASSERT(isExist(names, "hari_m"));
  CPPUNIT_ASSERT(isExist(names, "hari_s"));
}


void TestRrdaEncoder::getResourceMemoryTest(void) {

  RrdaEncoder encoder("../theme/lm_clocktheme_01.dat", "LMTC");

  // リソースが適切に返されるかのテスト
  auto_ptr<RrdaResource> config(encoder.getResource("config"));
  CPPUNIT_ASSERT(cmp(config, "../theme/lm_tc.xml"));

  auto_ptr<RrdaResource> base(encoder.getResource("base"));
  CPPUNIT_ASSERT(cmp(base, "../theme/clock_base.png"));

  auto_ptr<RrdaResource> digital(encoder.getResource("digital"));
  CPPUNIT_ASSERT(cmp(digital, "../theme/segment.png"));

  auto_ptr<RrdaResource> hari_h(encoder.getResource("hari_h"));
  CPPUNIT_ASSERT(cmp(hari_h, "../theme/hand_h.png"));

  auto_ptr<RrdaResource> hari_m(encoder.getResource("hari_m"));
  CPPUNIT_ASSERT(cmp(hari_m, "../theme/hand_m.png"));

  auto_ptr<RrdaResource> hari_s(encoder.getResource("hari_s"));
  CPPUNIT_ASSERT(cmp(hari_s, "../theme/hand_s.png"));
}
