/*!
  \file
  \brief RRDA リソースのエンコード

  \author Satofumi KAMIMURA

  $Id: RrdaEncoder.cpp 262 2009-01-30 13:25:37Z satofumi $

  \todo 制限エリアに対する操作を実装する
  \todo エラーメッセージの処理
  \todo 全体サイズを取得して、バッファオーバーランを防止させる
*/

#include "RrdaEncoder.h"
#include "RrdaResource.h"
#include <QString>
#include <cerrno>
#include <cstdlib>
#include <map>
#include <fstream>


using namespace std;


namespace
{
  typedef struct
  {
    unsigned long start;
    unsigned long size;
  } area_t;

  typedef vector<string> Strings;
  typedef map<string, area_t> AreaMap;
}


struct RrdaEncoder::pImpl
{
  string error_message_;
  string file_name_;
  string key_;
  AreaMap resource_files_;


  pImpl(const char* fileName, const char* key)
    : error_message_("no error."), file_name_(fileName), key_(key)
  {
    // リソース名の一覧を取得
    getResourceNames();
  }


  // リソース名の一覧を取得
  void getResourceNames(void)
  {
    if (file_name_.empty()) {
      // ファイルが指定されていない
      error_message_ = "no resource file name.";
      return;
    }

    ifstream fin(file_name_.c_str(), fstream::binary);
    if (! fin.is_open()) {
      error_message_ = strerror(errno);
      return;
    }
    resource_files_.clear();

    // フリーエリアからの探索
    area_t next_area;
    readDirectory(fin, &next_area);

    // 制限エリアからの探索
    // !!! fin の seek 位置を変更
    // !!! seekg(next_area.start, iso::beg);
    // !!! readDirectory(fin, area);
  }


  // ファイル内の構成を読み出す
  bool readDirectory(ifstream& fin, area_t* next_area,
                     bool encrypted = false)
  {
    if (encrypted) {
      // 復号化
      // !!!
    }

    string line;
    for (size_t i = 0; getline(fin, line); ++i) {

      if (i < 3) {
        // ３行目までは、専用のパース処理を行う
        if (i == 0) {
          // RRDA のチェック
          if (line.compare(0, 4, "RRDA")) {
            error_message_ = "No RRDA file.";
            return false;
          }

        } else if (i == 1) {
          // 全体のサイズは読み捨て
          // !!! チェックすべきだが、保留
          // return false;

        } else if (i == 2) {
          if (next_area) {
            // 次のディレクトリ範囲の取得
            parseArea(*next_area, line);
          }
        }
        continue;
      }

      // "END" を受信したら終了
      if (! line.compare("END")) {
        break;
      }

      // データ位置の追加
      QString csv_line = line.c_str();
      QString name = csv_line.section(',', 2, 2);

      area_t area;
      parseArea(area, line);

      resource_files_.
        insert(pair<string, area_t>(name.toStdString(), area));
    }

    return true;
  }


  // 領域情報の取得
  void parseArea(area_t& area, const string& line)
  {
    QString csv_line = line.c_str();
    string start_hex = csv_line.section(',', 0, 0).toStdString();
    string size_hex = csv_line.section(',', 1, 1).toStdString();
    area.start = strtol(start_hex.c_str(), NULL, 16);
    area.size = strtol(size_hex.c_str(), NULL, 16);
  }
};


RrdaEncoder::RrdaEncoder(const char* fileName, const char* key)
  : pimpl(new pImpl(fileName, key))
{
}


RrdaEncoder::~RrdaEncoder(void)
{
}


// エラー文字列を返す
const char* RrdaEncoder::what(void) const
{
  return pimpl->error_message_.c_str();
}


// リソース名の一覧を返す
void RrdaEncoder::getResourceNames(vector<string>& names)
{
  for (AreaMap::const_iterator it = pimpl->resource_files_.begin();
       it != pimpl->resource_files_.end(); ++it) {
    names.push_back(it->first);
  }
}


// リソースを RrdaResource でラップして返す
RrdaResource* RrdaEncoder::getResource(const char* resource_name)
{
  ifstream fin(pimpl->file_name_.c_str(), fstream::binary);
  if (! fin.is_open()) {
    // ファイルが開けない
    pimpl->error_message_ = strerror(errno);
    return new RrdaResource(fin, 0);
  }

  AreaMap::const_iterator p = pimpl->resource_files_.find(resource_name);
  if (p == pimpl->resource_files_.end()) {
    // キーワードが登録されていない
    pimpl->error_message_ =
      string("unknown resource name: ") + resource_name;
    return new RrdaResource(fin, 0);
  }
  const area_t& area = p->second;
  fin.seekg(area.start, ios::beg);

  return new RrdaResource(fin, area.size);
}
