/* HuffmanDecoder.c */
/* 2008/11/05       */

#include "StdAfx.h"

#include "HuffmanDecoder.h"

/* */

static INT32 lookup1_values(INT32 entries, INT32 dim)
{
	INT32 r = (INT32)floor(exp(log(entries) / dim));
	if ((INT32)floor(pow(r + 1, dim)) <= entries) {
		r += 1;
	}
	return r;
}

static FLOAT float32_unpack(UINT32 x)
{
	UINT32 mantissa =  x & 0x001fffff;
	UINT32 exp      = (x & 0x7fe00000) >> 21;
	UINT32 sign     =  x & 0x80000000;
	double res = (sign != 0) ? -(double)mantissa : (double)mantissa;
	return (FLOAT)ldexp(res, exp - 788);
}

/* */

static BOOL HuffmanDeocder_SetupLookup1(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	HuffmanDecoder_t* huff,
	const INT8*       length,
	INT32             length_size,
	INT32             count);

static BOOL HuffmanDeocder_SetupDecoder(
	DecoderContext_t* ctx,
	HuffmanDecoder_t* huff,
	const INT8*       length,
	INT32             length_size,
	INT32             count);

static BOOL HuffmanDeocder_SetupCodeLookup(
	DecoderContext_t* ctx,
	HuffmanDecoder_t* huff,
	INT32             count);

/* */

BOOL QV_HuffmanDeocder_Setup(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	HuffmanDecoder_t* huff)
{
	BOOL result = FALSE;

	MemoryPool_t* pool = ctx->Pool;

	UINT32 code;

	INT32 i;
	INT32 entries;
	INT32 ordered;

	INT8* code_length;

	INT32 code_count;

	INT32 map_type;

	VOID* top = QV_MemoryPool_EnterBackyard(pool);

	code = QV_BitDecoder_GetBits(d, 24);
	if (code != 0x564342) {
		THROW0(ctx, "HuffmanDeocder_Setup", "OutOfSyncHeader")
	}

	huff->Dimensions = QV_BitDecoder_GetBits(d, 16);

	entries = QV_BitDecoder_GetBits(d, 24);
	if (entries == 0) {
		THROW0(ctx, "HuffmanDeocder_Setup", "Invalid.Entries")
	}

	huff->Entries = entries;

	if (ilog(huff->Dimensions) + ilog(entries) > 24) {
		THROW0(ctx, "HuffmanDeocder_Setup", "CVE-2008-1423")
	}

	code_length = (INT8*)QV_MemoryPool_AllocateBackyard(pool, entries);
	if (code_length == NULL) {
		THROW0(ctx, "HuffmanDeocder_Setup", "OutOfMemory")
	}

	ordered = QV_BitDecoder_GetBits(d, 1);
	if (ordered == 0) {
		INT32 sparse = QV_BitDecoder_GetBits(d, 1);
		if (sparse != 0) {
			code_count = 0;

			for (i = 0; i < entries; i++) {
				INT32 f = QV_BitDecoder_GetBits(d, 1);
				if (f != 0) {
					code_length[i] = QV_BitDecoder_GetBits(d, 5) + 1;
					code_count += 1;
				} else {
					code_length[i] = 0;
				}
			}

		} else {
			for (i = 0; i < entries; i++) {
				code_length[i] = QV_BitDecoder_GetBits(d, 5) + 1;
			}

			code_count = entries;
		}

	} else {
		INT32 current = 0;
		INT8  length  = QV_BitDecoder_GetBits(d, 5) + 1;
		while (current < entries) {
			INT32 limit = entries - current;
			INT32 n = QV_BitDecoder_GetBits(d, ilog(limit));
			if (current + n > entries) {
				THROW0(ctx, "HuffmanDeocder_Setup", "OutOfEntries")
			}
			for (i = 0; i < n; i++) {
				code_length[current + i] = length;
			}
			current += n;
			length  += 1;
		}

		code_count = entries;
	}

	map_type = QV_BitDecoder_GetBits(d, 4);
	if (map_type != 0 && map_type != 1) {
		THROW0(ctx, "HuffmanDeocder_Setup", "InvalidMapType")
	}

	if (map_type == 1) {
		if (!HuffmanDeocder_SetupLookup1(ctx, d, huff, code_length, entries, code_count)) {
			goto ErrorEnd;
		}

		if (entries != code_count) {
			INT8* dd;

			INT8* d_len = (INT8*)QV_MemoryPool_AllocateBackyard(pool, code_count);
			if (d_len == NULL) {
				THROW0(ctx, "HuffmanDeocder_Setup", "OutOfMemory")
			}

			dd = d_len;
			for (i = 0; i < entries; i++) {
				if (code_length[i] > 0) {
					*(dd++) = code_length[i];
				}
			}

			code_length = d_len;
			entries     = code_count;
		}
	}

	if (!HuffmanDeocder_SetupDecoder(ctx, huff, code_length, entries, code_count)) {
		goto ErrorEnd;
	}

	if (!HuffmanDeocder_SetupCodeLookup(ctx, huff, code_count)) {
		goto ErrorEnd;
	}

	huff->IndexCount = code_count;

	result = TRUE;

ErrorEnd:

	QV_MemoryPool_LeaveBackyard(pool, top);

	return result;
}

/* */

BOOL HuffmanDeocder_SetupLookup1(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	HuffmanDecoder_t* huff,
	const INT8*       length,
	INT32             length_size,
	INT32             count)
{
	BOOL result = FALSE;

	MemoryPool_t* pool = ctx->Pool;

	INT32 i;

	FLOAT min, delta;
	INT32 quant, sequence;
	INT32 q_length;

	INT32* values;

	FLOAT* p;

	VOID* top = QV_MemoryPool_EnterBackyard(pool);

	min      = float32_unpack(QV_BitDecoder_GetBits(d, 32));
	delta    = float32_unpack(QV_BitDecoder_GetBits(d, 32));
	quant    = QV_BitDecoder_GetBits(d, 4) + 1;
	sequence = QV_BitDecoder_GetBits(d, 1);

	if (huff->Dimensions == 0) {
		THROW0(ctx, "HuffmanDeocder_SetupLookup1", "CVE-2008-1419")
	}

	q_length = lookup1_values(length_size, huff->Dimensions);

	values = (INT32*)QV_MemoryPool_AllocateBackyard(pool, sizeof(INT32) * q_length);
	if (values == NULL) {
		THROW0(ctx, "HuffmanDeocder_SetupLookup1", "OutOfMemory")
	}

	for (i = 0; i < q_length; i++) {
		values[i] = QV_BitDecoder_GetBits(d, quant);
	}

	huff->Lookup = (FLOAT*)QV_MemoryPool_Allocate(pool, sizeof(FLOAT) * huff->Dimensions * count);
	if (huff->Lookup == NULL) {
		THROW0(ctx, "HuffmanDeocder_SetupLookup1", "OutOfMemory")
	}

	p = huff->Lookup;

	for (i = 0; i < length_size; i++) {
		if (length[i] > 0) {
			FLOAT last = 0;
			INT32 offset = i;

			INT32 j;
			for (j = 0; j < huff->Dimensions; j++) {
				FLOAT v = values[offset % q_length] * delta + min + last;
				if (sequence != 0) {
					last = v;
				}
				offset /= q_length;

				*(p++) = v;
			}
		}
	}

	result = TRUE;

ErrorEnd:

	QV_MemoryPool_LeaveBackyard(pool, top);

	return result;
}

/* */

typedef struct DecoderEntry {
	UINT32 Code;
	UINT16 Index;
	UINT16 Length;
} DecoderEntry_t;

static int compare(const void* a, const void* b)
{
	const DecoderEntry_t* aa = (const DecoderEntry_t*)a;
	const DecoderEntry_t* bb = (const DecoderEntry_t*)b;

	if (aa->Length < bb->Length) {
		return -1;
	}
	if (aa->Length > bb->Length) {
		return 1;
	}

	if (aa->Code < bb->Code) {
		return -1;
	}
	if (aa->Code > bb->Code) {
		return 1;
	}

	return 0;
}

BOOL HuffmanDeocder_SetupDecoder(
	DecoderContext_t* ctx,
	HuffmanDecoder_t* huff,
	const INT8*       length,
	INT32             length_size,
	INT32             count)
{
	BOOL result = FALSE;

	MemoryPool_t* pool = ctx->Pool;

	DecoderEntry_t* entries;
	DecoderEntry_t* p;
	DecoderEntry_t* end;

	INT32 i, j;
	INT32 start;

	INT32 c_len;

	UINT32 code[32] = { 0 };

	VOID* top = QV_MemoryPool_EnterBackyard(pool);

	entries = (DecoderEntry_t*)QV_MemoryPool_AllocateBackyard(
		pool,
		count * sizeof(DecoderEntry_t));
	if (entries == NULL) {
		THROW0(ctx, "HuffmanDeocder_SetupDecoder", "OutOfMemory")
	}

	end = entries + count;

	for (start = 0; start < length_size; start++) {
		if (length[start] != 0) {
			break;
		}
	}

	if (start >= length_size) {
		THROW0(ctx, "HuffmanDeocder_SetupDecoder", "EmptyCodeBook")
	}

	p = entries;

	p->Code   = 0;
	p->Index  = start;
	p->Length = length[start];

	p += 1;

	for (i = 0; i < length[start]; i++) {
		code[i] = 1 << i;
	}

	for (i = start + 1; i < length_size; i++) {
		UINT32 bit;

		INT32 len = length[i];
		if (len == 0) {
			continue;
		}

		for (; len > 0; len--) {
			if (code[len - 1] != 0) {
				break;
			}
		}

		if (len == 0) {
			THROW0(ctx, "HuffmanDeocder_SetupDecoder", "InvalidCodeBook")
		}

		bit = code[len - 1];

		code[len - 1] = 0;

		p->Code   = bit;
		p->Index  = i;
		p->Length = length[i];

		p += 1;

		for (j = len + 1; j <= length[i]; j++) {
			code[j - 1] = bit | (1 << (j - 1));
		}
	}

	for (i = 0; i < 32; i++) {
		if (code[i] != 0) {
			THROW0(ctx, "HuffmanDeocder_SetupDecoder", "IncompleteCodeBook.CVE-2008-2009")
		}
	}

	qsort(
		entries,
		count,
		sizeof(DecoderEntry_t),
		compare);

	huff->CodeWord = (UINT32*)QV_MemoryPool_Allocate(pool, sizeof(UINT32) * count);
	if (huff->CodeWord == NULL) {
		THROW0(ctx, "HuffmanDeocder_SetupDecoder", "OutOfMemory")
	}

	huff->Index = (UINT16*)QV_MemoryPool_Allocate(pool, sizeof(UINT16) * count);
	if (huff->Index == NULL) {
		THROW0(ctx, "HuffmanDeocder_SetupDecoder", "OutOfMemory")
	}

	p = entries;
	i = 0;
	j = 0;

	for (c_len = 1; c_len <= 32; c_len++) {
		INT32 e_count = 0;

		while (p < end) {
			if (p->Length == c_len) {
				huff->CodeWord[i] = p->Code;
				huff->Index   [i] = p->Index;
				i += 1;

				p += 1;

				e_count += 1;

			} else {
				break;
			}
		}

		if (e_count > 0) {
			huff->Length   [j] = c_len;
			huff->CodeCount[j] = e_count;
			j += 1;
		}
	}

	for (; j < 32; j++) {
		huff->Length   [j] = 0;
		huff->CodeCount[j] = 0;
	}

	result = TRUE;

ErrorEnd:

	QV_MemoryPool_LeaveBackyard(pool, top);

	return result;
}

/* */

static INT32 HuffmanDecoder_LookupTable(
	const HuffmanDecoder_t* h,
	BitDecoder_t*           d);

static INT32 HuffmanDecoder_LookupSearch(
	const HuffmanDecoder_t* h,
	BitDecoder_t*           d);

#define LOOKUP_BITS 12

BOOL HuffmanDeocder_SetupCodeLookup(
	DecoderContext_t* ctx,
	HuffmanDecoder_t* huff,
	INT32             count)
{
	extern BOOL g_Enable_SSE2;

	BOOL result = FALSE;

	MemoryPool_t* pool = ctx->Pool;

	INT32 i;
	INT32 n;

	INT32 start;

	INT32 max_bits = 0;

	if (count >= (1 << 12)) {
		huff->LookupFunc = HuffmanDecoder_LookupSearch;

		result = TRUE;
		goto ErrorEnd;
	}

	for (i = 0; i < 32; i++) {
		if (max_bits < huff->Length[i]) {
			max_bits = huff->Length[i];
		}
	}

	huff->LookupBits = LOOKUP_BITS;
	if (huff->LookupBits > max_bits) {
		huff->LookupBits = max_bits;
	}

	n = 1 << huff->LookupBits;

	huff->CodeLookup = (UINT16*)QV_MemoryPool_Allocate(pool, sizeof(UINT16) * n);
	if (huff->CodeLookup == NULL) {
		THROW0(ctx, "HuffmanDeocder_SetupCodeLookup", "OutOfMemory")
	}

	memset(huff->CodeLookup, 0xff, sizeof(UINT16) * n);

	huff->NextCode  = 0;
	huff->NextStart = 0;

	start = 0;

	for (i = 0; i < 32; i++) {
		INT32 bits = huff->Length[i];
		if (bits == 0) {
			break;
		}

		if (bits <= huff->LookupBits) {
			INT32 m = huff->CodeCount[i];

			UINT32* p = huff->CodeWord + start;
			UINT32* e = p + m;

			UINT16* idx = huff->Index + start;

			INT32 rbits = huff->LookupBits - bits;
			INT32 rlen  = 1 << rbits;

			while (p < e) {
				UINT32 code  = *(p  ++);
				UINT16 index = *(idx++);

				UINT16 x = (UINT16)((bits << 12) | (index & 0x0fff));

				INT32 j;
				for (j = 0; j < rlen; j++) {
					UINT32 l = (j << bits) | code;
					huff->CodeLookup[l] = x;
				}
			}

			start += m;

		} else {
			huff->NextCode  = i;
			huff->NextStart = start;
			break;
		}
	}

	if (g_Enable_SSE2) {
		huff->LookupFunc = QV_HuffmanDecoder_LookupTable_SSE2;
	} else {
		huff->LookupFunc = HuffmanDecoder_LookupTable;
	}

	result = TRUE;

ErrorEnd:

	return result;
}

/* */

INT32 HuffmanDecoder_LookupTable(
	const HuffmanDecoder_t* h,
	BitDecoder_t*           d)
{
	UINT32 code = QV_BitDecoder_PeekBits(d, h->LookupBits);

	UINT16 x = h->CodeLookup[code];
	if (x != 0xffff) {
		INT32 bits = x >> 12;
		QV_BitDecoder_SkipBits(d, bits);
		return x & 0x0fff;

	} else {
		INT32 i;

		UINT32* s;
		UINT32* e;

		INT32 start = h->NextStart;
		for (i = h->NextCode; i < 32; i++) {
			INT32 count = h->CodeCount[i];

			INT32 len = h->Length[i];
			if (len == 0) {
				break;
			}

			s = h->CodeWord + start;
			e = s + count;

			code = QV_BitDecoder_PeekBits(d, len);

			while (s < e) {
				UINT32* p = s + (e - s) / 2;
				if (code == *p) {
					QV_BitDecoder_SkipBits(d, len);
					return h->Index[p - h->CodeWord];
				}

				if (code < *p) {
					e = p;
				} else {
					s = p + 1;
				}
			}

			start += count;
		}
	}

	return -1;
}

/* */

INT32 HuffmanDecoder_LookupSearch(
	const HuffmanDecoder_t* h,
	BitDecoder_t*           d)
{
	INT32 i;

	UINT32* s;
	UINT32* e;

	UINT32 code;

	INT32 start = 0;
	for (i = 0; i < 32; i++) {
		INT32 count = h->CodeCount[i];

		INT32 len = h->Length[i];
		if (len == 0) {
			break;
		}

		s = h->CodeWord + start;
		e = s + count;

		code = QV_BitDecoder_PeekBits(d, len);

		while (s < e) {
			UINT32* p = s + (e - s) / 2;
			if (code == *p) {
				QV_BitDecoder_SkipBits(d, len);
				return h->Index[p - h->CodeWord];
			}

			if (code < *p) {
				e = p;
			} else {
				s = p + 1;
			}
		}

		start += count;
	}

	return -1;
}

/* */

