/* TheoraDecoder.c */
/* 2009/06/26      */

#include "StdAfx.h"

#include "TheoraDecoder.h"

#include "SetupDecoder.h"

#include "FrameDecoder.h"

#include "CSConverter.h"

/* */

BOOL g_QT_Available_MMX  = FALSE;
BOOL g_QT_Available_SSE2 = FALSE;

BOOL g_QT_Enable_X86  = TRUE;
BOOL g_QT_Enable_MMX  = FALSE;
BOOL g_QT_Enable_SSE2 = FALSE;

BOOL QT_Initialize(void)
{
	INT32 info[4] = { 0 };

	__cpuid(info, 1);

	if ((info[3] & (1 << 23)) != 0) {
		g_QT_Available_MMX = TRUE;
		g_QT_Enable_MMX    = TRUE;
	}

	if ((info[3] & (1 << 26)) != 0) {
		g_QT_Available_SSE2 = TRUE;
		g_QT_Enable_SSE2    = TRUE;
	}

	return TRUE;
}

BOOL QT_SetEnableX86(BOOL b)
{
	BOOL e = g_QT_Enable_X86;

	g_QT_Enable_X86 = b;

	return e;
}

BOOL QT_SetEnableMMX(BOOL b)
{
	BOOL e = g_QT_Enable_MMX;

	g_QT_Enable_MMX = b;

	return e;
}

BOOL QT_SetEnableSSE2(BOOL b)
{
	BOOL e = g_QT_Enable_SSE2;

	g_QT_Enable_SSE2 = b;

	return e;
}

/* */

struct QTheoraDecoderSetup {

	MemoryPool_t Pool;

	IdentifyHeader_t Id;

	SetupHeader_t Setup;

	BlockIndex_t Index;

}; /* QTheoraDecoderSetup */

/* */

QTheoraDecoderSetup_t* QT_CreateDecoderSetup(void)
{
	QTheoraDecoderSetup_t* t = (QTheoraDecoderSetup_t*)malloc(sizeof(QTheoraDecoderSetup_t));
	if (t == NULL) {
		return NULL;
	}

	memset(t, 0, sizeof(QTheoraDecoderSetup_t));

	QT_MemoryPool_Init(&(t->Pool));

	return t;
}

void QT_ReleaseDecoderSetup(QTheoraDecoderSetup_t* setup)
{
	if (setup != NULL) {
		QT_MemoryPool_Release(&(setup->Pool));

		free(setup);
	}
}

BOOL QT_SetupDecoderSetupLacing(
	QTheoraDecoderSetup_t* setup,
	const VOID*            packet,
	SIZE_T                 size)
{
	extern BOOL g_QT_Enable_X86;
	extern BOOL g_QT_Enable_MMX;
	extern BOOL g_QT_Enable_SSE2;

	if (g_QT_Enable_SSE2 || g_QT_Enable_MMX) {
		ALIGN(0x10) BitReader_MMX_t d;
		QT_BitReader_MMX_Initialize(&d, packet, size);

		if (!QT_SetupDecoderSetupLacing_MMX(
			&(setup->Id),
			&(setup->Setup),
			&(setup->Pool),
			&d)) {
			return FALSE;
		}

		_mm_empty();

	} else if (g_QT_Enable_X86) {
		ALIGN(0x10) BitReader_X86_t d;
		QT_BitReader_X86_Initialize(&d, packet, size);

		if (!QT_SetupDecoderSetupLacing_X86(
			&(setup->Id),
			&(setup->Setup),
			&(setup->Pool),
			&d)) {
			return FALSE;
		}

	} else {
		BitReader_C_t d;
		QT_BitReader_C_Initialize(&d, packet, size);

		if (!QT_SetupDecoderSetupLacing_C(
			&(setup->Id),
			&(setup->Setup),
			&(setup->Pool),
			&d)) {
			return FALSE;
		}
	}

	if (!BlockIndex_Setup(
		&(setup->Index),
		&(setup->Pool),
		setup->Id.MX,
		setup->Id.MY)) {
		return FALSE;
	}

	return TRUE;
}

BOOL QT_GetFormatSetup(
	QTheoraDecoderSetup_t* setup,
	QT_Format_t*           format)
{
	if (setup == NULL || format == NULL) {
		return FALSE;
	}

	memset(format, 0, sizeof(QT_Format_t));

	format->CX = setup->Id.MX * 16;
	format->CY = setup->Id.MY * 16;

	format->PX = setup->Id.PDX;
	format->PY = setup->Id.PDY;
	format->PW = setup->Id.PCX;
	format->PH = setup->Id.PCY;

	/* bottom up */
	format->PY = format->CY - format->PH - format->PY;

	format->FRN = setup->Id.FRN;
	format->FRD = setup->Id.FRD;

	format->PRN = setup->Id.PRN;
	format->PRD = setup->Id.PRD;

	return TRUE;
}

/* */

struct QTheoraDecoder {

	MemoryPool_t Pool;

	FrameDecoder_t Decoder;

}; /* QTheoraDecoder */

/* */

QTheoraDecoder_t* QT_CreateDecoder(void)
{
	QTheoraDecoder_t* t = (QTheoraDecoder_t*)malloc(sizeof(QTheoraDecoder_t));
	if (t == NULL) {
		return NULL;
	}

	memset(t, 0, sizeof(QTheoraDecoder_t));

	QT_MemoryPool_Init(&(t->Pool));

	return t;
}

void QT_ReleaseDecoder(QTheoraDecoder_t* d)
{
	if (d != NULL) {
		QT_MemoryPool_Release(&(d->Pool));

		free(d);
	}
}

BOOL QT_SetupDecoder(
	QTheoraDecoder_t*      d,
	QTheoraDecoderSetup_t* setup)
{
	if (d == NULL || setup == NULL) {
		return FALSE;
	}

	if (!QT_FrameDecoder_Setup(
		&(d->Decoder),
		&(setup->Index),
		&(setup->Setup),
		&(d->Pool))) {
		return FALSE;
	}

	return TRUE;
}

BOOL QT_DecodeFrame(
	QTheoraDecoder_t* d,
	const VOID*       packet,
	SIZE_T            size,
	QT_Output_t*      output)
{
	if (d == NULL || packet == NULL || output == NULL) {
		return FALSE;
	}

	if (!QT_FrameDecoder_DecodeFrame(
		&(d->Decoder),
		packet,
		size)) {
		return FALSE;
	}

	{
		Plane_t* f = d->Decoder.Frame[1];

		output->Plane[0] = f[0].Plane;
		output->Plane[1] = f[1].Plane;
		output->Plane[2] = f[2].Plane;

		output->CX = f->CX;
		output->CY = f->CY;
	}

	return TRUE;
}

/* */

struct QTheoraCSC {

	VOID (*Convert)(const QT_Output_t*, QT_Frame_t*);

}; /* QTheoraCSC */

/* */

QTheoraCSC_t* QT_CreateCSC(void)
{
	QTheoraCSC_t* t = (QTheoraCSC_t*)malloc(sizeof(QTheoraCSC_t));
	if (t == NULL) {
		return NULL;
	}

	memset(t, 0, sizeof(QTheoraCSC_t));

	return t;
}

void QT_ReleaseCSC(QTheoraCSC_t* t)
{
	if (t != NULL) {
		free(t);
	}
}

BOOL QT_SetupCSC(
	QTheoraCSC_t* t,
	INT32         cs)
{
	extern BOOL g_QT_Enable_SSE2;
	extern BOOL g_QT_Enable_MMX;

	switch (cs) {
	case QTCS_YV12:
		if (g_QT_Enable_SSE2) {
			t->Convert = QT_CSConvert_YV12_SSE2;
		} else if (g_QT_Enable_MMX) {
			t->Convert = QT_CSConvert_YV12_MMX;
		} else {
			t->Convert = QT_CSConvert_YV12;
		}

		break;

	case QTCS_YUY2:
		if (g_QT_Enable_SSE2) {
			t->Convert = QT_CSConvert_YUY2_SSE2;
		} else if (g_QT_Enable_MMX) {
			t->Convert = QT_CSConvert_YUY2_MMX;
		} else {
			t->Convert = QT_CSConvert_YUY2;
		}

		break;

	default:
		return FALSE;
	}

	return TRUE;
}

BOOL QT_ConvertFrame(
	QTheoraCSC_t*      t,
	const QT_Output_t* output,
	QT_Frame_t*        frame)
{
	if (frame->X + frame->CX > output->CX ||
		frame->Y + frame->CY > output->CY) {
		return FALSE;
	}

	t->Convert(output, frame);

	return TRUE;
}

/* */

