/* BlockLacing.c */
/* 2009/05/27    */

#include "StdAfx.h"

#include "BlockLacing.h"

/* */

BOOL BlockLacing_Create(
	BlockLacing_t*  t,
	QM_Allocator_t* allocator,
	INT32           size)
{
	if (t == NULL) {
		return FALSE;
	}

	memset(t, 0, sizeof(BlockLacing_t));

	if (allocator == NULL) {
		return FALSE;
	}

	t->Allocator = allocator;

	t->LenSize = 16;
	t->Length  = (INT32*)allocator->Allocate(allocator->Context, t->LenSize * sizeof(INT32));
	if (t->Length == NULL) {
		return FALSE;
	}

	t->Size   = size;
	t->Buffer = (UINT8*)allocator->Allocate(allocator->Context, t->Size);
	if (t->Buffer == NULL) {
		return FALSE;
	}

	return TRUE;
}

void BlockLacing_Release(
	BlockLacing_t* t)
{
	if (t != NULL) {
		t->Allocator->Free(t->Allocator->Context, t->Length);
		t->Allocator->Free(t->Allocator->Context, t->Buffer);
	}
}

/* */

static const INT8 HLEN[16] = {
	-1,
	3,2,2,1,1,1,1,
	0,0,0,0,0,0,0,0
};

static BOOL DecodeFSize(
	const UINT8* start,
	const UINT8* end,
	INT32*       s,
	INT32*       len)
{
	INT32 v, l;

	const UINT8* p = start;

	UINT8 by;

	if (p >= end) {
		return FALSE;
	}

	by = *(p++);

	l = HLEN[by >> 4];
	if (l < 0) { /* too long */
		return FALSE;
	}

	v = by & ((1 << (7 - l)) - 1);

	while (l-- > 0) {
		if (p >= end) {
			return FALSE;
		}

		v = (v << 8) | *(p++);
	}

	*s   = v;
	*len = p - start;

	return TRUE;
}

static BOOL DecodeNSize(
	const UINT8* start,
	const UINT8* end,
	INT32*       s,
	INT32*       len)
{
	INT32 v, l, offs;

	const UINT8* p = start;

	UINT8 by;

	if (p >= end) {
		return FALSE;
	}

	by = *(p++);

	l = HLEN[by >> 4];
	if (l < 0) { /* too long */
		return FALSE;
	}

	offs = (1 << (6 + l * 7)) - 1;

	v = by & ((1 << (7 - l)) - 1);

	while (l-- > 0) {
		if (p >= end) {
			return FALSE;
		}

		v = (v << 8) | *(p++);
	}

	*s  += v - offs;
	*len = p - start;

	return TRUE;
}

/* */

static BOOL ResetLength(
	BlockLacing_t* t,
	INT32          count)
{
	INT32* p;

	INT32 s = count + 1;
	if (s <= t->LenSize) {
		return TRUE;
	}

	while (t->LenSize < s) {
		t->LenSize *= 2;
	}

	p = (INT32*)t->Allocator->Reallocate(t->Allocator->Context, t->Length, t->LenSize * sizeof(INT32));
	if (p == NULL) {
		return FALSE;
	}

	t->Length = p;

	return TRUE;
}

BOOL BlockLacing_Decode(
	BlockLacing_t* t,
	const VOID*    payload,
	INT32          size,
	INT32          flag)
{
	INT32 mode = (flag >> 1) & 0x3;

	const UINT8* p   = (const UINT8*)payload;
	const UINT8* end = p + size;

	if (t == NULL) {
		return FALSE;
	}

	t->Count = 0;

	switch (mode) {
	case 0: /* no lacing */
		t->Length[(t->Count)++] = size;
		break;

	case 1: /* Xiph lacing */
	{
		INT32 i, c;
		INT32 sz, ts, fs;

		if (p >= end) {
			return FALSE;
		}

		c = *(p++);
		if (!ResetLength(t, c)) {
			return FALSE;
		}

		sz = 0;
		for (i = 0; i < c; i++) {
			fs = 0;
			for (; ; ) {
				UINT8 by;

				if (p >= end) {
					return FALSE;
				}

				by = *(p++);

				fs += by;

				if (by != 0xff) {
					break;
				}
			}

			t->Length[(t->Count)++] = fs;

			sz += fs;
		}

		ts = (INT32)(end - p);
		fs = ts - sz;

		if (fs < 0) {
			return FALSE;
		}

		t->Length[(t->Count)++] = fs;

		break;
	}

	case 2: /* fixed-size lacing */
	{
		INT32 i, c;
		INT32 ts, fs;

		if (p >= end) {
			return FALSE;
		}

		c = *(p++);
		if (!ResetLength(t, c)) {
			return FALSE;
		}

		ts = (INT32)(end - p);
		fs = ts / (c + 1);

		if (ts != fs * (c + 1)) {
			return FALSE;
		}

		for (i = 0; i <= c; i++) {
			t->Length[(t->Count)++] = fs;
		}

		break;
	}

	case 3: /* EBML lacing */
	{
		INT32 i, c;
		INT32 sz, ts, fs = 0;

		if (p >= end) {
			return FALSE;
		}

		c = *(p++);
		if (!ResetLength(t, c)) {
			return FALSE;
		}

		sz = 0;
		for (i = 0; i < c; i++) {
			INT32 n = 0;

			if (i == 0) {
				if (!DecodeFSize(p, end, &fs, &n)) {
					return FALSE;
				}
			} else {
				if (!DecodeNSize(p, end, &fs, &n)) {
					return FALSE;
				}
			}

			p += n;

			t->Length[(t->Count)++] = fs;

			sz += fs;
		}

		ts = (INT32)(end - p);
		fs = ts - sz;

		if (fs < 0) {
			return FALSE;
		}

		t->Length[(t->Count)++] = fs;

		break;
	}

	} /* switch */

	t->p = p;

	t->Index = 0;

	return TRUE;
}

/* */

INT32 BlockLacing_ReadNext(
	BlockLacing_t* t,
	QM_Frame_t*    frame)
{
	INT32 fs;

	memset(frame, 0, sizeof(QM_Frame_t));

	if (t->Index >= t->Count) {
		return 0;
	}

	fs = t->Length[t->Index];

	if (t->Size < fs) {
		UINT8* p;

		while (t->Size < fs) {
			t->Size *= 2;
		}

		p = (INT8*)t->Allocator->Reallocate(t->Allocator->Context, t->Buffer, t->Size);
		if (p == NULL) {
			return -1;
		}

		t->Buffer = p;
	}

	memcpy(t->Buffer, t->p, fs);

	frame->Payload = t->Buffer;
	frame->Size    = fs;

	t->p     += fs;
	t->Index += 1;

	return 1;
}

/* */

