package zephyr.sanshusha.accessdejp;

import java.util.HashMap;
import java.util.TreeSet;

public class TagDecoder {

    private static final HashMap<Integer, String> tagMap = new HashMap<Integer, String>();

    private final static int ZENKAKU_START = 0xE1463830;
    private final static int HANKAKU_START = 0xE1463030;
    private final static int EXTCHAR_START = 0xE1463032;
    private final static int YOUREI_START = 0xE14C340A;
    private final static int WAKU_START = 0xE1564C;

    final static String WAKU_START_TAG = "<WAKU_START/>";
    final static String WAKU_END_TAG = "<WAKU_END/>";

    final static String SECTION_TAG = "<SECTION/>";
    final static String SUBSECTION_TAG = "<SUBSECTION/>";

    static {
        tagMap.put(ZENKAKU_START, ""); // 全角開始
        tagMap.put(HANKAKU_START, ""); // 半角開始
        tagMap.put(EXTCHAR_START, ""); // 特殊文字開始
        //
        tagMap.put(0xE14231, "<b>");
        tagMap.put(0xE14230, "</b>");
        tagMap.put(0xE14931, "<i>");
        tagMap.put(0xE14930, "</i>");
        tagMap.put(0xE14C31, SECTION_TAG); // インデント・１？
        tagMap.put(0xE14C32, SUBSECTION_TAG); // インデント・２？
        tagMap.put(0xE14C33, ""); // 行末？右三角矢印？用例の始まり
        tagMap.put(0xE15332, ""); // ゴシックフォント開始？（<b>とは別）
        tagMap.put(0xE15333, ""); // ゴシックフォント終了？（</b>とは別）
        tagMap.put(WAKU_START, WAKU_START_TAG); // 枠囲み独例文開始？
        tagMap.put(0xE1566C, WAKU_END_TAG); // 枠囲み独例文終了？
        tagMap.put(0xE17331, "<sub>");
        tagMap.put(0xE17330, "</sub>");
    }

    private static final HashMap<Integer, String> uniMap = new HashMap<Integer, String>();

    static {
        uniMap.put(0xE0, "à");
        uniMap.put(0xE4, "ä");
        uniMap.put(0xF6, "ö");
        uniMap.put(0xFC, "ü");
        uniMap.put(0xDF, "ß");
        uniMap.put(0xC4, "Ä");
        uniMap.put(0xD6, "Ö");
        uniMap.put(0xDC, "Ü");
    }

    private static final HashMap<Integer, String> extChMap = new HashMap<Integer, String>();

    private static int bytes2int(byte b0, byte b1) {
        return ((Util.sbyte2int(b0) << 8) + Util.sbyte2int(b1));
    }

    private static int str2int(String str) {
        byte[] data = str.getBytes();
        if (data.length == 2) {
            return bytes2int(data[0], data[1]);
        } else {
            System.err.println("PROGRAM ERROR: " + str);
            return -1;
        }
    }

    static {
        extChMap.put((int) '!', "́");
        extChMap.put(str2int("A~"), "à");
        extChMap.put(str2int("a!"), "á");
        extChMap.put(str2int("e!"), "é");
        extChMap.put(str2int("I!"), "ɪ́");
        extChMap.put(0x9221, "ɪ́");
        extChMap.put(str2int("i!"), "í");
        extChMap.put(str2int("o!"), "ó");
        extChMap.put(str2int("u!"), "ú");
        extChMap.put(0xE32E, "ǿ");
        extChMap.put(0xA5, "͜"); // &#x35c;, or .͡ &#x361;
        extChMap.put((int) 'C', "ç");
        extChMap.put((int) 'E', "ə");
        extChMap.put((int) 'g', "ɡ"); // &#x261;
        extChMap.put((int) 'I', "ɪ");
        extChMap.put((int) 'N', "ŋ");
        extChMap.put((int) 'O', "ɔ");
        extChMap.put((int) 'S', "ʃ");
        extChMap.put((int) 'Y', "ʏ");
        extChMap.put((int) '{', "œ");
        extChMap.put(0xC9, "ː");
        extChMap.put(0xD3, "ɛ");
        extChMap.put(0xD37E, "ɛ̀");
        extChMap.put(0xE3, "ø");
        extChMap.put(0xE7, "ʊ");
        //
        extChMap.put((int) 'a', "a");
        extChMap.put((int) 'f', "f");
        extChMap.put((int) 'p', "p");
        extChMap.put((int) 'r', "r");
        extChMap.put((int) 's', "s");
        extChMap.put((int) 't', "t");
        extChMap.put((int) 'y', "y");
    }

    private static final TreeSet<Character> unknownExChar = new TreeSet<Character>();

    private static int outExtChars(StringBuffer sb, byte[] inData, int start) {
        int i = start;
        while (inData[i] != BYTE_E1) {
            int v0 = Util.sbyte2int(inData[i]);
            int val2 = ((v0 << 8) + Util.sbyte2int(inData[i + 1]));
            String ext = extChMap.get(val2);
            if (ext != null) {
                sb.append(ext);
                i += 2;
            } else if ((ext = extChMap.get(v0)) != null) {
                sb.append(ext);
                i++;
            } else {
                char c = (char) v0;
                sb.append(c);
                if (!unknownExChar.contains(c)) {
                    unknownExChar.add(c);
                    System.err.println("unknown extChar: " + c
                            + String.format("\t%x %x %x", inData[i - 1], inData[i], inData[i + 1]));
                    System.err.println("\t" + sb.toString().replace("\n", " "));
                }
                i++;
            }
        }
        return i;
    }

    private static byte BYTE_E1 = Util.int2sbyte(0xE1);

    // private static byte BYTE_A5 = Util.int2sbyte(0xA5);

    String decode(byte[] inData, int length) {
        int charMode = HANKAKU_START;
        Buffer outBuf = new Buffer();
        StringBuffer sb = new StringBuffer();
        int i = 0;
        while (i < length) {
            byte b = inData[i];
            int c = Util.sbyte2int(b);
            if (b != BYTE_E1) {
                String uni;
                if (charMode == HANKAKU_START) {
                    if ((uni = uniMap.get(c)) != null) {
                        sb.append(outBuf.toString(false));
                        outBuf.clear();
                        sb.append(uni);
                    } else if (c == 0xB7) {
                        sb.append(outBuf.toString(false));
                        outBuf.clear();
                        sb.append("·");
                    } else {
                        outBuf.add(b);
                    }
                } else {
                    outBuf.add(b);
                }
                i++;
            } else {
                int tag3, tag4;
                tag3 = Util.sbyte2int(inData[i]) << 16;
                tag3 += Util.sbyte2int(inData[i + 1]) << 8;
                tag3 += Util.sbyte2int(inData[i + 2]);
                tag4 = (tag3 << 8) + Util.sbyte2int(inData[i + 3]);

                switch (tag4) {
                case HANKAKU_START:
                    sb.append(outBuf.toString(true)); // たまった全角文字列を出力
                    outBuf.clear();
                    charMode = tag4;
                    i += 4;
                    break;
                case ZENKAKU_START:
                    sb.append(outBuf.toString(false)); // たまった半角文字列を出力
                    outBuf.clear();
                    charMode = tag4;
                    i += 4;
                    break;
                case EXTCHAR_START:
                    sb.append(outBuf.toString(charMode == ZENKAKU_START)); // たまった文字列を出力
                    outBuf.clear();
                    i = outExtChars(sb, inData, i + 4);
                    break;
                case YOUREI_START:
                    sb.append(outBuf.toString(charMode == ZENKAKU_START)); // たまった文字列を出力
                    outBuf.clear();
                    sb.append("▷\n");
                    i += 4;
                    break;
                default:
                    String tagtxt;
                    if ((tagtxt = tagMap.get(tag3)) != null) {
                        sb.append(outBuf.toString(charMode == ZENKAKU_START));
                        outBuf.clear();
                        i += 3;
                        if (tag3 == WAKU_START) {
                            while (inData[i] != BYTE_E1) {
                                i++;
                            }
                        }
                        outBuf.add(tagtxt);
                    } else {
                        outBuf.add(b);
                        i++;
                    }
                    break;
                }
            }
        }
        sb.append(outBuf.toString(charMode == ZENKAKU_START));
        return sb.toString();
    }

    static void dumpUnknown() {
        for (char c : unknownExChar) {
            System.err.print(c);
        }
        if (!unknownExChar.isEmpty()) {
            System.err.println();
        }
    }
}
