//
//  AccountManager.m
//  GalapagosReader
//
//  Created by Yoshitaka Sakamaki on 平成22/02/25.
//  Copyright 2010 MyHOME. All rights reserved.
//
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//

#import "AccountManager.h"
static AccountManager *_accountManager = nil;
NSUserDefaults* defaults = nil;
NSString *ACCOUNT_NAME_GOOGLE = @"ACCOUNT_NAME_GOOGLE";
NSString *ACCOUNT_NAME_HATENA = @"ACCOUNT_NAME_HATENA";
NSString *ITEM_GOOGLE = @"GalapagosReader:GoogleSync";
NSString *ITEM_HATENA = @"GalapagosReader:HatenaSync";
NSString *APP_NAME = @"GalapagosReader";
NSString *ITEM_KIND = @"アプリケーションパスワード";

@interface AccountManager (Private) 
- (BOOL)addKeychainItem:(NSString *)keychainItemName 
		   withItemKind:(NSString *)keychainItemKind 
			forUsername:(NSString *)username 
		   withPassword:(NSString *)password;

- (NSString *)getPasswordFromKeychainItem:(NSString *)keychainItemName 
							 withItemKind:(NSString *)keychainItemKind 
							  forUsername:(NSString *)username;

- (NSString *)getPasswordFromSecKeychainItemRef:(SecKeychainItemRef)item;

- (BOOL)modifyKeychainItem:(NSString *)keychainItemName
			  withItemKind:(NSString *)keychainItemKind
			   forUsername:(NSString *)username
		   withNewPassword:(NSString *)newPassword;

- (BOOL)findKeychain:(NSString *)keychainItemName 
		withItemKind:(NSString *)keychainItemKind 
		 forUsername:(NSString *)username;
- (void)setAccountNameGoogle:(NSString *) name;
- (void)setAccountNameHatena:(NSString *) name;
- (void)setPasswordGoogle:(NSString *) password;
- (void)setPasswordHatena:(NSString *) password;
@end

@implementation AccountManager

-(id)init
{
	if ((self = [super init]) != nil)
	{
		defaults = [NSUserDefaults standardUserDefaults];
	}
	return self;
}

/* sharedManager
 * Returns the single instance of the refresh manager.
 */
+(AccountManager *)sharedManager
{
	if (!_accountManager)
		_accountManager = [[AccountManager alloc] init];
	return _accountManager;
}

- (void)setAccountNameGoogle:(NSString *) name
{
	[defaults setObject:name forKey:ACCOUNT_NAME_GOOGLE];
}

- (void)setAccountNameHatena:(NSString *) name
{
	[defaults setObject:name forKey:ACCOUNT_NAME_HATENA];
}

- (NSString *)getAccountNameGoogle
{
	return [defaults stringForKey:ACCOUNT_NAME_GOOGLE];
}

- (NSString *)getAccountNameHatena
{
	return [defaults stringForKey:ACCOUNT_NAME_HATENA];
}

- (void) setPasswordGoogle:(NSString *) password
{
	/* 新旧パスワードを比較し、変更をされていれば、キーチェーンの存在確認を行う。
	 * キーチェンが存在すれば更新。無ければ新規登録を行う。
	 */	
	if (NO == [[self getPasswordGoogle] isEqualToString:password])
	{
		if (NO == [self checkForExistanceOfKeychainItem:ITEM_GOOGLE])
		{
			//新規追加
			BOOL result = [self addKeychainItem:APP_NAME
								   withItemKind:ITEM_KIND
									forUsername:ITEM_GOOGLE
								   withPassword:password];
			if (YES == result){
				NSLog(@"パスワードを新規にセットしました。Google");
			}			
		} else {
			//更新
			BOOL result = [self modifyKeychainItem:APP_NAME
									  withItemKind:ITEM_KIND
									   forUsername:ITEM_GOOGLE
								   withNewPassword:password];
			if (YES == result){
				NSLog(@"パスワードを更新されました。Google");
			}				
		}
	}
}

- (void) setPasswordHatena:(NSString *) password
{
	/* 新旧パスワードを比較し、変更をされていれば、キーチェーンの存在確認を行う。
	 * キーチェンが存在すれば更新。無ければ新規登録を行う。
	 */
	if (NO == [[self getPasswordHatena] isEqualToString:password])
	{
		if (NO == [self checkForExistanceOfKeychainItem:ITEM_HATENA])
		{
			BOOL result = [self addKeychainItem:APP_NAME
								   withItemKind:ITEM_KIND
									forUsername:ITEM_HATENA
								   withPassword:password];
			if (YES == result){
				NSLog(@"パスワードを新規にセットしました。はてな");
			}				
		} else {
			BOOL result = [self modifyKeychainItem:APP_NAME
									  withItemKind:ITEM_KIND
									   forUsername:ITEM_HATENA
								   withNewPassword:password];
			if (YES == result){
				NSLog(@"パスワードを更新されました。はてな");
			}			
		}
	}
}

- (NSString *)getPasswordGoogle
{
	NSString * password = [self getPasswordFromKeychainItem:APP_NAME
											   withItemKind:ITEM_KIND
												forUsername:ITEM_GOOGLE];
	return password;
}

- (NSString *)getPasswordHatena
{
	NSString * password = [self getPasswordFromKeychainItem:APP_NAME
											   withItemKind:ITEM_KIND
												forUsername:ITEM_HATENA];
	return password;	
}

- (BOOL)checkForExistanceOfKeychainItem:(NSString*)forUsername
{
	return [self findKeychain:APP_NAME
				 withItemKind:ITEM_KIND
				  forUsername:forUsername];
}

- (BOOL)findKeychain:(NSString *)keychainItemName 
		withItemKind:(NSString *)keychainItemKind 
		 forUsername:(NSString *)username
{
	SecKeychainSearchRef search;
	SecKeychainItemRef item;
	SecKeychainAttributeList list;
	SecKeychainAttribute attributes[3];
    OSErr result;
    int numberOfItemsFound = 0;
	
	attributes[0].tag = kSecAccountItemAttr;
    attributes[0].data = (void *)[username UTF8String];
    attributes[0].length = [username length];
    
    attributes[1].tag = kSecDescriptionItemAttr;
    attributes[1].data = (void *)[keychainItemKind UTF8String];
    attributes[1].length = [keychainItemKind length];
	
	attributes[2].tag = kSecLabelItemAttr;
    attributes[2].data = (void *)[keychainItemName UTF8String];
    attributes[2].length = [keychainItemName length];
	
    list.count = 3;
    list.attr = attributes;
	
    result = SecKeychainSearchCreateFromAttributes(NULL, kSecGenericPasswordItemClass, &list, &search);
	
    if (result != noErr) {
        NSLog (@"status %d from SecKeychainSearchCreateFromAttributes\n", result);
    }
    
	while (SecKeychainSearchCopyNext (search, &item) == noErr) {
        CFRelease (item);
        numberOfItemsFound++;
    }
	
	NSLog(@"%d items found\n", numberOfItemsFound);
    CFRelease (search);
	return numberOfItemsFound;
}

- (BOOL)modifyKeychainItem:(NSString *)keychainItemName
			  withItemKind:(NSString *)keychainItemKind
			   forUsername:(NSString *)username
		   withNewPassword:(NSString *)newPassword
{
	SecKeychainAttribute attributes[3];
    SecKeychainAttributeList list;
    SecKeychainItemRef item;
	SecKeychainSearchRef search;
    OSStatus status;
	
    attributes[0].tag = kSecAccountItemAttr;
    attributes[0].data = (void *)[username UTF8String];
    attributes[0].length = [username length];
    
    attributes[1].tag = kSecDescriptionItemAttr;
    attributes[1].data = (void *)[keychainItemKind UTF8String];
    attributes[1].length = [keychainItemKind length];
	
	attributes[2].tag = kSecLabelItemAttr;
    attributes[2].data = (void *)[keychainItemName UTF8String];
    attributes[2].length = [keychainItemName length];
	
    list.count = 3;
    list.attr = attributes;
	
	SecKeychainSearchCreateFromAttributes(NULL, kSecGenericPasswordItemClass, &list, &search);
	SecKeychainSearchCopyNext (search, &item);
    status = SecKeychainItemModifyContent(item, &list, [newPassword length], [newPassword UTF8String]);
	
    if (status != 0) {
        NSLog(@"Error modifying item: %d", (int)status);
    }
	CFRelease (item);
	CFRelease(search);
	return !status;
}

- (BOOL)addKeychainItem:(NSString *)keychainItemName 
		   withItemKind:(NSString *)keychainItemKind 
			forUsername:(NSString *)username 
		   withPassword:(NSString *)password
{
	SecKeychainAttribute attributes[3];
    SecKeychainAttributeList list;
    SecKeychainItemRef item;
    OSStatus status;
	
    attributes[0].tag = kSecAccountItemAttr;
    attributes[0].data = (void *)[username UTF8String];
    attributes[0].length = [username length];
    
    attributes[1].tag = kSecDescriptionItemAttr;
    attributes[1].data = (void *)[keychainItemKind UTF8String];
    attributes[1].length = [keychainItemKind length];
	
	attributes[2].tag = kSecLabelItemAttr;
    attributes[2].data = (void *)[keychainItemName UTF8String];
    attributes[2].length = [keychainItemName length];
	
    list.count = 3;
    list.attr = attributes;
	
    status = SecKeychainItemCreateFromContent(kSecGenericPasswordItemClass, &list, [password length], [password UTF8String], NULL,NULL,&item);
    if (status != 0) {
        NSLog(@"Error creating new item: %d\n", (int)status);
    }
	return !status;
}

- (NSString *)getPasswordFromKeychainItem:(NSString *)keychainItemName 
							 withItemKind:(NSString *)keychainItemKind 
							  forUsername:(NSString *)username
{
    SecKeychainSearchRef search;
    SecKeychainItemRef item;
    SecKeychainAttributeList list;
    SecKeychainAttribute attributes[3];
    OSErr result;
    //int i = 0;
	
	attributes[0].tag = kSecAccountItemAttr;
    attributes[0].data = (void *)[username UTF8String];
    attributes[0].length = [username length];
    
    attributes[1].tag = kSecDescriptionItemAttr;
    attributes[1].data = (void *)[keychainItemKind UTF8String];
    attributes[1].length = [keychainItemKind length];
	
	attributes[2].tag = kSecLabelItemAttr;
    attributes[2].data = (void *)[keychainItemName UTF8String];
    attributes[2].length = [keychainItemName length];
	
    list.count = 3;
    list.attr = attributes;
	
    result = SecKeychainSearchCreateFromAttributes(NULL, kSecGenericPasswordItemClass, &list, &search);
	
    if (result != noErr) {
        NSLog (@"status %d from SecKeychainSearchCreateFromAttributes\n", result);
    }
	
	NSString *password = @"";
    if (SecKeychainSearchCopyNext (search, &item) == noErr) {
		password = [self getPasswordFromSecKeychainItemRef:item];
		if(!password) {
			password = @"";
		}
		CFRelease(item);
		CFRelease (search);
	}
	return password;
}

- (NSString *)getPasswordFromSecKeychainItemRef:(SecKeychainItemRef)item
{
    UInt32 length;
    char *password;
    SecKeychainAttribute attributes[8];
    SecKeychainAttributeList list;
    OSStatus status;
	
    attributes[0].tag = kSecAccountItemAttr;
    attributes[1].tag = kSecDescriptionItemAttr;
    attributes[2].tag = kSecLabelItemAttr;
    attributes[3].tag = kSecModDateItemAttr;
	
    list.count = 4;
    list.attr = attributes;
	
    status = SecKeychainItemCopyContent (item, NULL, &list, &length, 
                                         (void **)&password);
	
    // use this version if you don't really want the password,
    // but just want to peek at the attributes
    //status = SecKeychainItemCopyContent (item, NULL, &list, NULL, NULL);
    
    // make it clear that this is the beginning of a new
    // keychain item
    if (status == noErr) {
        if (password != NULL) {
			
            // copy the password into a buffer so we can attach a
            // trailing zero byte in order to be able to print
            // it out with printf
            char passwordBuffer[1024];
			
            if (length > 1023) {
                length = 1023; // save room for trailing \0
            }
            strncpy (passwordBuffer, password, length);
			
            passwordBuffer[length] = '\0';
			//printf ("passwordBuffer = %s\n", passwordBuffer);
			return [NSString stringWithUTF8String:passwordBuffer];
        }
		
        SecKeychainItemFreeContent (&list, password);
		
    } else {
        printf("Error = %d\n", (int)status);
		return @"Error getting password";
    }
	
	return nil;
}

/* changeAccountGoogle
 * ユーザが設定したアカウント情報と保存している情報を照らし合わせ、変更されていれば
 * アカウント情報を更新する。
 */
- (void) changeAccountGoogle:(NSString*)changeName :(NSString*)changePassword
{
	BOOL changeFlag = NO;
	
	if (NO == [[self getAccountNameGoogle] isEqualToString:changeName])
	{
		// アカウント名の保存
		[self setAccountNameGoogle:changeName];
		changeFlag = YES;
	}
	
	if (NO == [[self getPasswordGoogle] isEqualToString:changePassword])
	{
		// パスワード保存
		[self setPasswordGoogle:changePassword];
		changeFlag = YES;
	}
	
	if (changeFlag)
	{
		//LoginControllerGoogleへアカウント情報変更を通知
		[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_ChangeAccountGoogle" object:nil];
	}
}

/* changeAccountHatena
 * ユーザが設定したアカウント情報と保存している情報を照らし合わせ、変更されていれば
 * アカウント情報を更新する。
 */
- (void) changeAccountHatena:(NSString*)changeName :(NSString*)changePassword
{
	BOOL changeFlag = NO;
	
	if (NO == [[self getAccountNameHatena] isEqualToString:changeName])
	{
		// アカウント名の保存
		[self setAccountNameHatena:changeName];
		changeFlag = YES;
	}
	
	if (NO == [[self getPasswordHatena] isEqualToString:changePassword])
	{
		// パスワード保存
		[self setPasswordHatena:changePassword];
		changeFlag = YES;
	}
	
	if (changeFlag)
	{
		//LoginControllerHatenaへアカウント情報変更を通知
		[[NSNotificationCenter defaultCenter] postNotificationName:@"MA_Notify_ChangeAccountHatena" object:nil];
	}	
}

@end
