
/*
 * Original code by Erik John Resig (ejohn.org)
 * http://ejohn.org/blog/pure-javascript-html-parser/
 *
 */

X.Dom.Parser = {
	alphabets  : 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz',
	whiteSpace : '\t\r\n\f\b ',

	// Empty Elements - HTML 4.01
	empty : X.Dom.DTD.EMPTY,

	// Block Elements - HTML 4.01
	block : {address:1,applet:1,blockquote:1,button:1,center:1,dd:1,del:1,dir:1,div:1,dl:1,dt:1,fieldset:1,form:1,frameset:1,hr:1,iframe:1,ins:1,isindex:1,li:1,map:1,menu:1,noframes:1,noscript:1,object:1,ol:1,p:1,pre:1,script:1,table:1,tbody:1,td:1,tfoot:1,th:1,thead:1,tr:1,ul:1},

	// Inline Elements - HTML 4.01
	inline : {a:1,abbr:1,acronym:1,applet:1,b:1,basefont:1,bdo:1,big:1,br:1,button:1,cite:1,code:1,del:1,dfn:1,em:1,font:1,i:1,iframe:1,img:1,input:1,ins:1,kbd:1,label:1,map:1,object:1,q:1,s:1,samp:1,script:1,select:1,small:1,span:1,strike:1,strong:1,sub:1,sup:1,textarea:1,tt:1,u:1,'var':1},

	// Elements that you can, intentionally, leave open
	// (and which close themselves)
	closeSelf : {colgroup:1,dd:1,dt:1,li:1,options:1,p:1,tbody:1,td:1,tfoot:1,th:1,thead:1,tr:1}, // add tbody

	sisters : {
		th : { td : 1 },
		td : { th : 1 },
		dt : { dd : 1 },
		dd : { dt : 1 },
		colgroup : { caption : 1 },
		thead    : { caption : 1, colgroup : 1 },
		tfoot    : { caption : 1, colgroup : 1, thead : 1, tbody : 1 },
		tbody    : { caption : 1, colgroup : 1, thead : 1, tfoot : 1 }
	},
	/*
	 * http://www.tohoho-web.com/html/tbody.htm
	 * HTML4.01では、ヘッダとフッタを先読みして表示するために、<tbody> よりも <tfoot> の方を先に記述しなくてはならないと定義されています。
	 * IE5.0 などでは HEAD → BODY → FOOT の順に表示するのですが、
	 * <tfoot> に未対応の古いブラウザでは、HEAD → FOOT → BODY の順に表示されてしまいます。
	 * また、HTML5 では、<tfoot> と <tbody> の順番はどちらでもよいことになりました。
	 */

	// Attributes that have their values filled in disabled="disabled"
	fillAttrs : X.Dom.Attr.noValue, //{checked:1,compact:1,declare:1,defer:1,disabled:1,ismap:1,multiple:1,nohref:1,noresize:1,noshade:1,nowrap:1,readonly:1,selected:1};

	// Special Elements (can contain anything)
	special : { script : 1, style : 1, plaintext : 1, xmp : 1, textarea : 1 },
	
	exec : function( html, handler, async ){
		var special        = X.Dom.Parser.special,
			plainText      = X.Dom.Parser.plainText,
			startTime      = async && X.getTime(),
			_parseStartTag = X.Dom.Parser._parseStartTag,
			_parseEndTag   = X.Dom.Parser._parseEndTag,
			stack          = async ? async[ 1 ] : [],
			lastHtml       = html,
			chars, last, text, index;

		while ( html ) {
			chars = true;
			last  = stack[ stack.length - 1 ];
			
			// Make sure we're not in a script or style element
			if ( last && special[ last.toLowerCase() ] === 1 ) {
				if( 0 <= ( index = html.toLowerCase().indexOf( '</' + last.toLowerCase() ) ) ){
					handler.chars( html.substring( 0, index ) );
					if( index = _parseEndTag( stack, handler, html ) ){
						html = html.substring( index );
					} else {
						handler.chars( html );
						html = '';
					};
				} else {
					handler.chars( html );
					html = '';
				};
			} else {
				// Comment
				if ( html.indexOf("<!--") === 0 ) {
					if ( 0 < ( index = html.indexOf("-->") ) ) {
						handler.comment( html.substring( 4, index ) );
						html = html.substring( index + 3 );
						chars = false;
					};
	
				// end tag
				} else if ( html.indexOf("</") === 0 ) {
					if ( 2 < ( index = _parseEndTag( stack, handler, html ) ) ) {
						html = html.substring( index );
						chars = false;
					};
	
				// start tag
				} else if ( html.indexOf("<") === 0 ) {
					if( index = _parseStartTag( stack, last, handler, html ) ){
						html  = html.substring( index );
						chars = false;
					} else
					if( index === false ){
						return;
					};
				};

				if ( chars ) {
					index = html.indexOf("<");
					
					text = index < 0 ? html : html.substring( 0, index );
					html = index < 0 ? '' : html.substring( index );
					
					handler.chars( text );
				};

			};

			if( html === lastHtml ){
				handler.err( html );
				return;
			};
			
			if( async && startTime + 15 <= X.getTime() && html ){
				handler.progress( 1 - html.length / async[ 0 ] );
				X.Timer.once( 0, X.Dom.Parser.exec, [ html, handler, async ] );
				return;
			};
			
			lastHtml = html;
		};
		
		// Clean up any remaining tags
		X.Dom.Parser.parseEndTag( stack, handler );
		
		async && handler.complete();
	},

	_parseStartTag : function( stack, last, handler, html ){
		var alphabets = X.Dom.Parser.alphabets,
			whiteSpace = X.Dom.Parser.whiteSpace,
			saveAttr = X.Dom.Parser.saveAttr,
			uri   = X.Dom.DTD.ATTR_VAL_IS_URI,
			phase = 0,
			l     = html.length,
			i     = 0,
			attrs = [],
			tagName, empty,
			chr, start, attrName, quot, escape;
		
		while( i < l && phase < 9 ){
			chr = html.charAt( i );
			switch( phase ){
				case 0 :
					chr === '<' && ( ++phase );
					break;
				case 1 : // タグ名の開始を待つ
					alphabets.indexOf( chr ) !== -1 && ( ++phase && ( start = i ) );
					break;
				case 2 : // タグ名の終わりの空白文字を待つ
					whiteSpace.indexOf( chr ) !== -1 ?
						( ++phase && ( tagName = html.substring( start, i ) ) ) :
					( chr === '>' || ( empty = html.substr( i, 2 ) === '/>' ) ) &&
						( ( tagName = html.substring( start, i ) ) && ( phase = 9 ) );
					break;
				case 3 : // 属性名の開始を待つ
					alphabets.indexOf( chr ) !== -1 ?
						( ++phase && ( start = i ) ) :
					( chr === '>' || ( empty = html.substr( i, 2 ) === '/>' ) ) &&
						( phase = 9 );
					break;
				case 4 : // 属性名の終わりを待つ
					chr === '=' ?
						( ( phase = 6 ) && ( attrName = html.substring( start, i ) ) ) :
					whiteSpace.indexOf( chr ) !== -1 &&
						( ( phase = 5 ) && ( attrName = html.substring( start, i ) ) );
					break;
				case 5 : // 属性の = または次の属性または htmlタグの閉じ
					whiteSpace.indexOf( chr ) !== -1 ?// ie4 未対応の属性には cite = http:// となる
						1 :
					alphabets.indexOf( chr ) !== -1 ?
						( ( phase = 4 ) && ( attrs[ attrs.length ] = attrName ) && ( start = i ) ) :
					chr === '=' ?
						( phase = 6 ) :
					( chr === '>' || ( empty = html.substr( i, 2 ) === '/>' ) ) &&
						( ( phase = 9 ) && ( attrs[ attrs.length ] = attrName ) );
					break;
				case 6 : // 属性値の開始 quot を待つ
					( chr === '"' || chr === "'" ) ?
						( ( phase = 7 ) && ( quot = chr ) && ( start = i + 1 ) ):
					whiteSpace.indexOf( chr ) === -1 &&
						( ( phase = 8 ) && ( start = i ) ); // no quot
					break;
				case 7 : //属性値の閉じ quot を待つ
					!escape && ( chr === quot ) && ( phase = 3 ) && saveAttr( attrs, attrName, html.substring( start, i ) );
					break;
				case 8 : //閉じ quot のない属性の値
					whiteSpace.indexOf( chr ) !== -1 ?
						( ( phase = 3 ) && saveAttr( attrs, attrName, html.substring( start, i ) ) ) :
					( chr === '>' ) ?
						( ( phase = 9 ) && saveAttr( attrs, attrName, html.substring( start, i ) ) ) :
					( !escape && uri.indexOf( attrName ) === -1 && html.substr( i, 2 ) === '\/>' ) && // attr の val が uri で / で終わりかつ、未対応属性の場合
						( empty = true );
					break;
			};
			escape = chr === '\\' && !escape; // \\\\ is not escape for "
			++i;
		};
		if( phase === 9 ){
			if( X.Dom.Parser.parseStartTag( stack, last, handler, tagName, attrs, empty, i ) === false ) return false;
			return i;
		};
		return 0; // error
	},

	_parseEndTag : function( stack, handler, html ){
		var alphabets = X.Dom.Parser.alphabets,
			whiteSpace = X.Dom.Parser.whiteSpace,
			phase = 0,
			l     = html.length,
			i     = 0,
			tagName,
			chr, start;
		
		while( i < l && phase < 9 ){
			chr = html.charAt( i );
			switch( phase ){
				case 0 :
					html.substr( i, 2 ) === '</' && ( ++phase && ++i );
					break;
				case 1 : // タグ名の開始を待つ
					alphabets.indexOf( chr ) !== -1 && ( ++phase && ( start = i ) );
					break;
				case 2 : // タグ名の終わりの空白文字を待つ
					whiteSpace.indexOf( chr ) !== -1 && ( ++phase );
					( chr === '>' ) && ( phase = 9 );
					( phase !== 2 ) && ( tagName = html.substring( start, i ) );
					break;
				case 3 : // タグの終了を待つ
					chr === '>' && ( phase = 9 );
					break;
			};
			++i;
		};
		if( phase === 9 ){
			X.Dom.Parser.parseEndTag( stack, handler, tagName );
			return i;
		};
		return 0; // error
	},

	saveAttr : function( attrs, name, value ){
		name  = name.toLowerCase();
		value = X.Dom.Parser.fillAttrs[ name ] === 1 ? name : value;
		attrs[ attrs.length ] = {
			name    : name,
			value   : value,
			escaped :
				value.indexOf( '"' ) !== -1 ?
					value.split( '"' ).join( '\\"' ).split( '\\\\"' ).join( '\\"' ) :
					value
		};
	},

	parseStartTag : function( stack, last, handler, tagName, attrs, unary, index ) {
		var tagLower = tagName.toLowerCase(),
			inline   = X.Dom.Parser.inline,
			parseEndTag = X.Dom.Parser.parseEndTag,
			sisters  = X.Dom.Parser.sisters;
		if ( X.Dom.Parser.block[ tagLower ] === 1 ) {
			while ( last && inline[ last.toLowerCase() ] === 1 ) {
				parseEndTag( stack, handler, last );
				last = stack[ stack.length - 1 ];
			};
		};
		X.Dom.Parser.closeSelf[ tagLower ] === 1 && ( last === tagName || ( sisters[ tagLower ] && sisters[ tagLower ][ last.toLowerCase() ] === 1 ) ) && parseEndTag( stack, handler, last );
		unary = X.Dom.Parser.empty[ tagLower ] === 1 || !!unary;
		!unary && ( stack[ stack.length ] = tagName );
		
		return handler.start( tagName, attrs, unary, index );
	},

	parseEndTag : function( stack, handler, tagName ) {
		var pos = 0, i = stack.length;
		// If no tag name is provided, clean shop
		
		// Find the closest opened tag of the same type
		if ( tagName )
			for ( pos = i; 0 <= pos; )
				if ( stack[ --pos ] === tagName )
					break;
		
		if ( 0 <= pos ) {
			// Close all the open elements, up the stack
			for ( ; pos < i; )
				handler.end( stack[ --i ] );
			
			// Remove the open elements from the stack
			stack.length = pos;
		};
	}
	
};

X.Dom._htmlStringToXNode = {
	flat : null,
	nest : [],
	err : function( html ){
		X.Dom._htmlStringToXNode.flat.length = 0;
		X.Dom._htmlStringToXNode.ignoreError !== true && X.Notification.warn( 'X.Dom.Parser() error ' + html );
	},
	start : function( tagName, attrs, noChild, length ){
		var xnode,
			nest   = X.Dom._htmlStringToXNode.nest,
			flat   = X.Dom._htmlStringToXNode.flat,
			l      = nest.length,
			attr, name, i, _attrs; //, toIndex;
		if( l ){
			xnode = nest[ l - 1 ].create( tagName );
		} else {
			xnode = flat[ flat.length ] = X.Dom.Node.create( tagName );
		};
		if( !noChild ) nest[ l ] = xnode;
		if( i = attrs.length ){
			_attrs = {};
			for( ; i; ){
				if( attr = attrs[ --i ] ){
					if( typeof attr === 'string' ){
						name = attr;
						_attrs[ name ] = true;
					} else {
						name = attr.name;
						_attrs[ name ] = attr.escaped;
					};
				};
			};
			xnode.attr( _attrs );
		};
	},
	end : function(){
		0 < X.Dom._htmlStringToXNode.nest.length && ( --X.Dom._htmlStringToXNode.nest.length );
	},
	chars : function( text ){
		if( X.Dom._htmlStringToXNode.nest.length ){
			X.Dom._htmlStringToXNode.nest[ X.Dom._htmlStringToXNode.nest.length - 1 ].createText( text );
		} else {
			X.Dom._htmlStringToXNode.flat[ X.Dom._htmlStringToXNode.flat.length ] = X.Dom.Node.createText( text );
		};
	},
	comment : X.emptyFunction
};

X.Dom.parse = function( html, ignoreError ){
	var worker = X.Dom._htmlStringToXNode, ret;
	worker.flat = [];
	worker.nest.length = 0;
	worker.ignoreError = ignoreError;
	X.Dom.Parser.exec( html, worker );
	ret = worker.flat;
	delete worker.flat;
	return ret;
};

X.Dom._asyncHtmlStringToXNode = {
	err : function( html ){
		X.Dom._htmlStringToXNode.err( html );
		this.asyncDispatch( 0, { type : X.Event.ERROR } );
	},
	start   : X.Dom._htmlStringToXNode.start,
	end     : X.Dom._htmlStringToXNode.end,
	chars   : X.Dom._htmlStringToXNode.chars,
	comment : X.emptyFunction,
	
	progress : function( pct ){
		this.asyncDispatch( 0, { type : X.Event.PROGRESS, progress : pct } );
	},
	complete : function(){
		var ret = X.Dom._htmlStringToXNode.flat;
		delete X.Dom._htmlStringToXNode.flat;
		this.asyncDispatch( 0, { type : X.Event.SUCCESS, xnodes : ret } );
	}
};

X.Dom.asyncParse = function( html, ignoreError ){
	var dispatcher = X.Class._override( new X.EventDispatcher(), X.Dom._asyncHtmlStringToXNode ),
		worker = X.Dom._htmlStringToXNode;
	dispatcher.listenOnce( X.Event.SUCCESS, dispatcher, dispatcher.kill );
	worker.flat = [];
	worker.nest.length = 0;
	worker.ignoreError = ignoreError;
	X.Dom.Parser.exec( html, dispatcher, [ html.length, [] ] );
	return dispatcher;
};
