﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using Progressive.PecaStarter.Common;
using Progressive.PecaStarter.DAO;
using Progressive.PecaStarter.Entity;
using Progressive.PecaStarter.Service;
using Progressive.PecaStarter.ViewModel.Broadcast;
using Progressive.PecaStarter.ViewModel.Command;
using Progressive.PecaStarter.Model;

namespace Progressive.PecaStarter.ViewModel
{
    public class BroadcastViewModel : ViewModelBase
    {
        public YellowPagesViewModel YellowPagesViewModel { get; private set; }
        public WebApisViewModel WebApisViewModel { get; private set; }
        public ExternalSourceViewModel ExternalSourceViewModel { get; private set; }
        public CurrentTrackViewModel CurrentTrackViewModel { get; private set; }
        public SettingsViewModel SettingsViewModel { get; private set; }
        public RelaySelectDialogViewModel RelaySelectDialogViewModel { get; private set; }
        public CommandBase BroadcastCommand { get; private set; }
        public CommandBase UpdateCommand { get; private set; }
        public CommandBase StopCommand { get; private set; }
        public CommandBase RemoveItemCommand { get; private set; }
        public CommandBase GetRelaysCommand { get; private set; }
        public bool IsBroadcast { get { return id != PeercastDefine.NULL_RELAY_ID; } }
        public string Id
        {
            get { return id; }
            set
            {
                id = value;
                NotifyPropertyChanged("Id");
                NotifyPropertyChanged("IsBroadcast");
            }
        }
        public DateTime StartAt { get; set; }
        public string Feedback
        {
            get { return feedback; }
            set
            {
                feedback = value;
                NotifyPropertyChanged("Feedback");
            }
        }
        public string Alert
        {
            get { return alert; }
            set
            {
                alert = value;
                feedback = value;
                NotifyPropertyChanged("Alert");
            }
        }
        public string Title { get { return new ApplicationModel().ApplicationTitle; } }
        public bool IsAvailable
        {
            get
            {
                if (
                    YellowPagesViewModel.SelectedIndex < 0 ||
                    ExternalSourceViewModel.Name.Text.Length <= 0 ||
                    Uri.EscapeDataString(ExternalSourceViewModel.Name.Text).Length > PeercastDefine.MAX_PARAMETER_LENGTH ||
                    Uri.EscapeDataString(ExternalSourceViewModel.Description.Text).Length > PeercastDefine.MAX_PARAMETER_LENGTH ||
                    Uri.EscapeDataString(ExternalSourceViewModel.Genre.Text).Length + Uri.EscapeDataString(YellowPagesViewModel.SelectedYellowPages.Prefix).Length > PeercastDefine.MAX_PARAMETER_LENGTH ||
                    Uri.EscapeDataString(ExternalSourceViewModel.Contact.Text).Length > PeercastDefine.MAX_PARAMETER_LENGTH ||
                    Uri.EscapeDataString(ExternalSourceViewModel.Comment.Text).Length > PeercastDefine.MAX_PARAMETER_LENGTH ||
                    SettingsViewModel.ServerAddress.Length == 0)
                {
                    return false;
                }
                return true;
            }
        }
        public double Left { get; set; }
        public double Top { get; set; }
        public double Height { get; set; }
        public double Width { get; set; }
        public List<string> VisitedSites { get; set; }
        private string id;
        private string feedback;
        private string alert;

        public BroadcastViewModel()
        {
            YellowPagesViewModel = new YellowPagesViewModel(this);
            WebApisViewModel = new WebApisViewModel(this);
            ExternalSourceViewModel = new ExternalSourceViewModel(this);
            CurrentTrackViewModel = new CurrentTrackViewModel(this);
            SettingsViewModel = new SettingsViewModel(this);
            StartAt = DateTime.MinValue;
            Left = 0;
            Top = 0;
            Height = 0;
            Width = 0;
            VisitedSites = new List<string> { };
            id = PeercastDefine.NULL_RELAY_ID;
            feedback = "";
            alert = "";
            Load();
            BroadcastCommand = new BroadcastCommand(this);
            UpdateCommand = new UpdateCommand(this);
            StopCommand = new StopCommand(this);
            RemoveItemCommand = new RemoveItemCommand(this);
            GetRelaysCommand = new GetRelaysCommand(this);
            RelaySelectDialogViewModel = new RelaySelectDialogViewModel(this);
            PropertyChanged += OnIsBroadcastPropertyChanged;
            YellowPagesViewModel.PropertyChanged += (s, e) => NotifyCanExecuteChanged();
            ExternalSourceViewModel.PropertyChanged += (s, e) => NotifyCanExecuteChanged();
            SettingsViewModel.PropertyChanged += (s, e) => NotifyCanExecuteChanged();
            ExternalSourceViewModel.PropertyChanged += (s, e) => RemoveItemCommand.NotifyCanExecuteChanged();
            CurrentTrackViewModel.PropertyChanged += (s, e) => RemoveItemCommand.NotifyCanExecuteChanged();
        }
        ~BroadcastViewModel()
        {
            Save();
        }

        public void Save()
        {
            List<string> prefixes = new List<string>();
            foreach (var yp in YellowPagesViewModel.YellowPageses)
            {
                prefixes.Add(yp.Prefix);
            }
            SaveDataDAOFactory.Create().Update(new SaveData()
            {
                Left = Left,
                Top = Top,
                Height = Height,
                Width = Width,
                VisitedSites = VisitedSites,
                YellowPagesPrefixes = prefixes,
                YellowPagesSelectedIndex = YellowPagesViewModel.SelectedIndex,
                ExternalSourceUrls = ExternalSourceViewModel.Url.History,
                ExternalSourceNames = ExternalSourceViewModel.Name.History,
                ExternalSourceGenres = ExternalSourceViewModel.Genre.History,
                ExternalSourceDescriptions = ExternalSourceViewModel.Description.History,
                ExternalSourceContacts = ExternalSourceViewModel.Contact.History,
                ExternalSourceComments = ExternalSourceViewModel.Comment.History,
                ExternalSourceType = (Common.Type)Enum.Parse(typeof(Common.Type), ExternalSourceViewModel.Type),
                CurrentTrackArtists = CurrentTrackViewModel.Artist.History,
                CurrentTrackTitles = CurrentTrackViewModel.Title.History,
                CurrentTrackAlbums = CurrentTrackViewModel.Album.History,
                CurrentTrackGenres = CurrentTrackViewModel.Genre.History,
                CurrentTrackContacts = CurrentTrackViewModel.Contact.History,
                SettingsServerAddress = SettingsViewModel.ServerAddress,
                SettingsIsNotifyIcon = SettingsViewModel.IsNotifyIcon,
                SettingsIsSavePosition = SettingsViewModel.IsSavePosition,
                SettingsLogging = SettingsViewModel.Logging,
                SettingsLogPath = SettingsViewModel.LogPath,
                SettingsDelay = SettingsViewModel.Delay
            });
        }

        public void Load()
        {
            var entity = SaveDataDAOFactory.Create().Select();
            if (entity == null)
            {
                return;
            }
            foreach (var prefix in entity.YellowPagesPrefixes)
            {
                YellowPagesViewModel.ParseGenre(prefix);
            }
            Left = entity.Left;
            Top = entity.Top;
            Height = entity.Height;
            Width = entity.Width;
            VisitedSites = entity.VisitedSites != null ? entity.VisitedSites : new List<string> { };
            YellowPagesViewModel.SelectedIndex = entity.YellowPagesSelectedIndex;
            ExternalSourceViewModel.Url.History = entity.ExternalSourceUrls;
            ExternalSourceViewModel.Name.History = entity.ExternalSourceNames;
            ExternalSourceViewModel.Genre.History = entity.ExternalSourceGenres;
            ExternalSourceViewModel.Description.History = entity.ExternalSourceDescriptions;
            ExternalSourceViewModel.Contact.History = entity.ExternalSourceContacts;
            ExternalSourceViewModel.Comment.History = entity.ExternalSourceComments;
            ExternalSourceViewModel.Type = entity.ExternalSourceType.ToString();
            CurrentTrackViewModel.Artist.History = entity.CurrentTrackArtists;
            CurrentTrackViewModel.Title.History = entity.CurrentTrackTitles;
            CurrentTrackViewModel.Album.History = entity.CurrentTrackAlbums;
            CurrentTrackViewModel.Genre.History = entity.CurrentTrackGenres;
            CurrentTrackViewModel.Contact.History = entity.CurrentTrackContacts;
            SettingsViewModel.ServerAddress = entity.SettingsServerAddress;
            SettingsViewModel.IsNotifyIcon = entity.SettingsIsNotifyIcon;
            SettingsViewModel.IsSavePosition = entity.SettingsIsSavePosition;
            SettingsViewModel.Logging = entity.SettingsLogging;
            SettingsViewModel.LogPath = entity.SettingsLogPath;
            SettingsViewModel.Delay = entity.SettingsDelay;
        }

        private void OnIsBroadcastPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if ("IsBroadcast" != e.PropertyName)
            {
                return;
            }
            NotifyCanExecuteChanged();
        }

        private void NotifyCanExecuteChanged()
        {
            BroadcastCommand.NotifyCanExecuteChanged();
            UpdateCommand.NotifyCanExecuteChanged();
            StopCommand.NotifyCanExecuteChanged();
        }
    }
}
