/*

Copyright (c) 2003-2004, AXE, Inc.  All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<time.h>
#include	<sys/time.h>
#include	<string.h>
#include	"opvp_common.h"
#include	"opvp_driver.h"
#include	"opvp_null.h"

/* macros */
#if (_PDAPI_VERSION_MAJOR_ == 0 && _PDAPI_VERSION_MINOR_ < 2)
#define	InitBrush	{ \
				OPVP_cspaceStandardRGB, \
				{255,255,255,255}, \
				NULL, \
				0, \
				0 \
			}
#else
#define InitBrush       { \
                                OPVP_cspaceStandardRGB, \
                                {255,255,255,255}, \
                                0, \
                                0, \
                                NULL \
                        }
#endif

/* definition */
typedef	struct {
	OPVP_Point	pos;
	OPVP_Brush	sCol;
	OPVP_Brush	fCol;
	OPVP_Brush	bCol;
	OPVP_LineStyle	lStyle;
	OPVP_LineCap	lCap;
	OPVP_LineJoin	lJoin;
	OPVP_Fix	mLimit;
	OPVP_PaintMode	pMode;
} GS;
typedef	struct	_GS_stack {
	struct	_GS_stack	*prev;
	struct	_GS_stack	*next;
	GS			gs;
} GS_stack;

/* global variables */
int	errorno = OPVP_OK;

/* private variables */
static	int		outputStream = -1;
static	int		pContext = -1;
static	OPVP_CTM	ctm = {1,0,0,1,0,0};
static	OPVP_api_procs	apiList;
static	OPVP_ColorSpace	csList[3] =
			{
				OPVP_cspaceStandardRGB,
				OPVP_cspaceDeviceGray,
				OPVP_cspaceBW
			};
static	OPVP_ColorSpace	colorSpace = OPVP_cspaceStandardRGB;
static	OPVP_ROP	ropList[5] =
			{
				OPVP_ropPset,
				OPVP_ropPreset,
				OPVP_ropOr,
				OPVP_ropAnd,
				OPVP_ropXor
			};
static	OPVP_ROP	rasterOp = OPVP_ropPset;
static	OPVP_FillMode	fill = OPVP_fillModeWinding;
static	float		alphaConst = 0;
static	OPVP_Fix	lineWidth = {0,1};
static	OPVP_PaintMode	paint = OPVP_paintModeOpaque;
static	int		dashNum = 0;
static	OPVP_Fix	*dash = (OPVP_Fix*)NULL;
static	OPVP_Fix	dashOffset = {0,0};
static	OPVP_ClipRule	clip = OPVP_clipRuleEvenOdd;
static	GS		currentGS = {
				     {0,0},InitBrush,InitBrush,InitBrush,
				     OPVP_lineStyleSolid,OPVP_lineCapButt,
				     OPVP_lineJoinMiter,0,OPVP_paintModeOpaque
				    };
static	GS_stack	*GSstack = NULL;

/*
 * initialize sequence
 * execute automatically in dlopen()
 */
int	_init(
	void
	)
{
printf("loading DRIVER...\n");
printf("  apiList address is %08X\n", &apiList);
	/* nop */
	return 0;
}

/*
 * finalize sequence
 * execute automatically in dlclose()
 */
int	_fini(
	void
	)
{
printf("unloading DRIVER...\n");
	/* nop */
	return 0;
}

/*
 * private functions
 */
static
int	GetUnique(
	void
	)
{
	struct	timeval	tv;
	gettimeofday(&tv,NULL);
	return (tv.tv_sec&0x7fffff)<<8|(tv.tv_usec&0xff000)>>12;
}

static
int	SetApiList(
	void
	)
{
	apiList.OpenPrinter	= OpenPrinter;
	apiList.ClosePrinter	= ClosePrinter;
	apiList.StartJob	= StartJob;
	apiList.EndJob		= EndJob;
	apiList.StartDoc	= StartDoc;
	apiList.EndDoc		= EndDoc;
	apiList.StartPage	= StartPage;
	apiList.EndPage		= EndPage;
	apiList.ResetCTM	= ResetCTM;
	apiList.SetCTM		= SetCTM;
	apiList.GetCTM		= GetCTM;
	apiList.InitGS		= InitGS;
	apiList.SaveGS		= SaveGS;
	apiList.RestoreGS	= RestoreGS;
	apiList.QueryColorSpace	= QueryColorSpace;
	apiList.SetColorSpace	= SetColorSpace;
	apiList.GetColorSpace	= GetColorSpace;
	apiList.QueryROP	= QueryROP;
	apiList.SetROP		= SetROP;
	apiList.GetROP		= GetROP;
	apiList.SetFillMode	= SetFillMode;
	apiList.GetFillMode	= GetFillMode;
	apiList.SetAlphaConstant= SetAlphaConstant;
	apiList.GetAlphaConstant= GetAlphaConstant;
	apiList.SetLineWidth	= SetLineWidth;
	apiList.GetLineWidth	= GetLineWidth;
	apiList.SetLineDash	= SetLineDash;
	apiList.GetLineDash	= GetLineDash;
	apiList.SetLineDashOffset= SetLineDashOffset;
	apiList.GetLineDashOffset= GetLineDashOffset;
	apiList.SetLineStyle	= SetLineStyle;
	apiList.GetLineStyle	= GetLineStyle;
	apiList.SetLineCap	= SetLineCap;
	apiList.GetLineCap	= GetLineCap;
	apiList.SetLineJoin	= SetLineJoin;
	apiList.GetLineJoin	= GetLineJoin;
	apiList.SetMiterLimit	= SetMiterLimit;
	apiList.GetMiterLimit	= GetMiterLimit;
	apiList.SetPaintMode	= SetPaintMode;
	apiList.GetPaintMode	= GetPaintMode;
	apiList.SetStrokeColor	= SetStrokeColor;
	apiList.SetFillColor	= SetFillColor;
	apiList.SetBgColor	= SetBgColor;
	apiList.NewPath		= NewPath;
	apiList.EndPath		= EndPath;
	apiList.StrokePath	= StrokePath;
	apiList.FillPath	= FillPath;
	apiList.StrokeFillPath	= StrokeFillPath;
	apiList.SetClipPath	= SetClipPath;
	apiList.SetCurrentPoint	= SetCurrentPoint;
	apiList.LinePath	= LinePath;
	apiList.PolygonPath	= PolygonPath;
	apiList.RectanglePath	= RectanglePath;
	apiList.RoundRectanglePath= RoundRectanglePath;
	apiList.BezierPath	= BezierPath;
	apiList.ArcPath		= ArcPath;
	apiList.DrawBitmapText	= DrawBitmapText;
	apiList.DrawImage	= DrawImage;
	apiList.StartDrawImage	= StartDrawImage;
	apiList.TransferDrawImage= TransferDrawImage;
	apiList.EndDrawImage	= EndDrawImage;
	apiList.StartScanline	= StartScanline;
	apiList.Scanline	= Scanline;
	apiList.EndScanline	= EndScanline;
	apiList.StartRaster	= StartRaster;
	apiList.TransferRasterData= TransferRasterData;
	apiList.SkipRaster	= SkipRaster;
	apiList.EndRaster	= EndRaster;
	apiList.StartStream	= StartStream;
	apiList.TransferStreamData= TransferStreamData;
	apiList.EndStream	= EndStream;
#if (_PDAPI_VERSION_MAJOR_ > 0 || _PDAPI_VERSION_MINOR_ >= 2)
	apiList.QueryDeviceCapability = QueryDeviceCapability;
	apiList.QueryDeviceInfo = QueryDeviceInfo;
	apiList.ResetClipPath = ResetClipPath;
#endif

	return 0;
}

static
int	Output(
	char		*format,
	char		**variable
	)
{
	static	char	buffer[1024];
	static	char	tfmt[1024];
	char		*temp;
	OPVP_Fix	fix;
	double		f;
	char		*p;
	if (outputStream != -1) {
		strcpy(tfmt,format);
		if (variable) {
			temp = *variable;
			if ((p=strstr(tfmt,"%F"))) {
				*(++p)='f';
				fix = *((OPVP_Fix*)variable);
				OPVP_Fix2f(fix,f);
				snprintf(buffer,sizeof(buffer),tfmt,f);
			} else if (strstr(tfmt,"%f")) {
				snprintf(buffer,sizeof(buffer),tfmt,
				         *((float*)variable));
			} else snprintf(buffer,sizeof(buffer),tfmt,temp);
		} else {
			snprintf(buffer,sizeof(buffer),tfmt);
		}
		write(outputStream,buffer,strlen(buffer));
	}
	return 0;
}

static int OutputData(void *pdata, int n)
{
    int i,j;
    char buf[10];
    unsigned char *p = pdata;
    char *tab = "\t";
    char *lf = "\n";

    write(outputStream,lf,strlen(lf));
    for (i = 0;i < n;i += 16) {
	write(outputStream,tab,strlen(tab));
	for (j = 0;j < 16 && i+j < n;j++) {
	    sprintf(buf,"%02x ",p[i+j]);
	    write(outputStream,buf,strlen(buf));
	}
	write(outputStream,lf,strlen(lf));
    }
}
/*
 * ------------------------------------------------------------------------
 * Creating and Managing Print Contexts
 * ------------------------------------------------------------------------
 */

/*
 * OpenPrinter
 */
int	OpenPrinter(
	int		outputFD,
	char		*printerModel,
	int		*nApiEntry,
	OPVP_api_procs	**apiEntry
	)
{
	errorno = OPVP_OK;

	if (pContext != -1) {
		errorno = OPVP_BADREQUEST;
		return -1;
	}

	//pContext = GetUnique();
	pContext = 0x12345678;

	outputStream = outputFD;

	SetApiList();
	*nApiEntry = sizeof(apiList) / sizeof(int(*)());
	*apiEntry = &apiList;

	Output("OpenPrinter:outputFD=%d",(char**)&outputFD);
	Output(":printerModel=%s",(char**)&printerModel);
	Output(":nApiEntry=%d",(char**)nApiEntry);
	Output(":printerContext=%d\n",(char**)&pContext);

	return pContext;
}

/*
 * ClosePrinter
 */
static
int	ClosePrinter(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("ClosePrinter:printerContext=%d\n",(char**)&printerContext);

	outputStream = -1;
	pContext = -1;

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Job Control Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartJob
 */
static
int	StartJob(
	int		printerContext,
	char		*jobInfo
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("StartJob:printerContext=%d",(char**)&printerContext);
	Output(":jobInfo=%s\n",(char**)&jobInfo);

	return OPVP_OK;
}

/*
 * EndJob
 */
static
int	EndJob(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("EndJob:printerContext=%d\n",(char**)&printerContext);

	return OPVP_OK;
}

/*
 * StartDoc
 */
static
int	StartDoc(
	int		printerContext,
	char		*docInfo
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("StartDoc:printerContext=%d",(char**)&printerContext);
	Output(":docInfo=%s\n",(char**)&docInfo);

	return OPVP_OK;
}

/*
 * EndDoc
 */
static
int	EndDoc(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("EndDoc:printerContext=%d\n",(char**)&printerContext);

	return OPVP_OK;
}

/*
 * StartPage
 */
static
int	StartPage(
	int		printerContext,
	char		*pageInfo
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("StartPage:printerContext=%d",(char**)&printerContext);
	Output(":pageInfo=%s\n",(char**)&pageInfo);

	return OPVP_OK;
}

/*
 * EndPage
 */
static
int	EndPage(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("EndPage:printerContext=%d\n",(char**)&printerContext);

	return OPVP_OK;
}

#if (_PDAPI_VERSION_MAJOR_ > 0 || _PDAPI_VERSION_MINOR_ >= 2)
static
int     QueryDeviceCapability(
	int		printerContext,
	int		queryflag,
	int		buflen,
	char		*infoBuf
	)
{
	char *dummyInfo = "deviceCapability";

	errorno = OPVP_OK;
	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("QueryDeviceCapability:printerContext=%d\n",
	   (char**)&printerContext);
	Output(":queryflag=%d\n",(char**)&queryflag);
	Output(":buflen=%d\n",(char**)&buflen);

	if (infoBuf != NULL) {
	    int rlen = strlen(dummyInfo);

	    if (rlen > buflen-1) rlen = buflen-1;
	    strncpy(infoBuf,dummyInfo,rlen+1);
	}

	return OPVP_OK;
}

static
int     QueryDeviceInfo(
	int		printerContext,
	int		queryflag,
	int		buflen,
	char		*infoBuf
	)
{
	char *dummyInfo = "deviceInfo";

	errorno = OPVP_OK;
	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("QueryDeviceInfo:printerContext=%d\n",
	   (char**)&printerContext);
	Output(":queryflag=%d\n",(char**)&queryflag);
	Output(":buflen=%d\n",(char**)&buflen);

	if (infoBuf != NULL) {
	    int rlen = strlen(dummyInfo);

	    if (rlen > buflen-1) rlen = buflen-1;
	    strncpy(infoBuf,dummyInfo,rlen+1);
	}

	return OPVP_OK;
}

#endif
/*
 * ------------------------------------------------------------------------
 * Graphics State Object Operations
 * ------------------------------------------------------------------------
 */

/*
 * ResetCTM
 */
static
int	ResetCTM(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("ResetCTM:printerContext=%d\n",(char**)&printerContext);
	Output("ResetCTM\n",NULL);

	ctm.a = 1;
	ctm.b = 0;
	ctm.c = 0;
	ctm.d = 1;
	ctm.e = 0;
	ctm.f = 0;

	return OPVP_OK;
}

/*
 * SetCTM
 */
static
int	SetCTM(
	int		printerContext,
	OPVP_CTM	*pCTM
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!pCTM) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("SetCTM:printerContext=%d\n",(char**)&printerContext);
	Output("SetCTM:CTM={%f,",(char**)&(pCTM->a));
	Output("%f,",(char**)&(pCTM->b));
	Output("%f,",(char**)&(pCTM->c));
	Output("%f,",(char**)&(pCTM->d));
	Output("%f,",(char**)&(pCTM->e));
	Output("%f}\n",(char**)&(pCTM->f));

	ctm = *pCTM;

	return OPVP_OK;
}

/*
 * GetCTM
 */
static
int	GetCTM(
	int		printerContext,
	OPVP_CTM	*pCTM
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!pCTM) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetCTM:printerContext=%d\n",(char**)&printerContext);
	Output("GetCTM:CTM={%f,",(char**)&(ctm.a));
	Output("%f,",(char**)&(ctm.b));
	Output("%f,",(char**)&(ctm.c));
	Output("%f,",(char**)&(ctm.d));
	Output("%f,",(char**)&(ctm.e));
	Output("%f}\n",(char**)&(ctm.f));

	*pCTM = ctm;

	return OPVP_OK;
}

/*
 * InitGS
 */
static
int	InitGS(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("InitGS:printerContext=%d\n",(char**)&printerContext);
	Output("InitGS\n",NULL);

	return OPVP_OK;
}

/*
 * SaveGS
 */
static
int	SaveGS(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SaveGS:printerContext=%d\n",(char**)&printerContext);
	Output("SaveGS\n",NULL);

	if (!GSstack) {
		GSstack = calloc(sizeof(GS_stack),1);
		GSstack->prev = NULL;
		GSstack->next = NULL;
	} else {
		GSstack->next = calloc(sizeof(GS_stack),1);
		GSstack->next->prev = GSstack;
		GSstack->next->next = NULL;
		GSstack = GSstack->next;
	}
	GSstack->gs = currentGS;

	return OPVP_OK;
}

/*
 * RestoreGS
 */
static
int	RestoreGS(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("RestoreGS:printerContext=%d\n",(char**)&printerContext);
	Output("RestoreGS\n",NULL);

	if (!GSstack) {
		currentGS = GSstack->gs;
		if (GSstack->prev) {
			GSstack = GSstack->prev;
			free(GSstack->next);
			GSstack->next = NULL;
		} else {
			free(GSstack);
			GSstack = NULL;
		}
	}

	return OPVP_OK;
}

/*
 * QueryColorSpace
 */
static
int	QueryColorSpace(
	int		printerContext,
	OPVP_ColorSpace	*pcspace,
	int		*pnum
	)
{
	int		i;
	int		n;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!pcspace) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}
	if (!pnum) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	n = sizeof(csList)/sizeof(OPVP_ColorSpace);

	//Output("QueryColorSpace:printerContext=%d\n",(char**)&printerContext);
	Output("QueryColorSpace:",NULL);

	for (i=0;i<*pnum;i++) {
		if (i >= n) {
			break;
		} else {
			Output("%d ",(char**)&(csList[i]));
			pcspace[i] = csList[i];
		}
	}
	Output("\n",NULL);
	*pnum = n;

	return OPVP_OK;
}

/*
 * SetColorSpace
 */
static
int	SetColorSpace(
	int		printerContext,
	OPVP_ColorSpace	cspace
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetColorSpace:printerContext=%d\n",(char**)&printerContext);
	Output("SetColorSpace:cspace=%d\n",(char**)&cspace);

	colorSpace = cspace;

	return OPVP_OK;
}

/*
 * GetColorSpace
 */
static
int	GetColorSpace(
	int		printerContext,
	OPVP_ColorSpace	*pcspace
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!pcspace) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetColorSpace:printerContext=%d\n",(char**)&printerContext);
	Output("GetColorSpace:cspace=%d\n",(char**)&colorSpace);

	*pcspace = colorSpace;

	return OPVP_OK;
}

/*
 * QueryROP
 */
static
int	QueryROP(
	int		printerContext,
	int		*pnum,
	int		*prop
	)
{
	int		i;
	int		n;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!pnum) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}
	if (!prop) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	n = sizeof(ropList)/sizeof(OPVP_ROP);

	//Output("QueryROP:printerContext=%d\n",(char**)&printerContext);
	Output("QueryROP:",NULL);

	for (i=0;i<*pnum;i++) {
		if (i >= n) {
			break;
		} else {
			Output("%d ",(char**)&(ropList[i]));
			prop[i] = ropList[i];
		}
	}
	Output("\n",NULL);
	*pnum = n;

	return OPVP_OK;
}

/*
 * SetROP
 */
static
int	SetROP(
	int		printerContext,
	int		rop
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetROP:printerContext=%d\n",(char**)&printerContext);
	Output("SetROP:rop=%d\n",(char**)&rop);

	rasterOp = rop;

	return OPVP_OK;
}

/*
 * GetROP
 */
static
int	GetROP(
	int		printerContext,
	int		*prop
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!prop) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetROP:printerContext=%d\n",(char**)&printerContext);
	Output("GetROP:rop=%d\n",(char**)&rasterOp);

	*prop = rasterOp;

	return OPVP_OK;
}

/*
 * SetFillMode
 */
static
int	SetFillMode(
	int		printerContext,
	OPVP_FillMode	fillmode
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetFillMode:printerContext=%d\n",(char**)&printerContext);
	Output("SetFillMode:fillmode=%d\n",(char**)&fillmode);

	fill = fillmode;

	return OPVP_OK;
}

/*
 * GetFillMode
 */
static
int	GetFillMode(
	int		printerContext,
	OPVP_FillMode	*pfillmode
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!pfillmode) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetFillMode:printerContext=%d\n",(char**)&printerContext);
	Output("GetFillMode:fillmode=%d\n",(char**)&fill);

	*pfillmode = fill;

	return OPVP_OK;
}

/*
 * SetAlphaConstant
 */
static
int	SetAlphaConstant(
	int		printerContext,
	float		alpha
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetAlphaConstant:printerContext=%d\n",(char**)&printerContext);
	Output("SetAlphaConstant:alpha=%f\n",(char**)&alpha);

	alphaConst = alpha;

	return OPVP_OK;
}

/*
 * GetAlphaConstant
 */
static
int	GetAlphaConstant(
	int		printerContext,
	float		*palpha
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!palpha) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetAlphaConstant:printerContext=%d\n",(char**)&printerContext);
	Output("GetAlphaConstant:alpha=%f\n",(char**)&alphaConst);

	*palpha = alphaConst;

	return OPVP_OK;
}

/*
 * SetLineWidth
 */
static
int	SetLineWidth(
	int		printerContext,
	OPVP_Fix	width
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetLineWidth:printerContext=%d\n",(char**)&printerContext);
	Output("SetLineWidth:width=%F\n",(char**)&width);

	lineWidth = width;

	return OPVP_OK;
}

/*
 * GetLineWidth
 */
static
int	GetLineWidth(
	int		printerContext,
	OPVP_Fix	*pwidth
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!pwidth) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetLineWidth:printerContext=%d\n",(char**)&printerContext);
	Output("GetLineWidth:width=%F\n",(char**)&lineWidth);

	*pwidth = lineWidth;

	return OPVP_OK;
}

/*
 * SetLineDash
 */
static
int	SetLineDash(
	int		printerContext,
	OPVP_Fix	pdash[],
	int		num
	)
{
	int		i;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (num<0) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}
	if ((num>0)&&(!pdash)) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	if (num) {
		if (dash) {
			dash = realloc(dash,(sizeof(OPVP_Fix))*num);
		} else {
			dash = malloc((sizeof(OPVP_Fix))*num);
		}
		if (!dash) {
			errorno = OPVP_FATALERROR;
			return -1;
		}
	} else {
		if (dash) {
			free(dash);
			dash = (OPVP_Fix*)NULL;
		}
	}

	//Output("SetLineDash:printerContext=%d\n",(char**)&printerContext);
	Output("SetLineDash:",NULL);

	if (dash) {
		for (i=0;i<num;i++) {
			Output("%F ",(char**)&(pdash[i]));
			dash[i] = pdash[i];
		}
	}
	Output("\n",NULL);

	dashNum = num;

	return OPVP_OK;
}

/*
 * GetLineDash
 */
static
int	GetLineDash(
	int		printerContext,
	OPVP_Fix	pdash[],
	int		*pnum
	)
{
	int		i;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!pdash) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}
	if (!pnum) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetLineDash:printerContext=%d\n",(char**)&printerContext);
	Output("GetLineDash:",NULL);

	for (i=0;i<*pnum;i++) {
		if (i >= dashNum) {
			break;
		} else {
			Output("%F ",(char**)&(dash[i]));
			pdash[i] = dash[i];
		}
	}
	Output("\n",NULL);
	*pnum = dashNum;

	return OPVP_OK;
}

/*
 * SetLineDashOffset
 */
static
int	SetLineDashOffset(
	int		printerContext,
	OPVP_Fix	offset
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetLineDashOffset:printerContext=%d\n",(char**)&printerContext);
	Output("SetLineDashOffset:offset=%F\n",(char**)&offset);

	dashOffset = offset;

	return OPVP_OK;
}

/*
 * GetLineDashOffset
 */
static
int	GetLineDashOffset(
	int		printerContext,
	OPVP_Fix	*poffset
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!poffset) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetLineDashOffset:printerContext=%d\n",(char**)&printerContext);
	Output("GetLineDashOffset:offset=%F\n",(char**)&dashOffset);

	*poffset = dashOffset;

	return OPVP_OK;
}

/*
 * SetLineStyle
 */
static
int	SetLineStyle(
	int		printerContext,
	OPVP_LineStyle	linestyle
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetLineStyle:printerContext=%d\n",(char**)&printerContext);
	Output("SetLineStyle:linestyle=%d\n",(char**)&linestyle);

	currentGS.lStyle = linestyle;

	return OPVP_OK;
}

/*
 * GetLineStyle
 */
static
int	GetLineStyle(
	int		printerContext,
	OPVP_LineStyle	*plinestyle
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!plinestyle) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetLineStyle:printerContext=%d\n",(char**)&printerContext);
	Output("SetLineStyle:linestyle=%d\n",(char**)&(currentGS.lStyle));

	*plinestyle = currentGS.lStyle;

	return OPVP_OK;
}

/*
 * SetLineCap
 */
static
int	SetLineCap(
	int		printerContext,
	OPVP_LineCap	linecap
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetLineCap:printerContext=%d\n",(char**)&printerContext);
	Output("SetLineCap:linecap=%d\n",(char**)&linecap);

	currentGS.lCap = linecap;

	return OPVP_OK;
}

/*
 * GetLineCap
 */
static
int	GetLineCap(
	int		printerContext,
	OPVP_LineCap	*plinecap
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!plinecap) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetLineCap:printerContext=%d\n",(char**)&printerContext);
	Output("GetLineCap:linecap=%d\n",(char**)&(currentGS.lCap));

	*plinecap = currentGS.lCap;

	return OPVP_OK;
}

/*
 * SetLineJoin
 */
static
int	SetLineJoin(
	int		printerContext,
	OPVP_LineJoin	linejoin
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetLineJoin:printerContext=%d\n",(char**)&printerContext);
	Output("SetLineJoin:linejoin=%d\n",(char**)&linejoin);

	currentGS.lJoin = linejoin;

	return OPVP_OK;
}

/*
 * GetLineJoin
 */
static
int	GetLineJoin(
	int		printerContext,
	OPVP_LineJoin	*plinejoin
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!plinejoin) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetLineJoin:printerContext=%d\n",(char**)&printerContext);
	Output("GetLineJoin:linejoin=%d\n",(char**)&(currentGS.lJoin));

	*plinejoin = currentGS.lJoin;

	return OPVP_OK;
}

/*
 * SetMiterLimit
 */
static
int	SetMiterLimit(
	int		printerContext,
	OPVP_Fix	miterlimit
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetMiterLimit:printerContext=%d\n",(char**)&printerContext);
	Output("SetMiterLimit:miterlimit=%d\n",(char**)&miterlimit);

	currentGS.mLimit = miterlimit;

	return OPVP_OK;
}

/*
 * GetMiterLimit
 */
static
int	GetMiterLimit(
	int		printerContext,
	OPVP_Fix	*pmiterlimit
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!pmiterlimit) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetMiterLimit:printerContext=%d\n",(char**)&printerContext);
	Output("GetMiterLimit:miterlimit=%d\n",(char**)&(currentGS.mLimit));

	*pmiterlimit = currentGS.mLimit;

	return OPVP_OK;
}

/*
 * SetPaintMode
 */
static
int	SetPaintMode(
	int		printerContext,
	OPVP_PaintMode	paintmode
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetPaintMode:printerContext=%d\n",(char**)&printerContext);
	Output("SetPaintMode:paintmode=%d\n",(char**)&paintmode);

	currentGS.pMode = paintmode;

	return OPVP_OK;
}

/*
 * GetPaintMode
 */
static
int	GetPaintMode(
	int		printerContext,
	OPVP_PaintMode	*ppaintmode
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!ppaintmode) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("GetPaintMode:printerContext=%d\n",(char**)&printerContext);
	Output("GetPaintMode:paintmode=%d\n",(char**)&(currentGS.pMode));

	*ppaintmode = currentGS.pMode;

	return OPVP_OK;
}

/*
 * SetStrokeColor
 */
static
int	SetStrokeColor(
	int		printerContext,
	OPVP_Brush	*brush
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!brush) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("SetStrokeColor:printerContext=%d\n",(char**)&printerContext);
	Output("SetStrokeColor:",NULL);
	if (brush->pbrush) {
		Output("BRUSH:",NULL);
	} else {
		Output("SOLID:",NULL);
	}
	Output("%X,",(char**)&(brush->color[0]));
	Output("%X,",(char**)&(brush->color[1]));
	Output("%X,",(char**)&(brush->color[2]));
	Output("%X",(char**)&(brush->color[3]));
	if (brush->pbrush) {
		Output(":width=%d",(char**)&(brush->pbrush->width));
		Output(":height=%d",(char**)&(brush->pbrush->height));
		Output(":pitch=%d",(char**)&(brush->pbrush->pitch));
#if (_PDAPI_VERSION_MAJOR_ == 0 && _PDAPI_VERSION_MINOR_ < 2)
		if (brush->pbrush->data) {
		    OutputData(brush->pbrush->data,
		      brush->pbrush->pitch*brush->pbrush->height);
		}
#else
		OutputData(brush->pbrush->data,
		  brush->pbrush->pitch*brush->pbrush->height);
#endif
	}
	Output("\n",NULL);

	currentGS.sCol = *brush;

	return OPVP_OK;
}

/*
 * SetFillColor
 */
static
int	SetFillColor(
	int		printerContext,
	OPVP_Brush	*brush
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!brush) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("SetFillColor:printerContext=%d\n",(char**)&printerContext);
	Output("SetFillColor:",NULL);
	if (brush->pbrush) {
		Output("BRUSH:",NULL);
	} else {
		Output("SOLID:",NULL);
	}
	Output("%X,",(char**)&(brush->color[0]));
	Output("%X,",(char**)&(brush->color[1]));
	Output("%X,",(char**)&(brush->color[2]));
	Output("%X",(char**)&(brush->color[3]));
	if (brush->pbrush) {
		Output(":width=%d",(char**)&(brush->pbrush->width));
		Output(":height=%d",(char**)&(brush->pbrush->height));
		Output(":pitch=%d",(char**)&(brush->pbrush->pitch));
#if (_PDAPI_VERSION_MAJOR_ == 0 && _PDAPI_VERSION_MINOR_ < 2)
		if (brush->pbrush->data) {
		    OutputData(brush->pbrush->data,
		      brush->pbrush->pitch*brush->pbrush->height);
		}
#else
		OutputData(brush->pbrush->data,
		  brush->pbrush->pitch*brush->pbrush->height);
#endif
	}
	Output("\n",NULL);

	currentGS.fCol = *brush;

	return OPVP_OK;
}

/*
 * SetBgColor
 */
static
int	SetBgColor(
	int		printerContext,
	OPVP_Brush	*brush
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!brush) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("SetBgColor:printerContext=%d\n",(char**)&printerContext);
	Output("SetBgColor:",NULL);
	if (brush->pbrush) {
		Output("BRUSH:",NULL);
	} else {
		Output("SOLID:",NULL);
	}
	Output("%X,",(char**)&(brush->color[0]));
	Output("%X,",(char**)&(brush->color[1]));
	Output("%X,",(char**)&(brush->color[2]));
	Output("%X",(char**)&(brush->color[3]));
	if (brush->pbrush) {
		Output(":width=%d",(char**)&(brush->pbrush->width));
		Output(":height=%d",(char**)&(brush->pbrush->height));
		Output(":pitch=%d",(char**)&(brush->pbrush->pitch));
#if (_PDAPI_VERSION_MAJOR_ == 0 && _PDAPI_VERSION_MINOR_ < 2)
		if (brush->pbrush->data) {
		    OutputData(brush->pbrush->data,
		      brush->pbrush->pitch*brush->pbrush->height);
		}
#else
		OutputData(brush->pbrush->data,
		  brush->pbrush->pitch*brush->pbrush->height);
#endif
	}
	Output("\n",NULL);

	currentGS.bCol = *brush;

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Path Operations
 * ------------------------------------------------------------------------
 */

/*
 * NewPath
 */
static
int	NewPath(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("NewPath:printerContext=%d\n",(char**)&printerContext);
	Output("NewPath\n",NULL);

	return OPVP_OK;
}

/*
 * EndPath
 */
static
int	EndPath(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("EndPath:printerContext=%d\n",(char**)&printerContext);
	Output("EndPath\n",NULL);

	return OPVP_OK;
}

/*
 * StrokePath
 */
static
int	StrokePath(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("StrokePath:printerContext=%d\n",(char**)&printerContext);
	Output("StrokePath\n",NULL);

	return OPVP_OK;
}

/*
 * FillPath
 */
static
int	FillPath(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("FillPath:printerContext=%d\n",(char**)&printerContext);
	Output("FillPath\n",NULL);

	return OPVP_OK;
}

/*
 * StrokeFillPath
 */
static
int	StrokeFillPath(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("StrokeFillPath:printerContext=%d\n",(char**)&printerContext);
	Output("StrokeFillPath\n",NULL);

	return OPVP_OK;
}

/*
 * SetClipPath
 */
static
int	SetClipPath(
	int		printerContext,
	OPVP_ClipRule	clipRule
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetClipPath:printerContext=%d\n",(char**)&printerContext);
	Output("ResetClipPath:clipRule=%d\n",(char**)&clipRule);

	clip = clipRule;

	return OPVP_OK;
}

#if (_PDAPI_VERSION_MAJOR_ > 0 || _PDAPI_VERSION_MINOR_ >= 2)
static
int	ResetClipPath(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	Output("ResetClipPath\n",NULL);

	return OPVP_OK;
}
#endif

/*
 * SetCurrentPoint
 */
static
int	SetCurrentPoint(
	int		printerContext,
	OPVP_Fix	x,
	OPVP_Fix	y
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SetCurrentPoint:printerContext=%d\n",(char**)&printerContext);
	Output("SetCurrentPoint:(x,y)=(%F,",(char**)&x);
	Output("%F)\n",(char**)&y);

	currentGS.pos.x = x;
	currentGS.pos.y = y;

	return OPVP_OK;
}

/*
 * LinePath
 */
static
int	LinePath(
	int		printerContext,
	int		flag,
	int		npoints,
	OPVP_Point	*points
	)
{
	int		i;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!points) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("LinePath:printerContext=%d\n",(char**)&printerContext);
	Output("LinePath:flag=%d:",(char**)&flag);
	//Output("LinePath:npoints=%d\n",(char**)&npoints);
	Output("(%F,",(char**)&currentGS.pos.x);
	Output("%F)",(char**)&currentGS.pos.y);

	for (i=0;i<npoints;i++) {
		Output("-(%F,",(char**)&((points[i]).x));
		Output("%F)",(char**)&((points[i]).y));
		currentGS.pos.x = (points[i]).x;
		currentGS.pos.y = (points[i]).y;
	}
	Output("\n",NULL);

	return OPVP_OK;
}

/*
 * PolygonPath
 */
static
int	PolygonPath(
	int		printerContext,
	int		npolygons,
	int		*nvertexes,
	OPVP_Point	*points
	)
{
	int		i, j, p;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!nvertexes) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}
	if (!points) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("PolygonPath:printerContext=%d\n",(char**)&printerContext);
	Output("PolygonPath",NULL);

	for (p=0,i=0;i<npolygons;i++) {
		//Output(":%02d",(char**)&i);
		Output(":(%F,",(char**)&((points[p]).x));
		Output("%F)",(char**)&((points[p]).y));
		currentGS.pos.x = (points[p]).x;
		currentGS.pos.y = (points[p]).y;
		for (j=1;j<nvertexes[i];j++) {
			p++;
			Output("-(%F,",(char**)&((points[p]).x));
			Output("%F)",(char**)&((points[p]).y));
		}
	}
	Output("\n",NULL);

	return OPVP_OK;
}

/*
 * RectanglePath
 */
static
int	RectanglePath(
	int		printerContext,
	int		nrectangles,
	OPVP_Rectangle	*rectangles
	)
{
	int		i;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!rectangles) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("RectanglePath:printerContext=%d\n",(char**)&printerContext);
	//Output("RectanglePath:nrectangles=%d\n",(char**)&nrectangles);
	Output("RectanglePath",NULL);

	for (i=0;i<nrectangles;i++) {
		//Output(":%02d",(char**)&i);
		Output(":(%F,",(char**)&((rectangles[i]).p0.x));
		Output("%F)-(",(char**)&((rectangles[i]).p0.y));
		Output("%F,",(char**)&((rectangles[i]).p1.x));
		Output("%F)",(char**)&((rectangles[i]).p1.y));
		currentGS.pos.x = (rectangles[i]).p0.x;
		currentGS.pos.y = (rectangles[i]).p0.y;
	}
	Output("\n",NULL);

	return OPVP_OK;
}

/*
 * RoundRectanglePath
 */
static
int	RoundRectanglePath(
	int		printerContext,
	int		nrectangles,
	OPVP_RoundRectangle	*rectangles
	)
{
	int		i;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!rectangles) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("RoundRectanglePath:printerContext=%d\n",(char**)&printerContext);
	//Output("RoundRectanglePath:nrectangles=%d\n",(char**)&nrectangles);
	Output("RoundRectanglePath",NULL);

	for (i=0;i<nrectangles;i++) {
		//Output(":%02d",(char**)&i);
		Output(":(%F,",(char**)&((rectangles[i]).p0.x));
		Output("%F)-(",(char**)&((rectangles[i]).p0.y));
		Output("%F,",(char**)&((rectangles[i]).p1.x));
		Output("%F)",(char**)&((rectangles[i]).p1.y));
		currentGS.pos.x = (rectangles[i]).p0.x;
		currentGS.pos.y = (rectangles[i]).p0.y;
	}
	Output("\n",NULL);

	return OPVP_OK;
}

/*
 * BezierPath
 */
#if (_PDAPI_VERSION_MAJOR_ == 0 && _PDAPI_VERSION_MINOR_ < 2)
static
int	BezierPath(
	int		printerContext,
	int		*npoints,
	OPVP_Point	*points
	)
{
	int		i, j, k;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!points) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("BezierPath:printerContext=%d\n",(char**)&printerContext);
	//Output("BezierPath:npoints[0]=%d\n",(char**)npoints);
	Output("BezierPath",NULL);

	for (i=0,j=0,k=0;npoints[i]>0;i++) {
		Output(":(%F,",(char**)&((points[k]).x));
		Output("%F)",(char**)&((points[k]).y));
		currentGS.pos.x = (points[k]).x;
		currentGS.pos.y = (points[k]).y;
		k++;
		for (j=1;j<npoints[i];j++,k++) {
			Output("-(%F,",(char**)&((points[k]).x));
			Output("%F)",(char**)&((points[k]).y));
			currentGS.pos.x = (points[k]).x;
			currentGS.pos.y = (points[k]).y;
		}
	}
	Output("\n",NULL);

	return OPVP_OK;
}
#else
static
int	BezierPath(
	int		printerContext,
	int		npoints,
	OPVP_Point	*points
	)
{
	int		i;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!points) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("BezierPath:printerContext=%d\n",(char**)&printerContext);
	//Output("BezierPath:npoints[0]=%d\n",(char**)npoints);
	Output("BezierPath",NULL);

	for (i=0;i<npoints;i++) {
		Output("-(%F,",(char**)&((points[i]).x));
		Output("%F)",(char**)&((points[i]).y));
		currentGS.pos.x = (points[i]).x;
		currentGS.pos.y = (points[i]).y;
	}
	Output("\n",NULL);

	return OPVP_OK;
}
#endif

/*
 * ArcPath
 */
static
int	ArcPath(
	int		printerContext,
	int		kind,
	int		dir,
	OPVP_Fix	bbx0,
	OPVP_Fix	bby0,
	OPVP_Fix	bbx1,
	OPVP_Fix	bby1,
	OPVP_Fix	x0,
	OPVP_Fix	y0,
	OPVP_Fix	x1,
	OPVP_Fix	y1
	)
{
	double		p0, p1;

	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("ArcPath:printerContext=%d\n",(char**)&printerContext);
	Output("ArcPath:kind=%d",(char**)&kind);
	Output(":dir=%d",(char**)&dir);
	Output(":(%F,",(char**)&bbx0);
	Output("%F)-(",(char**)&bby0);
	Output("%F,",(char**)&bbx1);
	Output("%F)",(char**)&bby1);
	Output(":(%F,",(char**)&x0);
	Output("%F)-(",(char**)&y0);
	Output("%F,",(char**)&x1);
	Output("%F)\n",(char**)&y1);

	OPVP_Fix2f(bbx0,p0);
	OPVP_Fix2f(bbx1,p1);
	currentGS.pos.x = (p0<p1 ? bbx0 : bbx1);
	OPVP_Fix2f(bby0,p0);
	OPVP_Fix2f(bby1,p1);
	currentGS.pos.y = (p0<p1 ? bby0 : bby1);

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Text Operations
 * ------------------------------------------------------------------------
 */

/*
 * DrawBitmapText
 */
static
int	DrawBitmapText(
	int		printerContext,
	int		width,
	int		height,
	int		pitch,
	void		*fontdata
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!fontdata) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("DrawBitmapText:printerContext=%d\n",(char**)&printerContext);
	Output("DrawBitmapText:width=%d",(char**)&width);
	Output(":height=%d",(char**)&height);
	Output(":pitch=%d",(char**)&pitch);
	OutputData(fontdata,height*pitch/8);

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Bitmap Image Operations
 * ------------------------------------------------------------------------
 */

/*
 * DrawImage
 */
static
int	DrawImage(
	int		printerContext,
	int		sourceWidth,
	int		sourceHeight,
	int		colorDepth,
	OPVP_ImageFormat	imageFormat,
	OPVP_Rectangle	destinationSize,
	int		count,
	void		*imagedata
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!imagedata) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("DrawImage:printerContext=%d\n",(char**)&printerContext);
	Output("DrawImage:sourceWidth=%d",(char**)&sourceWidth);
	Output(":sourceHeight=%d",(char**)&sourceHeight);
	Output(":colorDepth=%d",(char**)&colorDepth);
	Output(":imageFormat=%d",(char**)&imageFormat);
	Output(":(%F,",(char**)&(destinationSize.p0.x));
	Output("%F)-(",(char**)&(destinationSize.p0.y));
	Output("%F,",(char**)&(destinationSize.p1.x));
	Output("%F)\n",(char**)&(destinationSize.p1.y));
	Output("          count=%d\n",(char**)&count);
	OutputData(imagedata,count);
	return OPVP_OK;
}

/*
 * StartDrawImage
 */
static
int	StartDrawImage(
	int		printerContext,
	int		sourceWidth,
	int		sourceHeight,
	int		colorDepth,
	OPVP_ImageFormat	imageFormat,
	OPVP_Rectangle	destinationSize
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("StartDrawImage:printerContext=%d\n",(char**)&printerContext);
	Output("StartDrawImage:sourceWidth=%d",(char**)&sourceWidth);
	Output(":sourceHeight=%d",(char**)&sourceHeight);
	Output(":colorSpace=%d",(char**)&colorDepth);
	Output(":imageFormat=%d",(char**)&imageFormat);
	Output(":(%F,",(char**)&(destinationSize.p0.x));
	Output("%F)-(",(char**)&(destinationSize.p0.y));
	Output("%F,",(char**)&(destinationSize.p1.x));
	Output("%F)\n",(char**)&(destinationSize.p1.y));

	return OPVP_OK;
}

/*
 * TransferDrawImage
 */
static
int	TransferDrawImage(
	int		printerContext,
	int		count,
	void		*imagedata
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!imagedata) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("TransferDrawImage:printerContext=%d\n",(char**)&printerContext);
	Output("TransferDrawImage:count=%d\n",(char**)&count);
	OutputData(imagedata,count);

	return OPVP_OK;
}

/*
 * EndDrawImage
 */
static
int	EndDrawImage(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("EndDrawImage:printerContext=%d\n",(char**)&printerContext);
	Output("EndDrawImage\n",(char**)&printerContext);

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Scan Line Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartScanline
 */
static
int	StartScanline(
	int		printerContext,
	int		yposition
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("StartScanline:printerContext=%d\n",(char**)&printerContext);
	Output("StartScanline:yposition=%d\n",(char**)&yposition);

	return OPVP_OK;
}

/*
 * Scanline
 */
static
int	Scanline(
	int		printerContext,
	int		nscanpairs,
	int		*scanpairs
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!scanpairs) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("Scanline:printerContext=%d\n",(char**)&printerContext);
	Output("Scanline:nscanpairs=%d\n",(char**)&nscanpairs);
	OutputData(scanpairs,nscanpairs*sizeof(int)*2);

	return OPVP_OK;
}

/*
 * EndScanline
 */
static
int	EndScanline(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("EndScanline:printerContext=%d\n",(char**)&printerContext);
	Output("EndScanline\n",(char**)&printerContext);

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Raster Image Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartRaster
 */
static
int	StartRaster(
	int		printerContext,
	int		rasterWidth
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("StartRaster:printerContext=%d\n",(char**)&printerContext);
	Output("StartRaster:rasterWidth=%d\n",(char**)&rasterWidth);

	return OPVP_OK;
}

/*
 * TransferRasterData
 */
static
int	TransferRasterData(
	int		printerContext,
	int		count,
	unsigned char	*data
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!data) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("TransferRaster:printerContext=%d\n",(char**)&printerContext);
	Output("TransferRaster:count=%d\n",(char**)&count);
	OutputData(data,count);

	return OPVP_OK;
}

/*
 * SkipRaster
 */
static
int	SkipRaster(
	int		printerContext,
	int		count
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("SkipRaster:printerContext=%d\n",(char**)&printerContext);
	Output("SkipRaster:count=%d\n",(char**)&count);

	return OPVP_OK;
}

/*
 * EndRaster
 */
static
int	EndRaster(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("EndRaster:printerContext=%d\n",(char**)&printerContext);
	Output("EndRaster\n",NULL);

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Raster Image Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartStream
 */
static
int	StartStream(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("StartStream:printerContext=%d\n",(char**)&printerContext);
	Output("StartStream\n",NULL);

	return OPVP_OK;
}

/*
 * TransferStreamData
 */
static
int	TransferStreamData(
	int		printerContext,
	int		count,
	void		*data
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}
	if (!data) {
		errorno = OPVP_PARAMERROR;
		return -1;
	}

	//Output("TransferStreamData:printerContext=%d\n",(char**)&printerContext);
	Output("TransferStreamData:count=%d\n",(char**)&count);
	OutputData(data,count);

	return OPVP_OK;
}

/*
 * EndStream
 */
static
int	EndStream(
	int		printerContext
	)
{
	errorno = OPVP_OK;

	if (pContext != printerContext) {
		errorno = OPVP_BADCONTEXT;
		return -1;
	}

	//Output("EndStream:printerContext=%d\n",(char**)&printerContext);
	Output("EndStream\n",NULL);

	return OPVP_OK;
}

