/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringUtil;

import java.text.DecimalFormat;

/**
 * DECIMAL レンデラーは、カラムのデータをDecimal(１０進数、少数)表示する場合に
 * 使用するクラスです。
 * 負数の場合はspanタグclass="minus"を付けて出力します。
 *
 * 表示パラメータに与えられた文字列は、java.text.DecimalFormat を
 * 使用してフォーマットされます。
 * フォーマット変換前に、カンマなどの数値変換時にエラーになる情報を削除しておきます。
 *
 * フォーマットルールは、{@link java.text.DecimalFormat} を参照願います。
 * 標準のフォーマットは"#,##0.#"です。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.0.0 (2003/09/17) 新規作成
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 * @og.group データ表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_DECIMAL extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private final DecimalFormat format ;
	private final String	defValue ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_DECIMAL() {
		format = null;
//		defValue   = null;
		defValue   = ""; // 5.1.1.0 (2009/12/01)
	}

	/**
	 * 引数つきコンストラクター。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @og.rev 5.1.1.0 (2009/12/01) 初期値がnullの場合は、defValueはnullとする。
	 */
	private Renderer_DECIMAL( final DBColumn clm ) {

		String fm = clm.getRendererParam();
		if( fm == null || fm.length() == 0 || fm.equals( "_" ) ) {
			fm = "#,##0.#";
		}
		format = new DecimalFormat( fm );

		// 5.1.1.0 (2009/12/01)
		String defv = clm.getDefault();
		if( defv == null || defv.length() == 0 ) {
			defValue = "";
		}
		else {
			double dd = StringUtil.parseDouble( defv );
			defValue = format.format( dd );
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_DECIMAL( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 3.8.5.2 (2006/05/31) カンマ編集された数字の対応
	 * @og.rev 5.3.1.0 (2009/12/01) 値が0の場合は、初期値を適用する。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
//		if( value == null || (value.trim()).length() == 0 ) {
		if( value == null || (value.trim()).length() == 0 || "0".equals( value ) ) {
			return defValue;
		}

		double dd = StringUtil.parseDouble( value );

		String rtn ;
		synchronized( format ) {
			rtn = format.format( dd );
		}

		if( dd < 0.0 ) {
			rtn = "<span class=\"minus\">" + rtn + "</span>";
		}

		return rtn;
	}
}
