/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * COLUMN レンデラーは、データの値をカラム名と認識して、動的カラムを
 * 表示するクラスです。<p />
 *
 * データの値を、動的カラムのそのカラム属性を持ったフォームを表示します。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_COLUMN extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private final String lang ;
	private final String name ;			// 3.5.2.0 (2003/10/20) エラー時のキー

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.5.2.0 (2003/10/20) エラー時の表示で、name を使用します。
	 *
	 */
	public Renderer_COLUMN() {
		lang   = null;
		name   = null;
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.5.2.0 (2003/10/20) エラー時の表示で、name を使用します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_COLUMN( final DBColumn clm ) {
		lang = clm.getLang();
		name = clm.getName();
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_COLUMN( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 値で動的カラムが作成できない場合に、エラーメッセージを追加。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		ResourceManager resource = ResourceFactory.newInstance( lang ) ;
		DBColumn dbColumn = resource.getDBColumn( value );

		if( dbColumn == null ) {
			String errMsg = "指定のカラムの値[]で、動的カラムは作成できません。"
						+ HybsSystem.CR
						+ "  name=[" + name + "]"
						+ "  value=[" + value + "]";
			throw new HybsSystemException( errMsg );
		}

		return dbColumn.getLabel();
	}
}
