/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.RoleMode ;
import org.opengion.hayabusa.db.DBLastSql;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.StringUtil ;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * ファイルアップロード/ダウンロードリンクを作成するタグです(通常は result_info.jsp に組込み)。
 *
 * 画面検索した結果を、ダウンロードしたり、アップロードする機能は、画面リソースの
 * ロール制御で管理しています。
 * ただし、デフォルトで、以下の制限が掛かっています。
 *
 * ファイルダウンロード
 *   ・command="NEW|RENEW|FIRST|LAST|PREV|NEXT|VIEW" のみ
 * ファイルアップロード
 *   ・command="COPY|INSERT" のみ
 *
 * また、typeDownload="true" にするとDownload 処理を行う為の前処理を実施します。
 * Download 処理を行うには、指定の画面(呼び出し元)と、抜き出した時の
 * DBTableModel がマッチしているか判定します。
 * また、DBTableModel が存在しない、または、不備な場合(オーバーフロー)は、
 * 最終SQLを実行する必要があり、そもそも、抜出処理が出来ない状態では、
 * エラーを返す必要があります。
 * それら、一連のDownload 処理を行うかどうかを指定します。
 * true の場合、session の HybsSystem.DB_LAST_SQL_KEY に格納された
 * DBLastSql オブジェクトを取得し、scope が request または、overflow が
 * true の場合に、BODY部を実行します。(通常、queryタグを記述します。)
 * query の出力先は、scope を request にします。
 * それ以外は、tableId から取り出した DBTableModel を、scope = "request"
 * に(アドレスのみ)コピーします。
 * writeTable タグでは、scope = "request" から、DBTableModel を取り出します。

 * @og.formSample
 * ●形式：一般ユーザーが直接組み込むことはありません。
 * ●body：あり
 *
 * ●Tag定義：
 *   &lt;og:fileUD
 *       command            【TAG】command を指定します
 *       downloadImg        【TAG】ダウンロード時のアイコンを指定します(初期値:"{&#064;SYS.JSP}/image/fileDownload.gif")
 *       downloadJsp        【TAG】ダウンロード時の呼び出しJSPを指定します(初期値:"{&#064;SYS.JSP}/common/fileDownload.jsp")
 *       uploadImg          【TAG】アップロード時のアイコンを指定します(初期値:"{&#064;SYS.JSP}/image/fileUpload.gif")
 *       uploadJsp          【TAG】アップロード時の呼び出しJSPを指定します(初期値:"{&#064;SYS.JSP}/common/fileUpload.jsp")
 *       imgWidth           【TAG】アイコン幅サイズ(px)を指定します(初期値:20px)
 *       imgHeight          【TAG】アイコン高さサイズ(px)を指定します(初期値:20px)
 *       filename           【TAG】ダウンロード時ファイル名を指定します(初期値:"{&#064;GUI.LABEL}.xls")
 *       roles              【TAG】ロールズをセットします
 *       target             【TAG】ダウンロード文書を表示させるフレーム、またはウィンドウの名前を指定します
 *       errorMsg           【TAG】ダウンロード出来ない場合のエラーメッセージIDを指定します(初期値:ERR0028)
 *       typeDownload       【TAG】Download 処理を行うかどうかを指定します(初期値:false)
 *       editImg            【TAG】編集アイコンを指定します(初期値:"{&#064;SYS.JSP}/image/edit.png")
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:fileUD&gt;
 *
 * ●使用例
 *     &lt;og:fileUD command="{&#064;command}" /&gt;
 *
 *     &lt;og:fileUD
 *         command        = command設定 (ダウンロード="NEW|RENEW|FIRST|LAST|PREV|NEXT|VIEW" 、   アップロード="COPY|INSERT")
 *       [ downloadImg ]  = "{&#064;SYS.JSP}/image/fileDownload.gif"   ダウンロード時のアイコン
 *       [ downloadJsp ]  = "{&#064;SYS.JSP}/common/fileDownload.jsp"  ダウンロード時の呼び出しJSP
 *       [ uploadImg   ]  = "{&#064;SYS.JSP}/image/fileUpload.gif"     アップロード時のアイコン
 *       [ uploadJsp   ]  = "{&#064;SYS.JSP}/common/filePopup.jsp"     アップロード時の呼び出しJSP
 *       [ imgWidth    ]  = "20px"                                アイコン幅サイズ(px)
 *       [ imgHeight   ]  = "20px"                                アイコン高さサイズ(px)
 *       [ filename    ]  = "{&#064;GUI.LABEL}.xls"               ダウンロード時ファイル名
 *       [ roles       ]  = "GE|DOC"                              アクセスロールズ
 *       [ target      ]  = "_blank"                              ダウンロードファイルを開くターゲット
 *       [ errorMsg    ]  = "ERR0028"                             エラーメッセージID
 *       [ typeDownload]  = "false/true"                          Download 処理を行うかどうか(初期値:false)
 *     /&gt;
 *
 * @og.group ファイルアップロード/ダウンロード
 *
 * @og.rev 4.3.0.0 (2008/07/04) 新規追加
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileUDTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private String	sysJsp			= HybsSystem.sys( "JSP" );

	private String	command			= null;		// ダウンロード="NEW|RENEW" 、アップロード="COPY|INSERT"
	private String	downloadImg		= sysJsp + "/image/fileDownload.gif";	// ダウンロード時のアイコン
	private String	downloadJsp		= sysJsp + "/common/fileDownload.jsp";	// ダウンロード時の呼び出しJSP
	private String	uploadImg		= sysJsp + "/image/fileUpload.gif";		// アップロード時のアイコン
	private String	uploadJsp		= sysJsp + "/common/filePopup.jsp";		// アップロード時の呼び出しJSP
	private String	imgWidth		= "20px";						// アイコン幅サイズ(px)
	private String	imgHeight		= "20px";						// アイコン高さサイズ(px)
	private String	filename		= null;							// ダウンロード時ファイル名("{@GUI.LABEL}.xls")
	private String	roles			= null;							// アクセスロールズ
//	private String	target			= "_blank";						// ダウンロードファイルを開くターゲット
	private String	target			= null;							// ダウンロードファイルを開くターゲット
	private String	errorMsg		= "ERR0028";					// 選択されていません。もう一度、選択しなおして下さい。
	private boolean isError			= false;
	private boolean	typeDownload	= false;						// Download 処理を行うかどうか(初期値:false)

	private static final String	EDIT_GAMENID= "14_EDIT";					// 編集画面の画面ID
	private String				editImg		= sysJsp + "/image/edit.png";	// 編集時のアイコン

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 画面項目並べ替え対応
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doStartTag() {

		if( typeDownload ) {
			DBLastSql lastSql = (DBLastSql)getSessionAttribute( HybsSystem.DB_LAST_SQL_KEY );
			if( lastSql != null ) {
				String guikey = getRequest().getParameter( "gamenId" );
				if( lastSql.isGuiMatch( guikey ) ) {

					// view タグで仕込んだ clmNames を、Request 変数に設定します。
					setRequestAttribute( "DB.CLM_NAMES" , lastSql.getClmNames() );
					// 5.1.6.0 (2010/05/01) 画面項目並べ替え対応
					setRequestAttribute( "DB.VIEW_CLM_NAMES" , lastSql.getViewClmNames() );
					setRequestAttribute( "DB.VIEW_EDITABLE" , Boolean.toString( lastSql.isViewEditable() ) );

	//				if( lastSql.isRequest() || lastSql.isOverflow() ) {
					if( lastSql.isRequest() ) {
						// 5.1.6.0 (2010/05/01)
//						String sql = getUserInfo( "USER.LASTSQL" );
						String sql = getUserInfo( "LASTSQL" );
						if( sql != null ) {
							return( EVAL_BODY_BUFFERED );	// Body を評価する。( extends BodyTagSupport 時)
						}
					}
					String tableId = lastSql.getTableId();
					setRequestAttribute( HybsSystem.TBL_MDL_KEY, getSessionAttribute( tableId ) );
					return(SKIP_BODY);				// Body を評価しない
				}
			}

			// 処理が実行できない場合、エラーを出力します。
			jspPrint( getResource().getLabel( errorMsg ) );
			isError = true;
		}

		return(SKIP_BODY);				// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.3.0.1 (2008/08/11) callBack パラメータの設定漏れ修正、引数から見直し
	 * @og.rev 5.1.6.0 (2010/05/01) 帳票画面定義機能対応
	 * @og.rev 5.1.7.0 (2010/06/01) 進む・戻るボタンでもExcel出力ボタンを出す
	 * @og.rev 5.3.6.0 (2011/06/01) Edit機能見直し
	 * @og.rev 5.3.7.0 (2011/07/01) Excel取込で画面IDとアドレスが異なる場合にエラーとなるバグを修正
	 * @og.rev 5.3.7.0 (2011/07/01) 編集画面をリサイズできるように対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) PL/SQLを利用した検索でファイルダウンロードが出来るように対応します。
	 * @og.rev 5.4.2.0 (2011/12/01) 画面リソースで画面編集機能のON/OFFを設定できるようにする。
	 * @og.rev 5.5.2.4 (2012/05/16) nullに対する冗長な比較
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		if( isError      ) { return(SKIP_PAGE); }		// ページの残りの処理を行わない。
		if( typeDownload ) { return(EVAL_PAGE);	}		// ページの残りを評価する。

		boolean isDownload ;
		if( "NEW".equals( command ) || "RENEW".equals( command )
			|| "FIRST".equals( command ) || "LAST".equals( command )
			|| "PREV".equals( command ) || "NEXT".equals( command )
			|| "VIEW".equals( command ) ) {
			isDownload = true;
		}
		else if( "COPY".equals( command ) || "INSERT".equals( command ) ) {
			isDownload = false;
		}
		else {
			return(EVAL_PAGE);
		}

		byte bitMode ;
		if( roles != null ) {
			RoleMode roleMode = RoleMode.newInstance( roles );
			RoleMode userRoleMode = getUser().getRoleMode();
			bitMode = userRoleMode.getAccessBitMode( roleMode );
		}
		else {
			GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			bitMode = guiInfo.getBitMode();
		}

		if( filename == null ) {
			filename = getGUIInfoAttri( "LABEL" ) + ".xls" ;
		}
		String guikey = getGUIInfoAttri( "KEY" );	// 4.3.0.1 (2008/08/11)
		if( isDownload ) {
			if( RoleMode.isDownload( bitMode ) ) {
				// 4.3.0.1 (2008/08/11) callBack パラメータの設定漏れ修正、引数から見直し
				// <og:link href="{@SYS.JSP}/common/fileDownload.jsp" target="_blank" useGamenId="false"
				//		keys="filename,h_clms" vals="{@GUI.LABEL}.xls,{@h_clms}" >

				// 5.3.7.0 (2011/07/01) PL/SQLを利用した検索でファイルダウンロードが出来るように対応します。
				String href = downloadJsp + "?command=RENEW&gamenId=" + guikey + "&GAMENID=" + guikey + "&filename=" + StringUtil.urlEncode( filename ) ;

//				jspPrint( makeTag( downloadImg,downloadJsp,"Download File" ) );
				jspPrint( makeTag( downloadImg,href,"Download File" ) );

				// 5.1.6.0 (2010/05/01) 帳票画面定義機能対応
				// 5.3.6.0 (2011/06/01) Edit機能見直し
				GUIInfo editGui = getUser().getGUIInfo( EDIT_GAMENID );
				if( editGui != null ) {
					String hrefEdit = sysJsp + "/" + editGui.getAddress() + "/"
										+ "?command=NEW&gamenId=" + guikey + "&GAMENID=" + guikey;

					String editName = getRequestValue( "editName" );
					// 5.5.2.4 (2012/05/16) nullに対する冗長な比較
					if( editName != null ) {
//						hrefEdit += "&editName=" + ( editName == null ? "" : editName );
						hrefEdit += "&editName=" + editName;
					}

					// 5.4.2.0 (2011/12/01) 画面リソースで画面編集機能のON/OFFを設定できるようにする。
					if( RoleMode.isEdit( bitMode ) ) {
						hrefEdit += "&edit=true";
					}

					// 5.3.7.0 (2011/07/01) 編集画面をリサイズできるように対応します。
					hrefEdit = "javascript:ogOpen('" + hrefEdit + "',null,null,'width=800px,height=330px,top=350,left=100,resizable=yes',false);";
					jspPrint( makeTag( editImg,hrefEdit,"Edit" ) );
				}
			}
		}
		else {
			if( RoleMode.isUpload( bitMode ) ) {
				// 4.3.0.1 (2008/08/11) callBack パラメータの設定漏れ修正、引数から見直し
				// <og:link href="{@SYS.JSP}/common/filePopup.jsp" keys="callBack" vals="{@GUI.KEY}/{@SESSION.JSPID}" >

				String jspId  = (String)getSessionAttribute( "JSPID" );
				// 5.3.7.0 (2011/07/01) Excel取込で画面IDとアドレスが異なる場合にエラーとなるバグを修正
				String callBack = getGUIInfoAttri( "ADDRESS" ) + "/" + jspId + "?GAMENID=" + guikey;
				callBack = StringUtil.urlEncode( callBack );
//				String href = uploadJsp + "?callBack=" + guikey + "/" + jspId ;
				String href = uploadJsp + "?callBack=" +  callBack;

//				jspPrint( makeTag( uploadImg,uploadJsp,"Upload File" ) );
				jspPrint( makeTag( uploadImg,href,"Upload File" ) );
			}
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		sysJsp			= HybsSystem.sys( "JSP" );
		command			= null;		// ダウンロード="NEW|RENEW|FIRST|LAST|PREV|NEXT|VIEW" 、アップロード="COPY|INSERT"
		downloadImg		= sysJsp + "/image/fileDownload.gif";	// ダウンロード時のアイコン
		downloadJsp		= sysJsp + "/common/fileDownload.jsp";	// ダウンロード時の呼び出しJSP
		uploadImg		= sysJsp + "/image/fileUpload.gif";		// アップロード時のアイコン
		uploadJsp		= sysJsp + "/common/filePopup.jsp";		// アップロード時の呼び出しJSP
		imgWidth		= "20px";						// アイコン幅サイズ(px)
		imgHeight		= "20px";						// アイコン高さサイズ(px)
		filename		= null;							// ダウンロード時ファイル名("{@GUI.LABEL}.xls")
		roles			= null;							// アクセスロールズ
//		target			= "_blank";						// ダウンロードファイルを開くターゲット
		target			= null;							// ダウンロードファイルを開くターゲット
		errorMsg		= "ERR0028";					// 選択されていません。もう一度、選択しなおして下さい。
		typeDownload	= false;						// Download 処理を行うかどうか(初期値:false)
		isError			= false;
		editImg			= sysJsp + "/image/edit.png";	// 編集時のアイコン
	}

	/**
	 * リンクを作成します。
	 *
	 * @og.rev 4.3.0.1 (2008/08/11) callBack パラメータの設定漏れ修正、引数から見直し
	 *
	 * @param	imgsrc	イメージソース
	 * @param	href	リンクのURL
	 * @param	alt		イメージの説明文書
	 *
	 * @return	リンク文字列
	 */
//	private String makeTag( final String imgsrc, final String jspsrc,  final String alt ) {
	private String makeTag( final String imgsrc, final String href,  final String alt ) {
	//	  <og:link href="{@SYS.JSP}/common/fileDownload.jsp" target="_blank" useGamenId="false"
	//		keys="filename,h_clms" vals="{@GUI.LABEL}.xls,{@h_clms}" >
	//		<og:img width="20px" height="20px" src="{@SYS.JSP}/image/fileDownload.gif" alt="Download File"/>
	//	  </og:link>
	//	<og:equals val1="{@command}" val2="COPY|INSERT" matches="true" >
	//		<og:link href="{@SYS.JSP}/common/filePopup.jsp" keys="callBack" vals="{@GUI.KEY}/{@SESSION.JSPID}" >
	//			<og:img width="20px" height="20px" src="{@SYS.JSP}/image/fileUpload.gif" alt="Upload File"/>
	//		</og:link>
	//	</og:equals>

		// linkのBODY部にセットするイメージタグの作成
		Attributes imgAttri = new Attributes();
		imgAttri.set( "width"	, imgWidth );
		imgAttri.set( "height"	, imgHeight );
		imgAttri.set( "src"		, imgsrc );
		imgAttri.set( "alt"		, alt );

		String img = XHTMLTag.img( imgAttri );

		// 4.3.0.1 (2008/08/11) callBack パラメータの設定漏れ修正、引数から見直し
		// linkタグの作成
//		String guikey = getGUIInfoAttri( "KEY" );
//		String[] keys = new String[] { "filename","gamenId" };
//		String[] vals = new String[] {  filename , guikey   };

//		String path = XHTMLTag.urlEncode( keys,vals );

		Attributes linkAttri = new Attributes();
//		linkAttri.set( "href"	, jspsrc + "?" + path );
		linkAttri.set( "href"	, href );
		linkAttri.set( "body"	, img );
		linkAttri.set( "target"	, target );

		return XHTMLTag.link( linkAttri );
	}

	/**
	 * 【TAG】command を指定します。
	 *
	 * @og.tag
	 * command を指定します。
	 * ダウンロードは、"NEW|RENEW|FIRST|LAST|PREV|NEXT|VIEW" のみ有効です。
	 * アップロードは、"COPY|INSERT" のみ有効です。
	 * それ以外のコマンドでは、アイコンは表示されません。
	 * 例えば、NEXT/PREV など。
	 *
	 * @param	cmd コマンド[NEW/RENEW/FIRST/LAST/PREV/NEXT/VIEW]
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
	}

	/**
	 * 【TAG】ダウンロード時のアイコンを指定します(初期値:"{&#064;SYS.JSP}/image/fileDownload.gif")。
	 *
	 * @og.tag
	 * ダウンロード時に表示されるアイコンを指定します。
	 * 初期値:"{&#064;SYS.JSP}/image/fileDownload.gif" です。
	 *
	 * @param	img ダウンロード時のアイコン
	 */
	public void setDownloadImg( final String img ) {
		downloadImg = nval( getRequestParameter( img ),downloadImg );
	}

	/**
	 * 【TAG】ダウンロード時の呼び出しJSPを指定します(初期値:"{&#064;SYS.JSP}/common/fileDownload.jsp")。
	 *
	 * @og.tag
	 * ダウンロード時に呼び出されるJSPファイル名を指定します。
	 * 初期値:"{&#064;SYS.JSP}/common/fileDownload.jsp" です。
	 *
	 * @param	file ダウンロード時の呼び出しJSP
	 */
	public void setDownloadJsp( final String file ) {
		downloadJsp = nval( getRequestParameter( file ),downloadJsp );
	}

	/**
	 * 【TAG】アップロード時のアイコンを指定します(初期値:"{&#064;SYS.JSP}/image/fileUpload.gif")。
	 *
	 * @og.tag
	 * アップロード時に表示されるアイコンを指定します。
	 * 初期値:"{&#064;SYS.JSP}/image/fileUpload.gif" です。
	 *
	 * @param	img アップロード時のアイコン
	 */
	public void setUploadImg( final String img ) {
		uploadImg = nval( getRequestParameter( img ),uploadImg );
	}

	/**
	 * 【TAG】アップロード時の呼び出しJSPを指定します(初期値:"{&#064;SYS.JSP}/common/fileUpload.jsp")。
	 *
	 * @og.tag
	 * アップロード時に呼び出されるJSPファイル名を指定します。
	 * 初期値:"{&#064;SYS.JSP}/common/fileUpload.jsp" です。
	 *
	 * @param	file アップロード時の呼び出しJSP
	 */
	public void setUploadJsp( final String file ) {
		uploadJsp = nval( getRequestParameter( file ),uploadJsp );
	}

	/**
	 * 【TAG】アイコン幅サイズ(px)を指定します(初期値:20px)。
	 *
	 * @og.tag
	 * アップロード/ダウンロードのアイコンを表示する際の幅サイズ(px)を指定します。
	 * ファイルサイズは、アップロード/ダウンロード共通です。
	 * 初期値は、20px です。
	 *
	 * @param	width	アイコン幅サイズ(px)
	 */
	public void setImgWidth( final String width ) {
		imgWidth = nval( getRequestParameter( width ),imgWidth );
	}

	/**
	 * 【TAG】アイコン高さサイズ(px)を指定します(初期値:20px)。
	 *
	 * @og.tag
	 * アップロード/ダウンロードのアイコンを表示する際の高さサイズ(px)を指定します。
	 * ファイルサイズは、アップロード/ダウンロード共通です。
	 * 初期値は、20px です。
	 *
	 * @param	height	アイコン高さサイズ(px)
	 */
	public void setImgHeight( final String height ) {
		imgHeight = nval( getRequestParameter( height ),imgHeight );
	}

	/**
	 * 【TAG】ダウンロード時ファイル名を指定します(初期値:"{&#064;GUI.LABEL}.xls")。
	 *
	 * @og.tag
	 * ダウンロード時のファイル名を指定します
	 * 通常は、画面のラベル(言語対応)＋XLS拡張子です。
	 * 初期値は、"{&#064;GUI.LABEL}.xls" です。
	 *
	 * @param	file	ダウンロード時ファイル名("{&#064;GUI.LABEL}.xls")
	 */
	public void setFilename( final String file ) {
		filename = nval( getRequestParameter( file ),filename );
	}

	/**
	 * 【TAG】ロールズをセットします。
	 *
	 * @og.tag
	 * ここで指定したロールを元に、ユーザー毎のアクセス許可がチェックされます。
	 * アクセス許可されないと、表示されません。
	 * このロールを指定しない場合は、画面のロールが使用されます。
	 *
	 * @param	rls パラメータ
	 */
	public void setRoles( final String rls ) {
		roles = nval( getRequestParameter( rls ),roles );
	}

	/**
	 * 【TAG】ダウンロード文書を表示させるフレーム、またはウィンドウの名前を指定します。
	 *
	 * @og.tag フレーム名(ターゲット属性)を設定します。
	 *
	 * @param	flag ダウンロード文書のフレーム名(ターゲット属性)
	 */
	public void setTarget( final String flag ) {
		target = nval( getRequestParameter( flag ),target );
	}

	/**
	 * 【TAG】Download 処理を行うかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * Download 処理を行うには、指定の画面(呼び出し元)と、抜き出した時の
	 * DBTableModel がマッチしているか判定します。
	 * また、DBTableModel が存在しない、または、不備な場合(オーバーフロー)は、
	 * 最終SQLを実行する必要があり、そもそも、抜出処理が出来ない状態では、
	 * エラーを返す必要があります。
	 * それら、一連のDownload 処理を行うかどうかを指定します。
	 * true の場合、session の HybsSystem.DB_LAST_SQL_KEY に格納された
	 * DBLastSql オブジェクトを取得し、scope が request または、overflow が
	 * true の場合に、BODY部を実行します。(通常、queryタグを記述します。)
	 * query の出力先は、scope を request にします。
	 * それ以外は、tableId から取り出した DBTableModel を、scope = "request"
	 * に(アドレスのみ)コピーします。
	 * writeTable タグでは、scope = "request" から、DBTableModel を取り出します。
	 *
	 * 初期値は、false(抜き出しでは無くリンク作成) です。
	 *
	 * @param	type	Download 処理を行うかどうか(初期値:false)
	 */
	public void setTypeDownload( final String type ) {
		typeDownload = nval( getRequestParameter( type ),typeDownload );
	}

	/**
	 * 【TAG】ダウンロード出来ない場合のエラーメッセージIDを指定します(初期値:ERR0028)。
	 *
	 * @og.tag
	 * DBTableModel が、最後に作成された画面と、メモリ中の DBLastSql オブジェクトが
	 * 管理している情報が異なる場合は、ダウンロード出来ません。
	 * 同様に、DBLastSql オブジェクト 自身が作成されていない場合も出来ません。
	 * そのようなケースに表示するエラーメッセージのメッセージIDを指定します。
	 * 引数は、指定できません。
	 * 初期値は、『ERR0028：選択されていません。もう一度、選択しなおして下さい。』 です。
	 *
	 * @param	msg ダウンロード文書のフレーム名(ターゲット属性)
	 */
	public void setErrorMsg( final String msg ) {
		errorMsg = nval( getRequestParameter( msg ),errorMsg );
	}

	/**
	 * 【TAG】編集アイコンを指定します(初期値:"{&#064;SYS.JSP}/image/edit.png")。
	 *
	 * @og.tag
	 * 編集アイコンを指定します。
	 * 初期値:"{&#064;SYS.JSP}/image/edit.png" です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @param	img アップロード時のアイコン
	 */
	public void setEditImg( final String img ) {
		editImg = nval( getRequestParameter( img ),editImg );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "command"		,command		)
				.println( "downloadImg"	,downloadImg	)
				.println( "downloadJsp"	,downloadJsp	)
				.println( "uploadImg"	,uploadImg		)
				.println( "uploadJsp"	,uploadJsp		)
				.println( "imgWidth"	,imgWidth		)
				.println( "imgHeight"	,imgHeight		)
				.println( "filename"	,filename		)
				.println( "roles"		,roles			)
				.println( "target"		,target			)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
