/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.filter;

import org.opengion.fukurou.util.Closer;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletException;
import javax.servlet.ServletContext;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;

import java.util.Date;
import java.util.Locale ;
import java.text.DateFormat;
import java.text.SimpleDateFormat;

import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;

/**
 * AccessStopFilter は、Filter インターフェースを継承した アクセス制御クラスです。
 * web.xml で filter 設定することにより、Webアプリケーションへのアクセスを制御できます。
 * また、SYSTEM ユーザーは、このフィルターを常に通過します。
 *
 * フィルターに対してweb.xml でパラメータを設定します。
 *   ・startTime:停止開始時刻
 *   ・stopTime :停止終了時刻
 *   ・filename :停止時メッセージ表示ファイル名
 *
 * 【WEB-INF/web.xml】
 *     &lt;filter&gt;
 *         &lt;filter-name&gt;AccessStopFilter&lt;/filter-name&gt;
 *         &lt;filter-class&gt;org.opengion.hayabusa.filter.AccessStopFilter&lt;/filter-class&gt;
 *         &lt;init-param&gt;
 *             &lt;param-name&gt;startTime&lt;/param-name&gt;
 *             &lt;param-value&gt;070000&lt;/param-value&gt;
 *         &lt;/init-param&gt;
 *         &lt;init-param&gt;
 *             &lt;param-name&gt;stopTime&lt;/param-name&gt;
 *             &lt;param-value&gt;070000&lt;/param-value&gt;
 *         &lt;/init-param&gt;
 *         &lt;init-param&gt;
 *             &lt;param-name&gt;filename&lt;/param-name&gt;
 *             &lt;param-value&gt;jsp/custom/stopFile.html&lt;/param-value&gt;
 *         &lt;/init-param&gt;
 *     &lt;/filter&gt;
 *
 *     &lt;filter-mapping&gt;
 *         &lt;filter-name&gt;AccessStopFilter&lt;/filter-name&gt;
 *         &lt;url-pattern&gt;/jsp/*&lt;/url-pattern&gt;
 *     &lt;/filter-mapping&gt;
 *
 * @og.group フィルター処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class AccessStopFilter implements Filter {

	private static boolean stopFilter = false;
	private String startTime = "230000";		// 停止開始時刻
	private String stopTime  = "070000";		// 停止終了時刻
	private String filename  = null;			// 停止時メッセージ表示ファイル名
	private int    startStop = 0;

	/**
	 * フィルター処理本体のメソッドです。
	 *
	 * @og.rev 3.1.3.0 (2003/04/10) UTF-8 決め打ちで、stopFile.html を返送する。
	 * @og.rev 3.1.8.0 (2003/05/16) 文字エンコードが、UTF-8 になっていないのを修正。
	 *
	 * @param	request		ServletRequestオブジェクト
	 * @param	response	ServletResponseオブジェクト
	 * @param	chain		FilterChainオブジェクト
	 * @throws ServletException
	 */
	public void doFilter( final ServletRequest request,
							final ServletResponse response,
							final FilterChain chain)
								throws IOException, ServletException {

		if( isStop( request ) ) {
			BufferedReader in = null ;
			try {
				response.setContentType( "text/html; charset=UTF-8" );
				PrintWriter out = response.getWriter();
				in = new BufferedReader( new InputStreamReader(
								new FileInputStream( filename ) ,"UTF-8" ) );
				String str ;
				while( (str = in.readLine()) != null ) {
					out.println( str );
				}
				out.flush();
			}
			catch( UnsupportedEncodingException ex ) {
				String errMsg = "指定されたエンコーディングがサポートされていません。[UTF-8]" ;
				throw new RuntimeException( errMsg,ex );
			}
			catch( IOException ex ) {
				String errMsg = "ストリームがオープン出来ませんでした。[" + filename + "]" ;
				throw new RuntimeException( errMsg,ex );
			}
			finally {
				Closer.ioClose( in );
			}
			return;
		}

		chain.doFilter(request, response);
	}

	/**
	 * フィルターの初期処理メソッドです。
	 *
	 * フィルターに対してweb.xml で初期パラメータを設定します。
	 *   ・startTime:停止開始時刻
	 *   ・stopTime :停止終了時刻
	 *   ・filename :停止時メッセージ表示ファイル名
	 *
	 * @param	filterConfig	FilterConfigオブジェクト
	 */
	public void init(final FilterConfig filterConfig) {

		ServletContext context = filterConfig.getServletContext();
		String realPath = context.getRealPath( "/" );

		startTime = filterConfig.getInitParameter("startTime");
		stopTime  = filterConfig.getInitParameter("stopTime");
		filename  = realPath + filterConfig.getInitParameter("filename");

		if( startTime == null || stopTime == null ) {
			startStop = 0;
		}
		else {
			startStop = startTime.compareTo( stopTime );
		}
	}

	/**
	 * フィルターの終了処理メソッドです。
	 *
	 */
	public void destroy() {
		// ここでは処理を行いません。
	}

	/**
	 * フィルターの内部状態をチェックするメソッドです。
	 * 内部のフラグをもとに、停止/許可を求めます。
	 *
	 * @og.rev 3.1.8.0 (2003/05/16) 開始時刻と終了時刻を同一にしていると、画面からの制御が効かないバグを修正。
	 * @og.rev 5.5.3.2 (2012/06/08) 通過させるユーザーに、admin を追加します。
	 *
	 * @param request ServletRequestオブジェクト
	 *
	 * @return	(true:停止  false:実行許可)
	 */
	private boolean isStop( final ServletRequest request ) {
		String userID = ((HttpServletRequest)request).getRemoteUser() ;
		// 5.5.3.2 (2012/06/08) 通過させるユーザーに、admin を追加
//		if( "SYSTEM".equalsIgnoreCase( userID ) ) {
		if( "SYSTEM".equalsIgnoreCase( userID ) || "admin".equalsIgnoreCase( userID ) ) {
			return false;
		}

		// 4.0.0 (2005/01/31)
		DateFormat formatter = new SimpleDateFormat( "HHmmss",Locale.JAPAN );
		String time = formatter.format( new Date() );

		boolean rtnFlag = stopFilter;
		if( startStop < 0 ) {
			if( startTime.compareTo( time ) < 0 &&
				time.compareTo( stopTime )  < 0 ) {
					rtnFlag = true;
			}
		}
		else if( startStop > 0 ) {
			if( startTime.compareTo( time ) < 0 ||
				time.compareTo( stopTime )  < 0 ) {
					rtnFlag = true;
				}
		}
		return rtnFlag;
	}

	/**
	 * フィルターの強制停止/解除を設定するメソッドです。
	 *
	 * @og.rev 4.0.0 (2005/01/31) synchronized の廃止
	 *
	 * @param flag (true:停止  false:実行許可)
	 */
	public static void setStopFilter( final boolean flag ) {
		stopFilter = flag;
	}

	/**
	 * フィルターの内部状態(強制停止/解除)を取得するメソッドです。
	 * これは、現在、アクセス制限がどうなっているかという状態ではなく、
	 * 強制停止されているかどうかの確認メソッドです。
	 *
	 * @og.rev 4.0.0.0 (2007/11/29) getStopFilter() ⇒ isStopFilter() に変更
	 *
	 * @return	(true:停止  false:実行許可)
	 */
//	public static boolean getStopFilter() {
	public static boolean isStopFilter() {
		return stopFilter;
	}

	/**
	 * 内部状態を文字列で返します。
	 *
	 * @return	このクラスの文字列表示
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append( "AccessStopFilter" );
		sb.append( "[" ).append( startTime ).append( "],");
		sb.append( "[" ).append( stopTime  ).append( "],");
		sb.append( "[" ).append( filename  ).append( "],");
		return (sb.toString());
	}
}
