/*
 Copyright (c) 2009, hkrn All rights reserved.
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:
 
 Redistributions of source code must retain the above copyright notice, this
 list of conditions and the following disclaimer. Redistributions in binary
 form must reproduce the above copyright notice, this list of conditions and
 the following disclaimer in the documentation and/or other materials
 provided with the distribution. Neither the name of the hkrn nor
 the names of its contributors may be used to endorse or promote products
 derived from this software without specific prior written permission. 
 
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 DAMAGE.
 */

//
// $Id: Compiler.cs 116 2009-05-23 14:39:36Z hikarin $
//

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Windows.Controls;
using System.Windows.Threading;
using SlMML.Parsers;

namespace SlMML
{
    /// <summary>
    /// CompileAsyncのイベントを取り扱うメソッド
    /// </summary>
    /// <param name="sender">イベントの呼び出した元</param>
    /// <param name="e">CompileCompletedEventArgsオブジェクト</param>
    public delegate void CompileCompletedEventHandler(object sender, CompileCompletedEventArgs e);
    
    /// <summary>
    /// CompileCompletedイベントのデータを取り扱うクラス
    /// </summary>
    public class CompileCompletedEventArgs : EventArgs
    {
        public CompileCompletedEventArgs(MediaElement me)
        {
            m_element = me;
        }
        
        /// <summary>
        /// CompileAsyncによってコンパイルが完了した後に生成されたMediaElementを取得します
        /// </summary>
        public MediaElement Result
        {
            get
            {
                return m_element;
            }
        }

        private MediaElement m_element;
    }

    /// <summary>
    /// FlMMLに対応する記号で構成された文字列を同期あるいは非同期にコンパイルするクラス
    /// </summary>
    public class Compiler
    {
        public event CompileCompletedEventHandler CompileCompleted;

        #region 公開クラスメソッドの定義
        /// <summary>
        /// FlMMLに対応する記号で構成された文字列をコンパイルします
        /// </summary>
        /// <remarks>
        /// FlMMLに対応する記号文字列からひとつ以上のトラックを構成し、
        /// MediaElementを返します。このMediaElementはシーク不可能であり、
        /// Volumeは1に指定されます
        /// </remarks>
        /// <example>
        /// ドレミファソラシドを鳴らします
        /// <code>
        /// MediaElement me = SlMML.Compiler.Compile("cdefgab&lt;c");
        /// LayoutRoot.Children.Add(me);
        /// me.Play();
        /// me.MediaOpened += new RoutedEventHandler(media_MediaOpened);
        /// void media_MediaOpened(object sender, RoutedEventArgs e)
        /// {
        ///     MediaElement me = (MediaElement)e.OriginalSource;
        ///     me.Play();
        /// }
        /// </code>
        /// </example>
        /// <param name="stringToParse">FlMMLに対応する記号で構成された文字列</param>
        /// <returns>MediaElementオブジェクト</returns>
        public static MediaElement Compile(string stringToParse)
        {
            MediaElement element = new MediaElement();
            IParsable parser = new FlMMLStyleParser();
            element.Volume = 1;
            element.SetSource(parser.Parse(stringToParse));
            return element;
        }
        #endregion

        #region 公開メソッドの定義
        /// <summary>
        /// Compileメソッドを非同期で実行します
        /// </summary>
        /// <remarks>
        /// Compileと異なり、実行後すぐに実行した側に戻ります。
        /// コンパイル完了後、CompileCompleteEventHandlerイベントが発生します。
        /// MediaElementはThreadクラスで実行することはできないため、
        /// 内部ではDispatcherTimerクラスによって実行されます
        /// </remarks>
        /// <example>
        /// ドレミファソラシドを鳴らします
        /// <code>
        /// SlMML.Compiler compiler = new SlMML.Compiler();
        /// compiler.CompileCompleted += new SlMML.CompileCompletedEventHandler(compiler_CompileCompleted);
        /// compiler.CompileAsync("cdefgab&lt;c");
        /// void compiler_CompileCompleted(object sender, SlMML.CompileCompletedEventArgs e)
        /// {
        ///     MediaElemenet me = (MediaElement)e.Result;
        ///     LayoutRoot.Children.Add(me);
        ///     me.MediaOpened += new RoutedEventHandler(media_MediaOpened);
        ///     me.Play();
        /// }
        /// void media_MediaOpened(object sender, RoutedEventArgs e)
        /// {
        ///     MediaElement me = (MediaElement)e.OriginalSource;
        ///     me.Play();
        /// }
        /// </code>
        /// </example>
        /// <param name="stringToParse">FlMMLに対応する記号で構成された文字列</param>
        public void CompileAsync(string stringToParse)
        {
            m_stringToParse = stringToParse;
            DispatcherTimer timer = new DispatcherTimer();
            timer.Interval = TimeSpan.FromDays(1);
            timer.Tick += new EventHandler(CompileAsync_Tick);
            timer.Start();
            timer.Stop();
        }
#if DEBUG
        /// <summary>
        /// 文字列からトラックおよびイベントをオブジェクト形式に復元します。
        /// </summary>
        /// <remarks>
        /// テストおよびデバッグ時のみ使われるメソッドです。
        /// もともとJSON形式の文字列をオブジェクトに復元し、
        /// 同一であるかの比較をするために作られました
        /// </remarks>
        /// <param name="stringToParse">FlMMLに対応する記号で構成された文字列</param>
        /// <returns>トラックおよびイベントで構成された配列</returns>
        public static List<List<Dictionary<string, string>>> Dump(string stringToParse)
        {
            FlMMLStyleParser parser = new FlMMLStyleParser();
            parser.Parse(stringToParse);
            return parser.Dump();
        }
#endif
        #endregion

        #region 非公開メソッドの定義
        /// <summary>
        /// 文字列のコンパイルが完了されると呼び出されます
        /// </summary>
        /// <param name="e">コンパイル結果を含んだCompileCompletedEventArgsオブジェクト</param>
        protected virtual void OnCompileCompleted(CompileCompletedEventArgs e)
        {
            if (CompileCompleted != null)
                CompileCompleted(this, e);
        }

        private void CompileAsync_Tick(object sender, EventArgs e)
        {
            MediaElement me = Compiler.Compile(m_stringToParse);
            OnCompileCompleted(new CompileCompletedEventArgs(me));
        }
        #endregion

        #region メンバー変数の定義
        private string m_stringToParse;
        #endregion
    }
}
