//
//  AppController.m
//  PicoPico
//
//  Created by hkrn on 09/02/11.
//  Copyright 2009 hkrn. All rights reserved.
//

#import "AppController.h"

@implementation AppController

@synthesize status;

- (void)lock
{
	[(NSButton *)toggleButton setEnabled:NO];
	self.status = NSLocalizedString(@"Start compiling", @"");
}

- (void)toggleStateOn
{
	[(NSButton *)toggleButton setEnabled:YES];
}

- (void
)toggleStateOff
{
}

- (void)didCompile:(NSNotification *)notification
{
	self.status = NSLocalizedString(@"Compiled", @"");
}

- (void)didBuffer:(NSNotification *)notification
{
	NSDictionary *userInfo = [notification userInfo];
	NSNumber *progress = [userInfo objectForKey:@"progress"];
	NSNumber *eob = [userInfo objectForKey:@"eob"];
	NSUInteger percent = (NSUInteger)([progress doubleValue] * 100);
	self.status = [NSString stringWithFormat:NSLocalizedString(@"Buffer %d percent", @""), percent];
	if ([eob boolValue]) {
		self.status = NSLocalizedString(@"Playing", @"");
		[self toggleStateOn];
	}
}

- (void)didStop:(NSNotification *)notification
{
	[self toggleStateOff];
	self.status = NSLocalizedString(@"Stopped", @"");
}

- (void)didResume:(NSNotification *)notification
{
	self.status = NSLocalizedString(@"Resumed", @"");
}

- (id)init
{
	self = [super init];
	if (self != nil) {
		cached = @"";
		engine = [[MMLEngine alloc] init];
		timer = nil;
		NSNotificationCenter *center = [NSNotificationCenter defaultCenter];
		[center addObserver:self
				   selector:@selector(didCompile:)
					   name:MMLEngineDidCompile
					 object:nil];
		[center addObserver:self
				   selector:@selector(didBuffer:)
					   name:MMLSequencerDidBuffer
					 object:nil];
		[center addObserver:self
				   selector:@selector(didStop:)
					   name:MMLSequencerDidStop
					 object:nil];
		[center addObserver:self
				   selector:@selector(didResume:)
					   name:MMLEngineDidResume
					 object:nil];
	}
	return self;
}

- (void)dealloc
{
	if (timer != nil) {
		[timer invalidate];
		[timer release];
	}
	[cached release];
	[engine release];
	[super dealloc];
}

- (void)updateDuration:(NSTimer *)aTimer
{
	NSUInteger duration = engine.duration / 1000;
	NSUInteger timespan = engine.timespan / 1000;
	NSString *durationString = [NSString stringWithFormat:@"%02d:%02d", duration / 60, duration % 60];
	NSString *timespanString = [NSString stringWithFormat:@"%02d:%02d", timespan / 60, timespan % 60];
	//NSLog(@"%@ / %@", durationString, timespanString);
}

- (void)playMML:(NSString *)mmlText
{
	if (mmlText != nil && [mmlText length] > 0) {
		[self lock];
		//engine.filename = @"/Users/hkrn/test.wav";
		[engine play:mmlText];
		[cached release];
		cached = [mmlText retain];
	}
}

- (IBAction)play:(id)sender
{
	if ([engine isPlaying]) {
		return;
	}
	if (timer != nil) {
		[timer invalidate];
		[timer release];
	}
	timer = [[NSTimer scheduledTimerWithTimeInterval:1.0
											  target:self
											selector:@selector(updateDuration:)
											userInfo:nil
											 repeats:YES] retain];
	NSString *mmlText = [(NSTextField *)mmlTextField stringValue];
	if (cached != nil && [cached length] > 0 && [cached isEqualToString:mmlText]) {
		self.status = NSLocalizedString(@"Using cached", @"");
		[self lock];
		[engine play:cached];
		return;
	}
	NSRange range = [mmlText rangeOfString:@"pico:"];
	if (range.location != 0) {
		range = [mmlText rangeOfString:@"http://dic.nicovideo.jp/mml/"];
	}
	if (range.location == 0) {
		NSInteger n = 0;
		NSScanner *scanner = [[NSScanner alloc] initWithString:mmlText];
		[scanner setScanLocation:NSMaxRange(range)];
		if ([scanner scanInteger:&n]) {
			NSURL *url = [NSURL URLWithString:[NSString stringWithFormat:@"http://dic.nicovideo.jp/mml/%d", n]];
			[self playMML:[NSString stringWithContentsOfURL:url
												   encoding:NSUTF8StringEncoding
													  error:nil]];
			return;
		}
	}
	[self playMML:[(NSTextField *)mmlTextField stringValue]];
}

- (IBAction)togglePlay:(id)sender
{
	if ([engine isPlaying]) {
		[engine pause];
		[self toggleStateOff];
	}
	else {
		[self play:sender];
	}
}

- (IBAction)stop:(id)sender
{
	[cached release];
	cached = nil;
	[engine stop];
}

- (IBAction)setVolume:(id)sender
{
	engine.masterVolume = [(NSTextField *)sender intValue];
}

@end
