/* ////////// LICENSE INFO ////////////////////

 * Copyright (C) 2013 by NYSOL CORPORATION
 *
 * Unless you have received this program directly from NYSOL pursuant
 * to the terms of a commercial license agreement with NYSOL, then
 * this program is licensed to you under the terms of the GNU Affero General
 * Public License (AGPL) as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY EXPRESS OR IMPLIED WARRANTY, INCLUDING THOSE OF 
 * NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Please refer to the AGPL (http://www.gnu.org/licenses/agpl-3.0.txt)
 * for more details.

 ////////// LICENSE INFO ////////////////////*/
// =============================================================================
// kgmvavg.h 移動平均クラス
// =============================================================================
#include <cstdio>
#include <iostream>
#include <sstream>
#include <string>
#include <kgmvavg.h>
#include <kgError.h>
#include <kgConfig.h>

using namespace std;
using namespace kglib;
using namespace kgmod;
// -----------------------------------------------------------------------------
// コンストラクタ(モジュール名，バージョン登録)
// -----------------------------------------------------------------------------
kgMvavg::kgMvavg(void)
{
	#ifdef ENG_FORMAT
		#include <help/en/kgmvavgHelp.h>
	#else
		#include <help/jp/kgmvavgHelp.h>
	#endif
	_name    = "kgmvavg";
	_version = "1.0";
}
// -----------------------------------------------------------------------------
// 入出力ファイルオープン
// -----------------------------------------------------------------------------
void kgMvavg::setArgs(void)
{
	// パラメータチェック
	_args.paramcheck("f=,i=,o=,k=,t=,-w,-exp,alpha=,skip=");

	// 入出力ファイルオープン
	iFile_.open(_args.toString("i=",false), _env,_nfn_i);
  oFile_.open(_args.toString("o=",false), _env,_nfn_o);
	iFile_.read_header();
	oFile_.setPrecision(_precision);

	// f= 項目引数のセット
	vector< vector<kgstr_t> > vvs = _args.toStringVecVec("f=",':',2,true,true);
	fField_.set(vvs, &iFile_,_fldByNum);

	// k= 項目引数のセット
	vector<kgstr_t> vs = _args.toStringVector("k=",false);
	kField_.set(vs,  &iFile_,_fldByNum);

	// -weighted ,-exp の値をセット
	weighted_ = _args.toBool("-w");
	exp_ = _args.toBool("-exp");
	if(weighted_ && exp_){
		throw kgError("-w and -exp are exclusive options");
	}
	simple_= (!weighted_ && !exp_) ;

	// t= の値をセット
	kgstr_t strT=_args.toString("t=",false);
	if(strT.empty()){
		if(simple_ || weighted_){
			throw kgError("t= must be specified");
		}
	}
	term_ = atoi(strT.c_str());

	// alpha= の値をセット
	string strA=_args.toString("alpha=",false);
	if(strA.empty()){
		if(strT.empty()){
			throw kgError("t= or alpha= must be specified with -exp");
		}else{
			alpha_ = 2.0/static_cast<double>(term_+1);
		}
	}else{
		if(!exp_){
			throw kgError("alpha= can be specified only with -exp");
		}
		alpha_ = atof(strA.c_str());
	}

	// skip= の値をセット
	string strS=_args.toString("skip=",false);
	if(strS.empty()){
		if(simple_ || weighted_){
			skip_ = term_-1;
		}else{
			skip_ = 0;
		}
	}else{
		skip_ = atoi(strS.c_str());
	}

}
// ================================================================================
// 単純移動平均
// ================================================================================
// 移動平均出力
void kgMvavg::outputS(size_t pointer)
{
	for(std::size_t i=0; i<fField_.size(); i++){
		avg_.at(i) = sum_.at(i)/count_;
	}
	writeFld(iFile_.getNewFld(),fField_.getFlg_p(),&avg_);
}
// -----------------------------------------------------------------------------
// 移動平均
// -----------------------------------------------------------------------------
void kgMvavg::simpleMA()
{
	size_t pointer=0;
	bool   filled = false;
	bool   out    = false;
	count_ =0;
	size_t recNo=0;

	// データ集計＆出力
	while(iFile_.read()!=EOF){
		recNo++;
		//keybreakしたら出力
		if( iFile_.keybreak() ){
			for(size_t i=0; i<val_.size(); i++){
				val_.at(i)=0;
			}
			pointer=0;
			count_ =0;
			filled = false;
			out    = false;
			recNo=1;
			for(std::size_t i=0; i<fField_.size(); i++){
				sum_.at(i) = 0;
			}
			//ENDなら終了
			if((iFile_.status() & kgCSV::End )) break;
		}

		// sum_の更新
		size_t cell=pointer*fField_.size();
		for(vector<kgstr_t>::size_type i=0; i<fField_.size(); i++){
			sum_.at(i) -= val_.at(cell++);
		}

		// ------------------------------------------
		// データの値をFIFOキューにセット(null値は0)
		cell=pointer*fField_.size();
		for(vector<kgstr_t>::size_type i=0; i<fField_.size(); i++){
			char* str=iFile_.getNewVal(fField_.num(i));
			if(*str!='\0') { val_.at(cell++)=atof(str);}
			else					 { val_.at(cell++)=0; }
		}

		//--------------------------------------------
		// カウンタとtotal_の更新(filledがfalseの間だけ)
		//--------------------------------------------
		if(! filled){ count_=static_cast<double>(recNo); }

		// ------------------------------------------
		// 集約値の更新
		// 単純移動平均の場合は累計を求めるだけ
		// ------------------------------------------
		cell=pointer*fField_.size();
		for(vector<kgstr_t>::size_type i=0; i<fField_.size(); i++){
			sum_.at(i) += val_.at(cell++);
		}
		//--------------------------------------------
		// FIFOキューのポインタを更新
		//--------------------------------------------
		pointer++;

		//--------------------------------------------
		// 結果出力フラグの更新
		if(recNo>skip_){ out = true; }

		//--------------------------------------------
		// FIFOキューのポインタの循環判定処理
		// 指定期間(t=)到達判定
		if(pointer >= term_){
			pointer = 0;
			filled  = true;
		}

		// -------------------------------------------
		// 出力処理
		// -------------------------------------------
		if(out){ outputS(pointer);}
	}
}
// ================================================================================
// 線形加重移動平均
// ================================================================================
// 出力
void kgMvavg::outputW(size_t pointer,bool filled)
{
	// termに達するまでは
	if(! filled){
		for(size_t i=0; i<fField_.size(); i++){
			num_.at(i)=0;
			sum_.at(i)=0;
			for(size_t j=0; j<static_cast<size_t>(count_); j++){
				num_.at(i)+=static_cast<double>(j+1)*val_.at(j*fField_.size()+i);
				sum_.at(i)+=val_.at(j*fField_.size()+i);
			}
			avg_.at(i) = num_.at(i)/total_;
		}
	}else{
		for(std::size_t i=0; i<fField_.size(); i++){
			avg_.at(i) = num_.at(i)/total_;
		}
	}
	writeFld(iFile_.getNewFld(),fField_.getFlg_p(),&avg_);
}
// -----------------------------------------------------------------------------
// 線形加重移動平均
// -----------------------------------------------------------------------------
void kgMvavg::weightedMA()
{
	// val_をFIFOキューと見立てたときのデキュー
	size_t pointer=0;
	bool   filled = false;
	bool   out    = false;
	count_ =0;

	// val_をFIFOキューと見立てたときのデキュー
	size_t recNo=0;

	// データ集計＆出力
	while(iFile_.read()!=EOF){
		recNo++;
		//keybreakしたら出力
		if( iFile_.keybreak() ){
			for(size_t i=0; i<val_.size(); i++){
				val_.at(i)=0;
			}
			pointer=0;
			count_ =0;
			filled = false;
			out    = false;
			recNo=1;
			for(std::size_t i=0; i<fField_.size(); i++){
				sum_.at(i) = 0;
				num_.at(i) = 0;
			}
			//ENDなら終了
			if((iFile_.status() & kgCSV::End )) break;
		}
		// sum_, num_などの更新
		if(filled){
			size_t cell=pointer*fField_.size();
			for(vector<kgstr_t>::size_type i=0; i<fField_.size(); i++){
				num_.at(i) -= sum_.at(i);
				sum_.at(i) -= val_.at(cell++);
			}
		}
		// ------------------------------------------
		// データの値をFIFOキューにセット(null値は0)
		size_t cell=pointer*fField_.size();
		for(vector<kgstr_t>::size_type i=0; i<fField_.size(); i++){
			char* str=iFile_.getNewVal(fField_.num(i));
			if(*str!='\0'){
				val_.at(cell++)=atof(str);
			}else{
				val_.at(cell++)=0;
			}
		}

		//--------------------------------------------
		// カウンタとtotal_の更新(filledがfalseの間だけ)
		if(count_<static_cast<double>(term_)){
			count_=static_cast<double>(recNo);
			total_ = (count_+1)*count_/2.0;
		}

		// ------------------------------------------
		// 集約値の更新
		// Numerator、sumの更新
		// ------------------------------------------
		if(filled){
			cell=pointer*fField_.size();
			for(vector<kgstr_t>::size_type i=0; i<fField_.size(); i++){
				num_.at(i) = num_.at(i) + val_.at(cell)*count_;
				sum_.at(i) += val_.at(cell);
				cell++;
			}
		}

		//--------------------------------------------
		// FIFOキューのポインタを更新
		//--------------------------------------------
		pointer++;

		//--------------------------------------------
		// 結果出力フラグの更新
		//--------------------------------------------
		if(recNo>skip_){ out = true;}

		//--------------------------------------------
		// FIFOキューのポインタの循環判定処理
		// 指定期間(t=)到達判定
		//--------------------------------------------
		if(pointer >= term_){ pointer = 0; }

		// -------------------------------------------
		// 出力処理
		if(out){ outputW(pointer,filled);}

		//--------------------------------------------
		// 指定期間(t=)到達判定
		//--------------------------------------------
		if(pointer == 0){ filled = true; }
	}
}
// ================================================================================
// 指数平滑移動平均
// ================================================================================
// -----------------------------------------------------------------------------
// 指数平滑移動平均出力
// -----------------------------------------------------------------------------
void kgMvavg::outputE(size_t pointer)
{
	writeFld(iFile_.getNewFld(),fField_.getFlg_p(),&num_);
}
// -----------------------------------------------------------------------------
// 指数平滑移動平均
// -----------------------------------------------------------------------------
void kgMvavg::expMA()
{
	// val_をFIFOキューと見立てたときのデキュー
	size_t pointer=0;
	size_t recNo  =0;
	bool   out    = false;

	// データ集計＆出力
	while(iFile_.read()!=EOF){
		recNo++;

		//keybreakしたら出力
		if( iFile_.keybreak() ){
			for(size_t i=0; i<val_.size(); i++){
				val_.at(i)=0;
			}
			pointer=0;
			recNo  =1;
			out    = false;

			for(std::size_t i=0; i<fField_.size(); i++){
				num_.at(i) = 0;
			}

			//ENDなら終了
			if((iFile_.status() & kgCSV::End )) break;
		}
		// ------------------------------------------
		// データの値をFIFOキューにセット(null値は0)
		// ------------------------------------------
		size_t cell=pointer*fField_.size();
		for(vector<kgstr_t>::size_type i=0; i<fField_.size(); i++){
			char* str=iFile_.getNewVal(fField_.num(i));
			if(*str!='\0'){
				val_.at(cell++)=atof(str);
			}else{
				val_.at(cell++)=0;
			}
		}
		// -----------------------------------------
		// numの更新
		// -----------------------------------------
		cell=pointer*fField_.size();
		if(recNo==1){
			for(size_t i=0; i<fField_.size(); i++){
				num_.at(i) = val_.at(cell++);
			}
		}else{
			for(size_t i=0; i<fField_.size(); i++){
				num_.at(i) = alpha_*val_.at(cell) + (1.0-alpha_)*num_.at(i);
				cell++;
			}
		}

		//--------------------------------------------
		// FIFOキューのポインタを更新
		//--------------------------------------------
		pointer++;

		//--------------------------------------------
		// 結果出力フラグの更新
		//--------------------------------------------
		if(recNo>skip_) { out = true; }

		//--------------------------------------------
		// FIFOキューのポインタの循環判定処理
		// 指定期間(t=)到達判定
		//--------------------------------------------
		if(pointer >= term_){
			pointer = 0;
		}
		// -------------------------------------------
		// 出力処理
		// -------------------------------------------
		if(out){ outputE(pointer);}
	}
}
// -----------------------------------------------------------------------------
// 実行
// -----------------------------------------------------------------------------
void kgMvavg::writeFld(char** fld, const vector<int>* flg, const vector<double>* val)
{
	unsigned int i;
	int num;
	for(i=0; i<flg->size()-1; i++){ // csvの項目数で回す
		num=flg->at(i);                  // 対応するval位置
		if(num == -1){ oFile_.writeStr( *(fld+i)    , false );}
		else         { oFile_.writeDbl( val->at(num), false );}
	}
	num=flg->at(i);
	if(num == -1) { oFile_.writeStr( fld[i]      , true  );}
	else          { oFile_.writeDbl( val->at(num), true  );}

}
// -----------------------------------------------------------------------------
// 実行
// -----------------------------------------------------------------------------
void kgMvavg::run(void) try 
{
	// パラメータセット＆入出力ファイルオープン
	setArgs();

	// 入力ファイルにkey項目番号をセットする．
	iFile_.setKey(kField_.getNum());

	// 項目名の出力
  oFile_.writeFldName(fField_, true);

	// 集計用変数領域確保＆初期化
	val_.resize(fField_.size()*term_ ,0);
	sum_.resize(fField_.size() ,0);
	num_.resize(fField_.size() ,0);
	avg_.resize(fField_.size() ,0);
	if(simple_)				 { simpleMA();}
	else if(weighted_) { weightedMA();}
	else							 { expMA();}

	// 終了処理
	iFile_.close();
	oFile_.close();
	successEnd();

}catch(kgError& err){

	errorEnd(err);

}

