/**
 * @file ntmuart.c
 * @author Shinichiro Nakamura
 * @brief UARTの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmuart.h"
#include <LPC17xx.h>

#define IER_RBR     0x01
#define IER_THRE    0x02
#define IER_RLS     0x04

#define IIR_PEND    0x01
#define IIR_RLS     0x03
#define IIR_RDA     0x02
#define IIR_CTI     0x06
#define IIR_THRE    0x01

#define LSR_RDR     0x01
#define LSR_OE      0x02
#define LSR_PE      0x04
#define LSR_FE      0x08
#define LSR_BI      0x10
#define LSR_THRE    0x20
#define LSR_TEMT    0x40
#define LSR_RXFE    0x80

#define POWER_ON_UART0()    (LPC_SC->PCONP |= (1 <<  3))
#define POWER_ON_UART1()    (LPC_SC->PCONP |= (1 <<  4))
#define POWER_ON_UART2()    (LPC_SC->PCONP |= (1 << 24))
#define POWER_ON_UART3()    (LPC_SC->PCONP |= (1 << 25))

#define USE_TXD3_ON_P0_0()  (LPC_PINCON->PINSEL0 |= (0x2 <<  0))
#define USE_RXD3_ON_P0_1()  (LPC_PINCON->PINSEL0 |= (0x2 <<  2))
#define USE_TXD0_ON_P0_2()  (LPC_PINCON->PINSEL0 |= (0x1 <<  4))
#define USE_RXD0_ON_P0_3()  (LPC_PINCON->PINSEL0 |= (0x1 <<  6))
#define USE_TXD2_ON_P0_10() (LPC_PINCON->PINSEL0 |= (0x1 << 20))
#define USE_RXD2_ON_P0_11() (LPC_PINCON->PINSEL0 |= (0x1 << 22))
#define USE_TXD1_ON_P0_15() (LPC_PINCON->PINSEL0 |= (0x1 << 30))
#define USE_RXD1_ON_P0_16() (LPC_PINCON->PINSEL1 |= (0x1 <<  0))
#define USE_TXD3_ON_P0_25() (LPC_PINCON->PINSEL1 |= (0x3 << 18))
#define USE_RXD3_ON_P0_26() (LPC_PINCON->PINSEL1 |= (0x3 << 20))
#define USE_TXD1_ON_P2_0()  (LPC_PINCON->PINSEL4 |= (0x2 <<  0))
#define USE_RXD1_ON_P2_1()  (LPC_PINCON->PINSEL4 |= (0x2 <<  2))
#define USE_TXD2_ON_P2_8()  (LPC_PINCON->PINSEL4 |= (0x2 << 16))
#define USE_RXD2_ON_P2_9()  (LPC_PINCON->PINSEL4 |= (0x2 << 18))
#define USE_TXD3_ON_P4_28() (LPC_PINCON->PINSEL9 |= (0x3 << 24))
#define USE_RXD3_ON_P4_29() (LPC_PINCON->PINSEL9 |= (0x3 << 26))

/**
 * @brief UARTを初期化する。
 *
 * @param baudrate ボーレート。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_init(uint32_t baudrate)
{
    uint32_t freqdiv;
    uint32_t pclkdiv, pclk;

    /*
     * UART0を使用する。
     */
    POWER_ON_UART0();
    USE_TXD0_ON_P0_2();
    USE_RXD0_ON_P0_3();

    pclkdiv = (LPC_SC->PCLKSEL0 >> 6) & 0x03;
    switch (pclkdiv) {
        case 0x00:
        default:
            pclk = SystemCoreClock / 4;
            break;
        case 0x01:
            pclk = SystemCoreClock;
            break;
        case 0x02:
            pclk = SystemCoreClock / 2;
            break;
        case 0x03:
            pclk = SystemCoreClock / 8;
            break;
    }

    /* 8 bits, no Parity, 1 Stop bit */
    LPC_UART0->LCR = 0x83;
    /*baud rate */
    freqdiv = (pclk / 16) / baudrate ;
    LPC_UART0->DLM = freqdiv / 256;
    LPC_UART0->DLL = freqdiv % 256;
    /* DLAB = 0 */
    LPC_UART0->LCR = 0x03;
    /* Enable and reset TX and RX FIFO. */
    LPC_UART0->FCR = 0x07;

    /* Enable UART0 interrupt */
    LPC_UART0->IER = IER_RBR | IER_RLS;

    return 0;
}

/**
 * @brief UARTの読み込みを実行する。
 *
 * @param buf バッファ。
 * @param cnt 読み込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_read(char *buf, const int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) {
        while (!(LPC_UART0->LSR & LSR_RDR)) {
        }
        buf[i] = LPC_UART0->RBR;
    }
    return 0;
}

/**
 * @brief UARTの書き込みを実行する。
 *
 * @param buf バッファ。
 * @param cnt 書き込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_write(const char *buf, const int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) {
        while (!(LPC_UART0->LSR & LSR_THRE)) {
        }
        LPC_UART0->THR = buf[i];
    }
    return 0;
}

