/**
 * @file ntmuart.c
 * @author Shinichiro Nakamura
 * @brief UARTの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmuart.h"
#include <LPC13xx.h>

#define IER_RBR     0x01
#define IER_THRE    0x02
#define IER_RLS     0x04

#define LSR_RDR     0x01
#define LSR_OE      0x02
#define LSR_PE      0x04
#define LSR_FE      0x08
#define LSR_BI      0x10
#define LSR_THRE    0x20
#define LSR_TEMT    0x40
#define LSR_RXFE    0x80

/**
 * @brief UARTを初期化する。
 *
 * @param baudrate ボーレート。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_init(uint32_t baudrate)
{
    uint32_t freqdiv;
    uint32_t regval;

    /*
     * UART0を使用する。
     */
    LPC_IOCON->PIO1_6 &= ~0x07;
    LPC_IOCON->PIO1_6 |= 0x01;
    LPC_IOCON->PIO1_7 &= ~0x07;
    LPC_IOCON->PIO1_7 |= 0x01;

    /*
     * UARTのクロックを有効にする。
     */
    LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 12);
    LPC_SYSCON->UARTCLKDIV = 0x1;

    /*
     * フォーマットは8ビット、パリティなし、ストップビット1ビット。
     */
    LPC_UART->LCR = 0x83;
    regval = LPC_SYSCON->UARTCLKDIV;
    freqdiv = (((SystemCoreClock / LPC_SYSCON->SYSAHBCLKDIV) / regval) / 16) / baudrate;

    LPC_UART->DLM = freqdiv / 256;
    LPC_UART->DLL = freqdiv % 256;
    LPC_UART->LCR = 0x03;   /* DLAB = 0 */
    LPC_UART->FCR = 0x07;   /* Enable and reset TX and RX FIFO. */

    /*
     * ラインステータスをクリアするためにリードを実行する。
     */
    regval = LPC_UART->LSR;

    /*
     * TXとRXのFIFOのいずれにもデータがないことを確認する。
     */
    while ((LPC_UART->LSR & (LSR_THRE|LSR_TEMT)) != (LSR_THRE | LSR_TEMT)) {
    }

    while (LPC_UART->LSR & LSR_RDR) {
        /*
         * RXのFIFOからのデータをダンプする。
         */
        regval = LPC_UART->RBR;
    }

    LPC_UART->IER = IER_RBR | IER_RLS;
    return 0;
}

/**
 * @brief UARTの読み込みを実行する。
 *
 * @param buf バッファ。
 * @param cnt 読み込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_read(char *buf, const int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) {
        while (!(LPC_UART->LSR & LSR_RDR)) {
        }
        buf[i] = LPC_UART->RBR;
    }
    return 0;
}

/**
 * @brief UARTの書き込みを実行する。
 *
 * @param buf バッファ。
 * @param cnt 書き込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmuart_write(const char *buf, const int cnt)
{
    int i;
    for (i = 0; i < cnt; i++) {
        while (!(LPC_UART->LSR & LSR_THRE)) {
        }
        LPC_UART->THR = buf[i];
    }
    return 0;
}

