/**
 * @file ntmstr.h
 * @author Shinichiro Nakamura
 * @brief NTMが使用する文字列ユティリティの定義。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#ifndef NTMSTR_H
#define NTMSTR_H

#include "ntmint.h"

/*
 * BIN: b01010101, 01010101b
 * OCT: o12345678, 12345678o
 * DEC: d12345678, 12345678d
 * HEX: h1234ABCD, 1234ABCDh
 */

/**
 * @brief 基数の定義。
 */
typedef enum {
    NtmStrTypeBin,      /**< 基数は2進。 */
    NtmStrTypeOct,      /**< 基数は8進。 */
    NtmStrTypeDec,      /**< 基数は10進。 */
    NtmStrTypeHex,      /**< 基数は16進。 */
    NtmStrTypeInvalid   /**< 基数が不正。 */
} NtmStrType;

/**
 * @brief 与えられた数値を与えられた条件に従って文字列に変換する。
 * @details
 * 大文字と小文字の選択が可能である。
 * 出力桁数の選択が可能である。
 * 基数の選択が可能である。
 *
 * @param number 値。
 * @param string 文字列。
 * @param length 文字列の整形サイズ。
 * @param type 基数指定。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmstr_number_to_string(
        const uint32_t number,
        char *string,
        const uint32_t length,
        const NtmStrType type);

/**
 * @brief 与えられた文字列を数値として解釈し変換する。
 * @details
 * この関数は大文字と小文字を区別しない。
 * 書式に従うことで16進、10進、2進と自動判別する。
 * 基数指定がない場合のデフォルトは選択可能である。
 *
 * @param string 文字列。
 * @param number 値。
 * @param type 判別した基数。
 * @param defaultType 自動判別できなかった時に採用する値。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int ntmstr_string_to_number(
        const char *string,
        uint32_t *number,
        NtmStrType *type,
        const NtmStrType defaultType);

/**
 * @brief 与えられた文字が表示可能かどうかを返す。
 *
 * @param c 文字。
 *
 * @retval !0 表示可能である。
 * @retval 0 表示可能でない。
 */
int ntmstr_isprint(int c);

#endif

