/**
 * @file ntmcmd_write.c
 * @author Shinichiro Nakamura
 * @brief writeコマンドの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmcmd_write.h"
#include "ntmuart.h"
#include "ntmlibc.h"
#include "ntmio.h"
#include "ntmconf.h"
#include "ntmcmd_common.h"

#if (NTMCONF_ENABLE_WRITE == 1)

static const char *DISPTEXT_USAGE_WB = "WB [ADDR] [DATA]\r\n";
static const char *DISPTEXT_USAGE_WW = "WW [ADDR] [DATA]\r\n";
static const char *DISPTEXT_USAGE_WL = "WL [ADDR] [DATA]\r\n";

/**
 * @brief WRITEコマンド。
 *
 * @retval 0 与えられたパラメータはWRITEコマンドでなかった。
 * @retval 1 与えられたパラメータをWRITEコマンドとして処理した。
 */
int ntmcmd_write(int argc, char **argv, void *extobj)
{
    NtmAccessMode am = NtmAccessModeDefault;
    uint32_t adr;
    uint32_t dat;
    NtmStrType adr_type;
    NtmStrType dat_type;

    /*
     * コマンドからアクセスモードを決定する。
     */
    if ((ntmlibc_strcmp(argv[0], "wb") == 0) || (ntmlibc_strcmp(argv[0], "WB") == 0)) {
        am = NtmAccessModeByte;
    } else if ((ntmlibc_strcmp(argv[0], "ww") == 0) || (ntmlibc_strcmp(argv[0], "WW") == 0)) {
        am = NtmAccessModeWord;
    } else if ((ntmlibc_strcmp(argv[0], "wl") == 0) || (ntmlibc_strcmp(argv[0], "WL") == 0)) {
        am = NtmAccessModeLong;
    } else {
        /*
         * 与えられたコマンドは、このモジュールのコマンドに該当しなかった。
         */
        return 0;
    }

    /*
     * パラメータ数を検証する。
     */
    if (argc != 3) {
        switch (am) {
            case NtmAccessModeByte:
                ntmuart_write(DISPTEXT_USAGE_WB, ntmlibc_strlen(DISPTEXT_USAGE_WB));
                return 1;
            case NtmAccessModeWord:
                ntmuart_write(DISPTEXT_USAGE_WW, ntmlibc_strlen(DISPTEXT_USAGE_WW));
                return 1;
            case NtmAccessModeLong:
                ntmuart_write(DISPTEXT_USAGE_WL, ntmlibc_strlen(DISPTEXT_USAGE_WL));
                return 1;
            case NtmAccessModeDefault:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
            default:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
        }
    }

    /*
     * ユーザからの入力を取得する。
     */
    ntmstr_string_to_number(argv[1], &adr, &adr_type, NTMCONF_DEFAULT_STRTYPE);
    ntmstr_string_to_number(argv[2], &dat, &dat_type, NTMCONF_DEFAULT_STRTYPE);

    /*
     * ユーザからの入力を検証する。
     * 不正な指定が有った場合、エラーメッセージを出力して終了する。
     */
    if (adr_type == NtmStrTypeInvalid) {
        ntmuart_write(NTMCMD_COMMON_TEXT_INVADR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVADR));
        return 1;
    }
    if (dat_type == NtmStrTypeInvalid) {
        ntmuart_write(NTMCMD_COMMON_TEXT_INVDAT, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVDAT));
        return 1;
    }

    /*
     * アクセスモードに応じて処理を実行する。
     */
    switch (am) {
        case NtmAccessModeByte:
            {
                if (dat & 0xFFFFFF00) {
                    /*
                     * バイト書き込みにも関わらず0xFF以外のビットが立っているのはおかしい。
                     */
                    ntmuart_write(NTMCMD_COMMON_TEXT_INVDAT, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVDAT));
                    return 1;
                }
                /*
                 * 指定アドレスにデータを書き込む。
                 */
                ntmio_write_byte(adr, dat);
            }
            break;
        case NtmAccessModeWord:
            {
                if (dat & 0xFFFF0000) {
                    /*
                     * ワード書き込みにも関わらず0xFFFF以外のビットが立っているのはおかしい。
                     */
                    ntmuart_write(NTMCMD_COMMON_TEXT_INVDAT, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVDAT));
                    return 1;
                }
                /*
                 * 指定アドレスにデータを書き込む。
                 */
                ntmio_write_word(adr, dat);
            }
            break;
        case NtmAccessModeLong:
            {
                /*
                 * 指定アドレスのデータを読み込む。
                 */
                ntmio_write_long(adr, dat);

            }
            break;
        default:
            ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
            break;
    }
    return 1;
}

#endif

